/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include "xvasprintf.h"
#include "gosub.h"
#include "story.h"
#include "gettext-more.h"

static struct argp_option options[] = 
{
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_gosub_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_gosub_options_t*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
        {
          if (opt->subreddit == NULL && !opt->parsing_errors)
            opt->subreddit = arg;
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("That looks like more than one subreddit.  "
                              "Try putting it all in quotes."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify a subreddit to go to."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->subreddit = NULL;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

struct argp reddit_gosub_argp = { options, parse_opt, "SUBREDDIT-NAME",N_("Go to a subreddit.\v") };

int 
reddit_gosub_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_gosub_options_t gosub_options;
  gosub_options.state = state;

  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &gosub_options);
  if (!err && gosub_options.subreddit != NULL && !gosub_options.parsing_errors)
    return reddit_gosub (state, &gosub_options);
  else
    return 0;
}

int
reddit_is_in_subreddit (struct reddit_state_t *state)
{
  if (state->subreddit == NULL)
    return 0;
  if (strcmp (state->subreddit, "all") == 0)
    return 0;
  return 1;
}

int 
reddit_gosub (struct reddit_state_t *state, struct reddit_gosub_options_t *options)
{
  int err = 0;
  int response = 0;
  err = reddit_load_and_show_stories (state, "", 0, options->subreddit, &response);
  if (err)
    {
      fprintf (state->out, _("Error: Couldn't load page for subreddit `%s'.  "
                             "(HTTP /1.1 %d err=%d)\n"), options->subreddit,
                             response, err);
      err = 0;
    }
  return err;
}
