/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "xvasprintf.h"
#include "upvote.h"
#include "story.h"
#include "stories.h"
#include "gettext-more.h"
#include "read-file.h"

static struct argp_option options[] = 
{
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_upvote_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_upvote_options_t*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
        {
          if (opt->story_id == -1 && !opt->parsing_errors)
            {
              opt->story_id = argp_parse_story_id (state, arg, opt->state);
              if (opt->story_id == -1)
                opt->parsing_errors = 1;
            }
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("You may only upvote one story at a time."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify a story number to upvote."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->story_id = -1;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static char *
help_filter (int key, const char *text, void *input)
{
  struct reddit_upvote_options_t *o = (struct reddit_upvote_options_t*) input;
  return argp_parse_story_help_filter (key, text, o ? o->state : NULL);
}

struct argp reddit_upvote_argp = { options, parse_opt, "STORY-NUMBER",N_("Increases the number of people who liked a given story.  Tallies your vote in favour of a story.\v"), 0, help_filter};

int 
reddit_upvote_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_upvote_options_t upvote_options;
  upvote_options.state = state;

  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &upvote_options);
  if (!err && upvote_options.story_id != -1 && !upvote_options.parsing_errors)
    return reddit_upvote (state, &upvote_options);
  else
    return 0;
}

int 
reddit_vote_story (struct reddit_state_t *state, int dir, struct reddit_story_t *story, int *resp, char **errmsg)
{
  int err = 0;
  char *msg = NULL;
  char *post = xasprintf ("id=%s&dir=%d&vh=%s&uh=%s&renderstyle=%s",
                          story->id,
                          dir,
                          story->votehash,
                          state->modhash,
                          "html");
  err = reddit_post (state, "vote", post, resp, &msg);
  free (post);
  if (strstr (msg, "{}") == NULL)
    {
      if (!err)
        err = -1;
      *errmsg = msg;
    }
  else
    free (msg);
  return err;
}

int 
reddit_upvote_story (struct reddit_state_t *state, struct reddit_upvote_options_t *options, struct reddit_story_t *story, int *resp, char **errmsg)
{
  return reddit_vote_story (state, 1, story, resp, errmsg);
}

int 
reddit_upvote (struct reddit_state_t *state, struct reddit_upvote_options_t *options)
{
  int err = 0;
  if (state->logged_in == 0)
    {
      fprintf (state->out, _("You need to login to vote on a story."
                             "  Login using the `login' command.\n"));
      return 0;
    }
  struct reddit_story_t *story = 
    reddit_find_story_with_id (state, options->story_id);
  int response = 0;
  char *errmsg = NULL;
  if (story)
    err = reddit_upvote_story (state, options, story, &response, &errmsg);

  if (!err)
    reddit_update_modhash (state);
  if (!err)
    fprintf (state->out, _("The story has been upvoted.\n"));
  else
    {
      fprintf (state->out, _("Error: Couldn't upvote the story "
                             "HTTP /1.1 %d err=%d)\n"), response, err);
      if (errmsg)
        {
          fprintf (state->out, _("Maybe this gobbledegook will be useful "
                                 "in diagnosing what went wrong:\n"
                                 "%s\n"), errmsg);
          free (errmsg);
        }
    }
  return err;
}
