/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdlib.h>
#include <unistd.h>
#ifdef HAVE_LIBREADLINE
#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>
#endif
#include <glib.h>
#include "xvasprintf.h"
#include "reddit_priv.h"
#include "share.h"
#include "story.h"
#include "stories.h"
#include "gettext-more.h"

static struct argp_option options[] = 
{
    {"from", 'f', "NAME", 0, 
      N_("specify the NAME of who the message is from.") },
    {"reply-to", 'r', "EMAIL", 0, 
      N_("specify the return email-address of the message.") },
    {"message", 'm', 0, 0, 
      N_("type in your own blurb that precedes the link.") },
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_share_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_share_options_t*) state->input;
  switch (key)
    {
    case 'm':
      opt->add_message = 1;
      break;
    case 'f':
      opt->from_name = arg;
      break;
    case 'r':
      opt->reply_address = argp_parse_email (state, arg, opt->state);
      if (opt->reply_address == NULL && !opt->parsing_errors)
        opt->parsing_errors = 1;
      break;
    case ARGP_KEY_ARG:
        {
          if (opt->story_id == -1 && !opt->parsing_errors)
            {
              opt->story_id = argp_parse_story_id (state, arg, opt->state);
              if (opt->story_id == -1)
                opt->parsing_errors = 1;
            }
          else if (opt->to_email == NULL && !opt->parsing_errors)
            {
              opt->to_email = argp_parse_email (state, arg, opt->state);
              if (opt->to_email == NULL && !opt->parsing_errors)
                opt->parsing_errors = 1;
            }
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("You may only share one story at a time."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify an email address of a friend, "
                          "and a story number to share."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->add_message = -1;
      opt->story_id = -1;
      opt->to_email = NULL;
      opt->reply_address = NULL;
      opt->message = NULL;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

char *
argp_parse_email (struct argp_state *state, char *email, 
                  struct reddit_state_t *r)
{
  if (g_regex_match_simple ("[A-Z0-9._%+-]+@[A-Z0-9.-]+\\.[A-Z]{2,4}", 
                            email, G_REGEX_CASELESS, 0))
    return email;
  argp_failure (state, 0, 0, _("The email address `%s' is malformed."), email);
  return NULL;
}

static char *
help_filter (int key, const char *text, void *input)
{
  struct reddit_share_options_t *o = (struct reddit_share_options_t*) input;
  return argp_parse_story_help_filter (key, text, o ? o->state : NULL);
}

struct argp reddit_share_argp = { options, parse_opt, "STORY-NUMBER EMAIL-ADDRESS",N_("Share a story with a friend by email.\v"), 0, help_filter };

int 
reddit_share_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_share_options_t share_options;
  share_options.state = state;
  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &share_options);
  if (!err && share_options.to_email && share_options.story_id != -1 &&
      !share_options.parsing_errors)
    return reddit_share (state, &share_options);
  else
    return 0;
}

static int
reddit_share_story (struct reddit_state_t *state, struct reddit_share_options_t *options, struct reddit_story_t *story, int *resp, char **errmsg)
{
  int err = 0;
  char *post = 
    xasprintf ("parent=%s&share_from=%s&replyto=%s&share_to=%s&message=%s"
               "&id%%23sharelink_%s&uh=%s&renderstyle=%s",
               story->id,
               options->from_name ? options->from_name : state->username,
               options->reply_address ? options->reply_address: "", 
               options->to_email, 
               options->message,
               story->id,
               state->modhash,
               "html");
  char *msg = NULL;
  err = reddit_post (state, "share", post, resp, &msg);
  free (post);
  if (strstr (msg, "your link has been shared") == NULL)
    {
      if (!err)
        err = -1;
      *errmsg = msg;
    }
  else
    free (msg);
  return err;
}

int 
reddit_share (struct reddit_state_t *state, struct reddit_share_options_t *options)
{
  int err = 0;
  if (state->logged_in == 0)
    {
      reddintf (state, _("You need to login to share a story."
                         "  Login using the `login' command.\n"));
      return 0;
    }
  int malloced_message = 0;
  if (options->add_message && options->message == NULL)
    {
      reddintf (state, _("Type in your message and then press enter.\n"));
#ifdef HAVE_LIBREADLINE
      stifle_history (0);
      options->message = readline (_("Message: "));
      unstifle_history();
#else
      options->message = myreadline (_("Message: "));
#endif
      malloced_message = 1;
    }

  if (options->message == NULL)
    {
      options->message = 
        xasprintf ("%s from http://%s/ has shared a link with you.",
                   state->username, state->site);
      malloced_message = 1;
    }

  struct reddit_story_t *story = 
    reddit_find_story_with_id (state, options->story_id);
  int response = 0;
  char *errmsg = NULL;
  if (story)
    err = reddit_share_story (state, options, story, &response, &errmsg);

  if (!err)
    reddit_update_modhash (state);
  if (!err)
    reddintf (state, _("The story has been shared.\n"));
  else
    {
      reddintf (state, _("Error: Couldn't share the story.  "
                         "(HTTP /1.1 %d err=%d)\n"), response, err);
      if (errmsg)
        {
          reddintf (state, _("Maybe this gobbledegook will be useful "
                             "in diagnosing what went wrong:\n"
                             "%s\n"), errmsg);
          free (errmsg);
        }
    }
  if (malloced_message)
    free (options->message);
  return err;
}
