/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <error.h>
#include <string.h>
#include <argp.h>
#include <argz.h>
#include <unistd.h>
#include "xvasprintf.h"
#include "gettext-more.h"
#include "opts.h"
#include "reddit_priv.h"
#include "wallet.h"

#define FULL_VERSION PACKAGE_NAME " " PACKAGE_VERSION

const char *argp_program_version = FULL_VERSION;
const char *argp_program_bug_address = "<" PACKAGE_BUGREPORT ">";

#define OAO OPTION_ARG_OPTIONAL
#define OH OPTION_HIDDEN

struct arguments_t arguments;
static struct argp_option options[] = 
{
    { "quiet", OPT_QUIET, NULL, 0, N_("don't show the initial legalese") },
    { "login", OPT_LOGIN, "USER", 0, N_("login as USER to reddit" ) },
    { "host", OPT_HOST, "NAME", OPTION_HIDDEN, 
      N_("connect to NAME instead of " REDDIT_SITE) },
    { "proxy", OPT_PROXY, "HOST:PORT", 0, N_("use a proxy to access reddit") },
    { "wallet", OPT_PASSWORD_WALLET, "FILE", OPTION_ARG_OPTIONAL, 
      N_("store/retrieve a password in an encrypted FILE ") },
    { "no-initial-stories", OPT_NO_SHOW, 0, 0, 
      N_("do not show stories initially") },
    { "cookie-jar", OPT_COOKIEFILE, "FILE", 0, 
      N_("store/retrieve cookies from FILE") },
    { "script", OPT_SCRIPT, 0, OPTION_HIDDEN, 
      N_("same as -q --no-initial-stories, and also turns off the banner") },
    { 0 }
};

static char*
get_default_wallet_filename()
{
  return xasprintf("%s/.%s-wallet", getenv("HOME"), PACKAGE);
}

static void
init_options (struct reddit_options_t *app)
{
  app->quiet = -1;
  app->no_initial_stories = -1;
  app->banner = -1;
  char *wallet = getenv("REDDIT_WALLET");
  if (wallet && atoi (wallet))
    app->wallet = get_default_wallet_filename();
  return;
}

static char *
argp_parse_proxy (struct argp_state *state, char *p, struct reddit_state_t *r)
{
  if (g_regex_match_simple ("[\\w\\-_]+(\\.[\\w\\-_]+)+([\\w\\-\\.,@"
                            "?^=%&amp;:/~\\+#]*[\\w\\-\\@?^=%&amp;/~\\+#])?",
                            p, G_REGEX_CASELESS, 0))
    return p;
  argp_failure (state, 0, 0, _("The proxy `%s' is malformed."), p);
  return NULL;
}

static error_t 
parse_opt (int key, char *arg, struct argp_state *state) 
{
  struct arguments_t *arguments = (struct arguments_t *) state->input;

  switch (key) 
    {
    case OPT_SCRIPT:
      arguments->reddit.banner = 0;
      arguments->reddit.no_initial_stories = 1;
      arguments->reddit.quiet = 1;
      break;
    case OPT_COOKIEFILE:
      arguments->reddit.cookie_file = arg;
      break;
    case OPT_NO_SHOW:
      arguments->reddit.no_initial_stories = 1;
      break;
    case OPT_HOST:
      arguments->reddit.host = arg;
      break;
    case OPT_PASSWORD_WALLET:
      free (arguments->reddit.wallet);
      if (!arg)
        arguments->reddit.wallet = get_default_wallet_filename ();
      else
        arguments->reddit.wallet = strdup (arg);
      if (access (arguments->reddit.wallet, W_OK) != 0)
        {
          free (arguments->reddit.wallet);
          arguments->reddit.wallet = NULL;
        }
      break;
    case OPT_PROXY:
      if (argp_parse_proxy (state, arg, NULL))
        {
          arguments->reddit.proxy = arg;
          char *colon = strrchr (arg, ':');
          if (colon)
            {
              char *end = NULL;
              *colon = '\0';
              unsigned long val = strtoul (++colon, &end, 10);
              if ((int)val > 0)
                arguments->reddit.proxy_port = val;
            }
          else
            arguments->reddit.proxy_port = DEFAULT_PROXY_PORT;
        }
      break;
    case OPT_QUIET:
	arguments->reddit.quiet = 1;
      break;
    case OPT_LOGIN:
	arguments->reddit.username = arg;
      break;
    case ARGP_KEY_INIT:
      init_options (&arguments->reddit);
      break;
    case ARGP_KEY_ARG:
      if (arguments->reddit.subreddit == NULL)
        arguments->reddit.subreddit = arg;
      else
        argp_failure (state, 0, 0, 
                      _("It looks like you specified more than on subreddit.  "
                       "Maybe try putting it in quotes?"));
      break;
    case ARGP_KEY_END:
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

struct argp argp = { options, parse_opt, "[SUBREDDIT]", 
  N_("A bandwidth-saving client for reddit.com.\v"
     "The default password wallet file is `~/.reddit-wallet'.  "
     "The default cookie file is `" REDDIT_COOKIEJAR "'."), 0, 0, 
  PACKAGE }; 

static void 
set_default_options (struct reddit_options_t *app)
{
  if (app->quiet == -1)
    app->quiet = atoi (DEFAULT_QUIET_VALUE);
  if (app->no_initial_stories == -1)
    app->no_initial_stories = 0;
  if (app->banner == -1)
    app->banner = 1;
  return;
}

void 
parse_opts (int argc, char **argv, struct arguments_t *arguments)
{
  int retval;
  setenv ("ARGP_HELP_FMT", "no-dup-args-note", 1);
  retval = argp_parse (&argp, argc, argv, 0, 0, arguments); 
  if (retval < 0)
    argp_help (&argp, stdout, ARGP_HELP_EXIT_ERR|ARGP_HELP_SEE,PACKAGE_NAME);
  set_default_options (&arguments->reddit);

  argp_program_version = 0;
  argp_program_bug_address = 0;
  return;
}
