/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <argz.h>
#include <argp.h>
#include "opts.h"
#include "xvasprintf.h"
#include "trim.h"
#include "read-file.h"
#ifdef HAVE_LIBREADLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif
#include "gettext-more.h"
#include <curl/curl.h>
#include "reddit.h"
#include "reddit_priv.h"
#include "stories.h"

#include "upvote.h"
#include "downvote.h"
#include "discuss.h"
#include "gosub.h"
#include "return.h"
#include "ls.h"
#include "reload.h"
#include "next.h"
#include "prev.h"
#include "view.h"
#include "share.h"
#include "grep.h"
#include "whats.h"
#include "login.h"
#include "submit.h"
#include "save.h"
#include "unsave.h"
#include "whoami.h"

#define REDDIT_PROMPT           "reddit> "

enum 
{
  HISTORY = 0,
  BANNER,
  BROWSER,
  BROWSEROPTS,
  LOGIN,
  SUBMIT,
  SAVE,
  UNSAVE,
  VIEW,
  UPVOTE,
  DOWNVOTE,
  SHARE,
  GREP,
  DISCUSS,
  WHATS,
  NEXT,
  PREV,
  GOSUB, 
  GOBACK, 
  LS,
  PWD,
  WHOAMI,
  PAGESIZE,
  RELOAD,
  HELP,
  WARRANTY,
  QUIT,
  THE_END
};
static char *const cmds[] = 
{
  [HISTORY] = N_("history"),
  [BANNER] = N_("banner"),
  [BROWSER] = N_("browser"),
  [BROWSEROPTS] = N_("browser-options"),
  [LOGIN] = N_("login"),
  [SUBMIT] = N_("submit"),
  [SAVE] = N_("save"),
  [UNSAVE] = N_("unsave"),
  [VIEW] = N_("view"),
  [UPVOTE] = N_("upvote"),
  [DOWNVOTE] = N_("downvote"),
  [SHARE] = N_("share"),
  [GREP] = N_("grep"),
  [DISCUSS] = N_("discuss"),
  [WHATS] = N_("what's"),
  [NEXT] = N_("next"),
  [PREV] = N_("prev"),
  [GOSUB] = N_("gosub"),
  [GOBACK] = N_("return"),
  [LS] = N_("ls"),
  [PWD] = N_("pwd"),
  [WHOAMI] = N_("whoami"),
  [PAGESIZE] = N_("pagesize"),
  [RELOAD] = N_("reload"),
  [HELP] = N_("help"),
  [WARRANTY] = N_("warranty"),
  [QUIT] = N_("quit"),
  [THE_END] = NULL
};
static  struct argp * const parsers[] =
{
  [HISTORY] = NULL,
  [BANNER] = NULL,
  [BROWSER] = NULL,
  [BROWSEROPTS] = NULL,
  [VIEW] = &reddit_view_argp,
  [LOGIN] = &reddit_login_argp,
  [SUBMIT] = &reddit_submit_argp,
  [SAVE] = &reddit_save_argp,
  [UNSAVE] = &reddit_unsave_argp,
  [UPVOTE] = &reddit_upvote_argp,
  [DOWNVOTE] = &reddit_downvote_argp,
  [SHARE] = &reddit_share_argp,
  [GREP] = &reddit_grep_argp,
  [DISCUSS] = &reddit_discuss_argp,
  [WHATS] = &reddit_whats_argp,
  [NEXT] = &reddit_next_argp,
  [PREV] = &reddit_prev_argp,
  [GOSUB] = &reddit_gosub_argp,
  [GOBACK] = &reddit_return_argp,
  [LS] = &reddit_ls_argp,
  [PWD] = NULL,
  [WHOAMI] = &reddit_whoami_argp,
  [PAGESIZE] = NULL,
  [RELOAD] = &reddit_reload_argp,
  [HELP] = NULL,
  [WARRANTY] = NULL,
  [QUIT] = NULL,
  [THE_END] = NULL
};

static int 
get_command (char *const cmds[], struct argp* const parsers[], char *line, char **value, struct argp **argp)
{
  char *ptr = line;
  int retval = getsubopt (&ptr, cmds, value);
  if (retval >= 0)
    {
      if (parsers[retval] == NULL)
        {
          if (*value)
            {
              char *newvalue = strdup (*value);
              *value = newvalue;
            }
          return retval;
        }
    }

  *value = NULL;
  int i = 0;
  char *cmd = NULL;
  retval = sscanf (line, "%ms", &cmd);
  for (i = 0; cmds[i] != NULL; i++)
    if (retval == 1 && strcasecmp (cmd, cmds[i]) == 0)
      break;
  free (cmd);
  *argp = parsers[i];
  return i;
}

static void 
make_command_line (char *cmd, int *argc, char ***argv)
{
  char *s1 = strdup (cmd);
  if (s1)
    {
      char *term;
      *argv = NULL;
      *argc = 0;
      for (term = strtok (s1, " \t"); term != NULL; term = strtok (NULL, " \t"))
        {
          *argv = (char **) realloc (*argv, sizeof (char *) * ((*argc) + 1));
          if (*argv)
            (*argv)[*argc] = strdup (term);
          (*argc)++;
        }
      *argv = (char **) realloc (*argv, sizeof (char *) * ((*argc) + 1));
      if (*argv)
        (*argv)[(*argc)] = 0;
      free (s1);
    }
}

static void 
reddit_help (struct reddit_state_t *state, char * const cmds[], struct argp* const parsers[])
{
  fprintf (state->out, _("List of Commands:\n"));
  int i = 0;
  for (; cmds[i] != NULL; i++)
    {
      if (i == HELP || i == BROWSER || i == BROWSEROPTS || i == HISTORY || 
          i == BANNER)
        continue;
      fprintf(state->out, "\t%s %s\n", cmds[i], 
              parsers[i] ? parsers[i]->args_doc : "");
    }
}

static void
reddit_welcome (struct reddit_state_t *state)
{
  fprintf (state->out, "%s %s\n", PACKAGE, VERSION);
  fprintf (state->out, "Copyright (C) 2011 Ben Asselstine\n");
  fprintf (state->out, "%s\n", _("REDDIT is a registered trademark "
                              "of Advance Magazine Publishers Inc."));
  fprintf (state->out, "%s\n", 
           _("This is free software with ABSOLUTELY NO WARRANTY."));
  fprintf (state->out, _("For warranty details type `%s'.\n"), 
           _(cmds[WARRANTY]));
}

static int
reddit_parse_command (struct reddit_state_t *state, char *line, int *quit)
{
  int err = 0;

  if (strcmp (line, "\n") == 0 || strcmp (line, "\r\n") == 0 || *line == 0)
    return 0;

  state->command = line;
  int argc = 0;
  char **argv = NULL;
  //make_argv (line, &argc, &argv);
  make_command_line (line, &argc, &argv);
  char *value = NULL;
  struct argp *argp = NULL;
  int command = get_command (cmds, parsers, line, &value, &argp);
  switch (command)
    {
    case HISTORY:
      if (!value)
        err = reddit_history (state);
      else
        err = reddit_set_history (state, value);
      break;
    case BANNER:
      if (!value)
        reddit_unknown_command (state);
      else
        err = reddit_set_banner (state, value);
      break;
    case BROWSER:
      if (!value)
        err = reddit_browser (state);
      else
        err = reddit_set_browser (state, value);
      break;
    case BROWSEROPTS:
      if (!value)
        err = reddit_browser_options (state);
      else
        err = reddit_set_browser_options (state, value);
      break;
    case WARRANTY:
      if (!value)
        err = reddit_warranty (state);
      else
        reddit_unknown_command (state);
      break;
    case UPVOTE:
      if (!value && argp)
        err = reddit_upvote_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case DOWNVOTE:
      if (!value && argp)
        err = reddit_downvote_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case VIEW:
      if (!value && argp)
        err = reddit_view_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case SHARE:
      if (!value && argp)
        err = reddit_share_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case GREP:
      if (!value && argp)
        err = reddit_grep_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case GOSUB:
      if (!value && argp)
        err = reddit_gosub_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case GOBACK:
      if (!value && argp)
        err = reddit_return_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case HELP:
      if (!value)
        reddit_help (state, cmds, parsers);
      else
        reddit_unknown_command (state);
      break;
    case LOGIN:
      if (!value && argp)
        err = reddit_login_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      if (err)
        err=0;
      break;
    case SUBMIT:
      if (!value && argp)
        err = reddit_submit_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case SAVE:
      if (!value && argp)
        err = reddit_save_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case UNSAVE:
      if (!value && argp)
        err = reddit_unsave_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case NEXT:
      if (!value && argp)
        err = reddit_next_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case PREV:
      if (!value && argp)
        err = reddit_prev_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case WHATS:
      if (!value && argp)
        err = reddit_whats_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case DISCUSS:
      if (!value && argp)
        err = reddit_discuss_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case LS:
      if (!value && argp)
        err = reddit_ls_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case WHOAMI:
      if (!value && argp)
        err = reddit_whoami_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case RELOAD:
      if (!value && argp)
        err = reddit_reload_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case PAGESIZE:
      if (!value)
        err = reddit_page_size (state);
      else
        err = reddit_set_page_size (state, value);
      break;
    case PWD:
      if (!value)
        {
          if (state->subreddit == NULL)
            reddintf (state, "all\n");
          else
            reddintf (state, "%s\n", state->subreddit);
        }
      else
        reddit_unknown_command (state);
      break;
    case QUIT:
      if (!value)
        {
          if (quit)
            *quit = 1;
        }
      else
        reddit_unknown_command (state);
      break;
    default:
        {
          char *prev = NULL;
          char *reddit;
          int found = 0;
          while ((reddit = 
                  argz_next (state->reddits, state->reddits_len, prev)))
            {
              prev = reddit;
              if (strcasecmp (reddit, line) == 0)
                {
                  found = 1;
                  struct reddit_gosub_options_t gosub_options;
                  gosub_options.subreddit = reddit;
                  reddit_gosub (state, &gosub_options);
                  break;
                }
            }

          if (found == 0)
            {
              char *end = NULL;
              int low, high;
              reddit_get_story_id_range (state, &low, &high);
              unsigned long int val = strtoul (line, &end, 0);
              if (end == NULL || end == line || 
                  (int)val < low || (int) val > high)
                reddit_unknown_command (state);
              else
                {
                  struct reddit_view_options_t view_options;
                  view_options.story_id = (int) val;
                  reddit_view (state, &view_options);
                }
            }
        }
      break;
    }
  free (value);
  int i = 0;
  for (i = 0; i < argc; i++)
    free (argv[i]);
  free (argv);
  return err;
}

static int
reddit_parse_untrimmed_command (struct reddit_state_t *state, char *line, int *quit)
{
  int err = 0;
  if (strcmp (line, "\n") == 0 || strcmp (line, "\r\n") == 0 || *line == 0)
    return 0;
  char *trimmed_line = trim (line);
  if (trimmed_line[0] != '#') //it's not a commented out line
    err = reddit_parse_command (state, trimmed_line, quit);
  free (trimmed_line);
  return err;
}

#ifndef HAVE_LIBREADLINE
static char *
myreadline (char *prompt)
{
  char *line = NULL;
  size_t len = 0;
  fprintf (stdout, "%s", prompt);
  fflush (stdout);
  ssize_t bytesread = getline (&line, &len, stdin);
  if (bytesread == -1)
    return NULL;
  char *tmp = strrchr (line, '\n');
  if (tmp)
    tmp[0] = '\0';
  return line;
}
#endif

static int
reddit_interactive (struct reddit_state_t *state, struct reddit_options_t *arguments)
{
#ifdef HAVE_LIBREADLINE
      using_history ();
#endif
  int err = 0;
  int quit = 0;
  char *line;
  while (1)
    {
#ifdef HAVE_LIBREADLINE
      line = readline (REDDIT_PROMPT);
      if (line)
	add_history (line);
      rl_get_screen_size (NULL, &state->columns);
#else
      line = myreadline (REDDIT_PROMPT);
#endif

      err = reddit_parse_untrimmed_command (state, line, &quit);

      free (line);
      if (quit)
        break;
      if (err)
	break;

    }
  return err;
}

static void
add_default_subreddits (struct reddit_state_t *state)
{
  argz_add (&state->reddits, &state->reddits_len, "pics");
  argz_add (&state->reddits, &state->reddits_len, "reddit.com");
  argz_add (&state->reddits, &state->reddits_len, "funny");
  argz_add (&state->reddits, &state->reddits_len, "politics");
  argz_add (&state->reddits, &state->reddits_len, "AskReddit");
  argz_add (&state->reddits, &state->reddits_len, "WTF");
  argz_add (&state->reddits, &state->reddits_len, "gaming");
  argz_add (&state->reddits, &state->reddits_len, "science");
  argz_add (&state->reddits, &state->reddits_len, "worldnews");
  argz_add (&state->reddits, &state->reddits_len, "programming");
  argz_add (&state->reddits, &state->reddits_len, "atheism");
  argz_add (&state->reddits, &state->reddits_len, "technology");
  argz_add (&state->reddits, &state->reddits_len, "IAmA");
  argz_add (&state->reddits, &state->reddits_len, "comics");
  argz_add (&state->reddits, &state->reddits_len, "videos");
  argz_add (&state->reddits, &state->reddits_len, "todayilearned");
}

  
static int 
reddit_check_cookiefile_for_session_cookie (char *file)
{
  int session = 0;
  size_t len = 0;
  char *data = read_file (file, &len);
  if (data)
    {
      if (strstr (data, "reddit_session") != NULL)
        session = 1;
      free (data);
    }
  return session;
}

int 
reddit (struct reddit_options_t *arguments)
{
  struct reddit_state_t state;
  int err = 0;

  memset (&state, 0, sizeof (state));
  state.wallet = arguments->wallet;
  state.links_binary = g_find_program_in_path ("links");
  if (state.links_binary == NULL)
    {
      fprintf (stderr, _("Error: Couldn't find the `links' program in your "
                         "path.  It is required to run this software.\n"));
      return 0;
    }
  state.openssl_binary = g_find_program_in_path ("openssl");
  if (state.openssl_binary == NULL)
    {
      fprintf (stderr, _("Error: Couldn't find the `openssl' program in your "
                         "path.  It is required to run this software.\n"));
      return 0;
    }
  state.browser_binary = strdup (state.links_binary);
  if (arguments->host == NULL)
    state.site = strdup (REDDIT_SITE);
  else
    state.site = strdup (arguments->host);

  if (arguments->cookie_file == NULL)
    arguments->cookie_file= REDDIT_COOKIEJAR;

  state.banner = arguments->banner;
  state.page_size = 2;
  add_default_subreddits (&state);
  curl_global_init (CURL_GLOBAL_ALL);
  state.curl = curl_easy_init();
  curl_easy_setopt (state.curl, CURLOPT_COOKIEJAR, arguments->cookie_file);
  curl_easy_setopt (state.curl, CURLOPT_COOKIEFILE, arguments->cookie_file);
  if (arguments->proxy)
    {
      curl_easy_setopt (state.curl, CURLOPT_PROXY, arguments->proxy);
      curl_easy_setopt (state.curl, CURLOPT_PROXYPORT, arguments->proxy_port);
    }

  state.out = stdout;

  if (arguments->quiet == 0)
    reddit_welcome (&state);

  if (arguments->username)
    {
      struct reddit_login_options_t login_opts;
      memset (&login_opts, 0, sizeof (login_opts));
      login_opts.username = arguments->username;
      err = reddit_login (&state, &login_opts);
    }

#ifdef HAVE_LIBREADLINE
  rl_initialize ();
  rl_reset_screen_size ();
  rl_get_screen_size (NULL, &state.columns);
#endif

  state.logged_in = 
    reddit_check_cookiefile_for_session_cookie (arguments->cookie_file);

  if (state.logged_in) //we know we're logged in, but we don't have a username
    reddit_update_modhash (&state); //go get it. also gets our list of reddits.

  if (!err && arguments->no_initial_stories == 0)
    reddit_load_and_show_stories (&state, NULL, -1, arguments->subreddit, 
                                  NULL, NULL);

  if (!err)
    err = reddit_interactive (&state, arguments);
  curl_easy_cleanup(state.curl);
  curl_global_cleanup();
  free (state.username);
  free (state.links_binary);
  free (state.browser_binary);
  free (state.browser_options);
  free (state.openssl_binary);
  free (state.reddits);
  free (state.site);
  free (arguments->wallet);
  return err;
}
