/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <error.h>
#include <string.h>
#include <argp.h>
#include <argz.h>
#include <unistd.h>
#include "xvasprintf.h"
#include "gettext-more.h"
#include "opts.h"
#include "reddit_priv.h"

#define FULL_VERSION PACKAGE_NAME " " PACKAGE_VERSION

const char *argp_program_version = FULL_VERSION;
const char *argp_program_bug_address = "<" PACKAGE_BUGREPORT ">";

#define OAO OPTION_ARG_OPTIONAL
#define OH OPTION_HIDDEN

struct arguments_t arguments;
static struct argp_option options[] = 
{
    { "cookie-jar", OPT_COOKIEFILE, "FILE", 0, 
      N_("store/retrieve cookies from FILE") },
    { "host", OPT_HOST, "NAME", OPTION_HIDDEN, 
      N_("connect to NAME instead of " REDDIT_SITE) },
    { "login", OPT_LOGIN, "USER", 0, N_("login as USER to reddit" ) },
    { "no-initial-stories", OPT_NO_SHOW, 0, OPTION_HIDDEN, 
      N_("do not show stories initially") },
    { "proxy", OPT_PROXY, "HOST:PORT", 0, N_("use a proxy to access reddit") },
    { "subreddit", OPT_REDDIT, "NAME", 0, N_("display stories from this subreddit" ) },
    { "quiet", OPT_QUIET, NULL, 0, N_("don't show the initial legalese") },
    { 0 }
};

static void
init_options (struct reddit_options_t *app)
{
  app->quiet = -1;
  app->no_initial_stories = -1;
  app->banner = -1;
  app->script = NULL;
  app->cookie_file = NULL;
  return;
}

static char *
argp_parse_proxy (struct argp_state *state, char *p, struct reddit_state_t *r)
{
  if (g_regex_match_simple ("[\\w\\-_]+(\\.[\\w\\-_]+)+([\\w\\-\\.,@"
                            "?^=%&amp;:/~\\+#]*[\\w\\-\\@?^=%&amp;/~\\+#])?",
                            p, G_REGEX_CASELESS, 0))
    return p;
  argp_failure (state, 0, 0, _("The proxy `%s' is malformed."), p);
  return NULL;
}

static error_t 
parse_opt (int key, char *arg, struct argp_state *state) 
{
  struct arguments_t *arguments = (struct arguments_t *) state->input;

  switch (key) 
    {
    case OPT_COOKIEFILE:
      arguments->reddit.cookie_file = arg;
      break;
    case OPT_NO_SHOW:
      arguments->reddit.no_initial_stories = 1;
      break;
    case OPT_HOST:
      arguments->reddit.host = arg;
      break;
    case OPT_PROXY:
      if (argp_parse_proxy (state, arg, NULL))
        {
          arguments->reddit.proxy = arg;
          char *colon = strrchr (arg, ':');
          if (colon)
            {
              char *end = NULL;
              *colon = '\0';
              unsigned long val = strtoul (++colon, &end, 10);
              if ((int)val > 0)
                arguments->reddit.proxy_port = val;
            }
          else
            arguments->reddit.proxy_port = DEFAULT_PROXY_PORT;
        }
      break;
    case OPT_QUIET:
	arguments->reddit.quiet = 1;
      break;
    case OPT_LOGIN:
      printf("here i am2\n");
	arguments->reddit.username = arg;
      break;
    case OPT_REDDIT:
      arguments->reddit.subreddit = arg;
      break;
    case ARGP_KEY_INIT:
      init_options (&arguments->reddit);
      break;
    case ARGP_KEY_ARG:
      if (arguments->reddit.script == NULL)
        {
          arguments->reddit.banner = 0;
          arguments->reddit.no_initial_stories = 1;
          arguments->reddit.quiet = 1;
          arguments->reddit.script = arg;
        }
      else
        argp_failure (state, 1, 0, _("Only one script may be specified."));
      break;
    case ARGP_KEY_END:
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

struct argp argp = { options, parse_opt, "[SCRIPT-FILE]", 
  N_("A bandwidth-saving client for reddit.com.\v"
     "The default cookie file is `" REDDIT_COOKIEJAR "'."), 0, 0, 
  PACKAGE }; 

static void 
set_default_options (struct reddit_options_t *app)
{
  if (app->quiet == -1)
    app->quiet = atoi (DEFAULT_QUIET_VALUE);
  if (app->no_initial_stories == -1)
    app->no_initial_stories = 0;
  if (app->banner == -1)
    app->banner = 1;
  return;
}

void 
parse_opts (int argc, char **argv, struct arguments_t *arguments)
{
  int retval;
  setenv ("ARGP_HELP_FMT", "no-dup-args-note", 1);
  char *argz = NULL;
  size_t argz_len = 0;
  if (argc > 2)
    {
      if (strchr (argv[1], ' '))
        {
          argz_create_sep (argv[1], ' ', &argz, &argz_len);
        }
    }
  retval = argp_parse (&argp, argc, argv, 0, 0, arguments); 
  if (retval < 0)
    argp_help (&argp, stdout, ARGP_HELP_EXIT_ERR|ARGP_HELP_SEE,PACKAGE_NAME);
  set_default_options (&arguments->reddit);

  argp_program_version = 0;
  argp_program_bug_address = 0;
  return;
}
