/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include "xvasprintf.h"
#include "reddit_priv.h"
#include "wallet.h"
#include "getpass.h"
#include "base64.h"

char *munge_password (char *pass)
{
  char *munged_password = NULL;
  base64_encode_alloc (pass, strlen (pass), &munged_password);
  return munged_password;
}

char *demunge_password (char *munged_pass)
{
  char *pass = NULL;
  size_t outlen = 0;
  base64_decode_alloc (munged_pass, strlen (munged_pass), &pass, &outlen);
  pass = realloc (pass, outlen + 1);
  if (pass)
    pass[outlen] = '\0';
  return pass;
}

static FILE *
open_wallet (char *filename, char *passphrase, char *mode, char *o)
{
  char *cmd;

  if (strcmp (mode, "r") == 0)
    {
      FILE *fileptr = fopen (filename, mode);
      if (!fileptr)
        return NULL;
      fclose (fileptr);
      cmd = xasprintf ("%s enc -aes-256-cbc -d -in %s -k %s",
                       o, filename, passphrase);
    }
  else
    cmd = xasprintf ("%s enc -aes-256-cbc -e -k %s > %s",
                     o, passphrase, filename);

  FILE *fileptr = popen (cmd, mode);
  free (cmd);
  return fileptr;
}

static 
void close_wallet (FILE *fileptr)
{
  pclose (fileptr);
}

static int 
add_record_to_wallet (FILE *src, FILE *dest, char *user, char *site, char *pass)
{
  int added = 0;
  int same = 0;
  struct passwd pw;
  struct passwd *next = &pw;
  char buffer[1024];
  char *password = munge_password (pass);
  if (src)
    {
      while (fgetpwent_r (src, &pw, buffer, sizeof (buffer), &next) == 0)
        {
          if (strcmp (pw.pw_name, user) == 0 && strcmp (pw.pw_dir, site) == 0 &&
              strcmp (pw.pw_passwd, password) != 0)
            {
              pw.pw_passwd = password;
              added = 1;
            }
          else if (strcmp (pw.pw_name, user) == 0 && 
                   strcmp (pw.pw_dir, site) == 0 &&
                   strcmp (pw.pw_passwd, password) == 0)
            same = 1;
          putpwent (&pw, dest);
        }
    }
  if (!added && !same)
    {
      memset (&pw, 0, sizeof pw);
      pw.pw_name = user;
      pw.pw_passwd = password;
      pw.pw_dir = site;
      putpwent (&pw, dest);
      added = 1;
    }
  free (password);
  return added;
}

int 
add_to_wallet (char *filename, char *passphrase, char *username, char *website, char *password, char *openssl)
{
  FILE *fileptr = fopen (filename, "r");
  if (!fileptr)
    {
      char *p = initialize_wallet (filename, passphrase, openssl);
      free (p);
    }
  else
    fclose (fileptr);
  FILE *src = open_wallet (filename, passphrase, "r", openssl);
  char *tmpfilename = xasprintf("%s.tmp", filename);
  FILE *dest = open_wallet (tmpfilename, passphrase, "w", openssl);
  int modified = add_record_to_wallet (src, dest, username, website, password);
  if (src)
    close_wallet (src);
  close_wallet (dest);
  remove (filename);
  rename (tmpfilename, filename);
  free (tmpfilename);

  return modified;
}

char *
initialize_wallet (char *filename, char *passphrase, char *openssl)
{ 
  char *pass = NULL;
  char *pass2 = NULL;
  int same = 1;
  if (passphrase == NULL)
    {
      pass = strdup (getpass ("Passphrase: "));
      pass2 = strdup (("Confirm Passphrase: "));
      same = strcmp (pass, pass2) == 0;
      free (pass2);
      passphrase = pass;
    }

  if (same)
    {
      char *cmd = xasprintf ("%s enc -aes-256-cbc -e -k %s > %s",
                             openssl, passphrase, filename);
      FILE *fileptr = popen (cmd, "w");
      free (cmd);
      if (!fileptr)
        {
          free (pass);
          return NULL;
        }
      pclose (fileptr);
    }
  if (passphrase)
    {
      free (pass);
      return NULL;
    }
  return passphrase;
}

char *
get_password_from_wallet_file (FILE *f, char *username, char *website)
{
  char *password = NULL;
  struct passwd pw;
  struct passwd *next = NULL;
  char buffer[1024];
  while (fgetpwent_r (f, &pw, buffer, sizeof (buffer), &next) == 0)
    {
      if (strcmp (pw.pw_name, username) == 0 &&
          strcmp (pw.pw_dir, website) == 0)
        {
          password = demunge_password (pw.pw_passwd);
          break;
        }
    }
  return password;
}

char* 
get_password_from_wallet (char *filename, char *passphrase, char *username, char *website, char *openssl)
{
  char *password = NULL;
  FILE *fileptr = open_wallet (filename, passphrase, "r", openssl);
  if (!fileptr)
    return NULL;
  password = get_password_from_wallet_file (fileptr, username, website);
  close_wallet (fileptr);
  return password;
}
