/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include <s6350.h>
#include <iso15693.h>
#include <rfid.h>
#include <rfid_qsort.h>

static int block_signals(rfid_t* rfid)
{
  sigset_t sigset;

  if(sigemptyset(&rfid->sigset) < 0 ||
     sigemptyset(&sigset) < 0 ||
     sigaddset(&sigset, SIGINT) < 0 ||
     sigaddset(&sigset, SIGQUIT) < 0 ||
     sigaddset(&sigset, SIGTERM) < 0) {
    rfid->error = errno;
    return -1;
  }

  if(sigprocmask(SIG_BLOCK, &sigset, &rfid->sigset) < 0) {
    rfid->error = errno;
    return -1;
  }

  return 0;
}

static int release_signals(rfid_t* rfid)
{
  if(sigprocmask(SIG_SETMASK, &rfid->sigset, NULL) < 0) {
    rfid->error = errno;
    return -1;
  }

  if(sigemptyset(&rfid->sigset) < 0) {
    rfid->error = errno;
    return -1;
  }

  return 0;
}

int rfid_init(rfid_t* rfid)
{
  int retval = -1;
  rfid_reader_t* reader = 0;

  rfid->error = 0;
  if(sigemptyset(&rfid->sigset) < 0)
    return -1;

  if(rfid->options & RFID_OPTION_ISO15693) {
    if(iso15693_reader_alloc(&rfid->reader) < 0)
      goto finish;
  } else {
    if(s6350_reader_alloc(&rfid->reader) < 0)
      goto finish;
  }

  reader = rfid->reader;

  if(rfid->options & RFID_OPTION_DUMP_FRAMES)
    reader->verbose_f(reader, 2);

  reader->device = rfid->device;

  if(block_signals(rfid) < 0)
    goto finish;
  if(reader->init_f(reader) < 0)
    goto finish;

  rfid->last = rfid_transponder_alloc(rfid);

  retval = 0;

 finish:
  if(release_signals(rfid) < 0)
    return -1;

  return retval;
}

rfid_transponder_t* rfid_transponder_alloc(rfid_t* rfid)
{
  rfid_reader_t* reader = rfid->reader;
  rfid_transponder_t* transponder = 0;

  reader->transponder_alloc_f(reader, &transponder);

  return transponder;
}

void rfid_transponder_free(rfid_t* rfid, rfid_transponder_t* transponder)
{
  rfid_reader_t* reader = rfid->reader;

  reader->transponder_free_f(reader, transponder);
}

char* rfid_transponder_describe(rfid_t* rfid, rfid_transponder_t* transponder)
{
  rfid_reader_t* reader = rfid->reader;

  return reader->transponder_describe_f(reader, transponder);
}

int rfid_transponder_id_get(rfid_t* rfid, const rfid_transponder_t* transponder, char** idp)
{
  rfid_reader_t* reader = rfid->reader;

  return reader->transponder_id_get_f(reader, transponder, idp);
}

int rfid_transponder_id_set(rfid_t* rfid, rfid_transponder_t* transponder, char* id)
{
  rfid_reader_t* reader = rfid->reader;

  return reader->transponder_id_set_f(reader, transponder, id);
}

void rfid_end(rfid_t* rfid)
{
  rfid_transponder_free(rfid, rfid->last);
  rfid->reader->end_f(rfid->reader);
  rfid->reader->free_f(rfid->reader);
}

char* rfid_strerror(rfid_t* rfid)
{
  rfid_reader_t* reader = rfid->reader;
  char* message = (char*)malloc(512);
  
  sprintf(message, "Device %s: ", reader->device);

  if(rfid->error) {
      sprintf(message + strlen(message), "%d: %s", rfid->error, strerror(rfid->error));
  } else {
    char* error_string = reader->strerror_f(reader);
    sprintf(message + strlen(message), "%s", error_string);
    free(error_string);
  }

  return message;
}

int rfid_error(rfid_t* rfid)
{
  rfid_reader_t* reader = rfid->reader;

  if(rfid->error) {
    return rfid->error;
  } else {
    return reader->error_f(reader);
  }
}

static void transponders_grow(rfid_t* rfid, rfid_transponder_t*** transpondersp, int* transponders_lengthp)
{
  rfid_reader_t* reader = rfid->reader;
  int i;
  rfid_transponder_t** transponders = *transpondersp;
  int old_transponders_length = *transponders_lengthp;
  int transponders_length = *transponders_lengthp ? ((*transponders_lengthp) * 2) : 8;
  int size =  transponders_length * sizeof(rfid_transponder_t*);

  if(old_transponders_length > 0) {
      transponders = (rfid_transponder_t**)realloc(transponders, size);
  } else {
      transponders = (rfid_transponder_t**)malloc(size);
  }

  for(i = old_transponders_length; i < transponders_length; i++) {
    reader->transponder_alloc_f(reader, transponders + i);
  }

  *transpondersp = transponders;
  *transponders_lengthp = transponders_length;
}

static void transponders_free(rfid_t* rfid, rfid_transponder_t** transponders, int transponders_length)
{
  if(transponders) {
    int i;
    rfid_reader_t* reader = rfid->reader;

    for(i = 0; i < transponders_length; i++) {
      reader->transponder_free_f(reader, transponders[i]);
    }
    free(transponders);
  }
}

static int transponders_cmp(rfid_t* rfid, rfid_transponder_t** a, rfid_transponder_t** b)
{
  rfid_reader_t* reader = rfid->reader;
  return reader->transponder_cmp_f(reader, *a, *b);
}

static void transponders_sort(rfid_t* rfid, rfid_transponder_t** transponders, int transponders_length)
{
  rfid_qsort(transponders, transponders_length, sizeof(rfid_transponder_t*), (rfid_qsort_cmp)transponders_cmp, rfid);
}

typedef struct {
  int length;
  int size;
  rfid_transponder_t** transponders;
} rfid_transponders_t;

typedef struct {
  int events_mask;
  rfid_transponder_callback_t* transponder_callback;
  rfid_inventory_callback_t* inventory_callback;
  void* data;
  
  rfid_transponders_t current;
  rfid_transponders_t cache;
} rfid_events_context_t;

static int transponders_diff(rfid_t* rfid, rfid_events_context_t* context, int* changedp)
{
  rfid_transponders_t* current = &context->current;
  rfid_transponders_t* cache = &context->cache;
  int events_mask = context->events_mask;
  rfid_transponder_callback_t* transponder_callback = context->transponder_callback;
  void* data = context->data;
  rfid_reader_t* reader = rfid->reader;

  int current_index;
  int cache_index;

  /*
   * Compare the cache with the current transponders until one of the
   * list is exhausted.
   */
  for(current_index = 0, cache_index = 0;
      current_index < current->length || cache_index < cache->length;) {
    int retval;
    /*
     * -1 => transponders[i] is new on the RFID reader
     * 1 => cache[i] was removed from the RFID reader
     * 0 => transponders[i] is still on the RFID reader
     */
    int compare;

    if(current_index < current->length && cache_index < cache->length) {
      compare = reader->transponder_cmp_f(reader, current->transponders[current_index], cache->transponders[cache_index]);
    } else if(current_index < current->length) {
      /*
       * What's left in current transponders is new because it matches
       * none of the cached transponders.
       */
      compare = -1;
    } else {
      /*
       * What's left in transponders cache is deleted because it matches
       * none of the current transponders.
       */
      compare = 1;
    }

    if(compare < 0) {
      (*changedp)++;
      if(rfid->verbose == 1) fprintf(stderr, "N");
      if(block_signals(rfid) < 0)
	return RFID_CALLBACK_ERROR;
      /*
       * Fill structural information about the transponder.
       */
      if(reader->transponder_present_f(reader, current->transponders[current_index]) < 0) {
	/*
	 * The state changed, try again.
	 */
	if(reader->error_f(reader) == ENOENT)
	  rfid->error = EAGAIN;
	release_signals(rfid);
	return RFID_CALLBACK_ERROR;
      }
      /*
       * Read all data from it.
       */
      if(reader->read_f(reader, 0, current->transponders[current_index]->blocks, current->transponders[current_index]) < 0) {
	/*
	 * The state changed, try again.
	 */
	if(reader->error_f(reader) == ENOENT)
	  rfid->error = EAGAIN;
	release_signals(rfid);
	return RFID_CALLBACK_ERROR;
      }

      if(release_signals(rfid) < 0)
	return RFID_CALLBACK_ERROR;

      if(events_mask & RFID_EVENT_NEW) {
	if((retval = (*transponder_callback)(rfid, RFID_EVENT_NEW, current->transponders[current_index], data)) != 0) 
	  return retval;
      }
      current_index++;
    } else if(compare > 0) {
      (*changedp)++;
      if(rfid->verbose == 1) fprintf(stderr, "D");
      if(events_mask & RFID_EVENT_DELETE) {
	if((retval = (*transponder_callback)(rfid, RFID_EVENT_DELETE, cache->transponders[cache_index], data)) != 0) 
	  return retval;
      }
      cache_index++;
    } else if(compare == 0) {
      /*
       * Copy whatever is known about the transponder from the cache instead
       * of requesting data with a new present/read.
       */
      reader->transponder_copy_f(reader, current->transponders[current_index], cache->transponders[cache_index]);
      if(rfid->verbose == 1) fprintf(stderr, "A");
      if(events_mask & RFID_EVENT_AGAIN) {
	if((retval = (*transponder_callback)(rfid, RFID_EVENT_AGAIN, current->transponders[current_index], data)) != 0) 
	  return retval;
      }
      current_index++;
      cache_index++;
    }
  }

  return RFID_CALLBACK_OK;
}

static int event_loop(rfid_t* rfid, rfid_events_context_t* context)
{
  int retval = RFID_CALLBACK_OK;
  /* 5 milliseconds */
  struct timespec zzz = {
    0,
    5000000,
  };
  rfid_transponders_t* current = &context->current;
  rfid_transponders_t* cache = &context->cache;
  int events_mask = context->events_mask;
  rfid_inventory_callback_t* inventory_callback = context->inventory_callback;
  rfid_transponder_callback_t* transponder_callback = context->transponder_callback;
  rfid_reader_t* reader = rfid->reader;
  void* data = context->data;

  if((events_mask & (RFID_EVENT_NEW|RFID_EVENT_DELETE|RFID_EVENT_AGAIN)) &&
     !transponder_callback) {
    if(rfid->verbose) fprintf(stderr, "rfid_event_loop: events_mask set to one of RFID_EVENT_NEW, RFID_EVENT_DELETE or RFID_EVENT_AGAIN but transponder_callback argument is null\n");
    rfid->error = EINVAL;
    return -1;
  }

  if((events_mask & RFID_EVENT_INVENTORY) && !inventory_callback) {
    if(rfid->verbose) fprintf(stderr, "rfid_event_loop: events_mask set to one of RFID_EVENT_INVENTORY but inventory_callback argument is null\n");
    rfid->error = EINVAL;
    return -1;
  }
  
  rfid->error = 0;

  while(retval == RFID_CALLBACK_OK) {
    int changed = 0;

    if(block_signals(rfid) < 0)
      return -1;

    if(reader->inventory_f(reader, &current->transponders, &current->length, &current->size) < 0) {
      int inventory_error = reader->error_f(reader);
      if(release_signals(rfid) < 0)
	return -1;
      /*
       * Inventory tells us that it needs more room.
       */
      if(inventory_error == E2BIG) {
	transponders_grow(rfid, &current->transponders, &current->size);
	transponders_grow(rfid, &cache->transponders, &cache->size);
	continue;
      }
      /*
       * Ignore recoverable errors.
       */
      if(inventory_error == EAGAIN) {
	continue;
      }

      return -1;
    }

    if(release_signals(rfid) < 0)
      return -1;

    transponders_sort(rfid, current->transponders, current->length);

    if((retval = transponders_diff(rfid, context, &changed)) < 0) {
      /*
       * Recoverable error (typically the state changed in the middle of
       * the analysis of the current state).
       */
      if(rfid->error == EAGAIN) {
	rfid->error = 0;
	retval = RFID_CALLBACK_OK;
	continue;
      }
      /*
       * Everything else is a genuine error, either from the rfid
       * library or because a callback returned on error.
       */
      return -1;
    }

    if((retval == RFID_CALLBACK_OK) && changed && (events_mask & RFID_EVENT_INVENTORY)) {
      if((retval = (*inventory_callback)(rfid, current->transponders, current->length, data)) < 0)
	return -1;
    }
    
    /*
     * Swap cache & current transponders. 
     */
    {
      rfid_transponders_t tmp;

      tmp = *cache;
      *cache = *current;
      *current = tmp;
    }

    /*
     * Short pause to avoid using all the CPU. More a safety
     * measure than really usefull since the delay imposed
     * by the serial line dialog are more than likely to introduce
     * enough delay anyway.
     */
    nanosleep(&zzz, 0);
  }

  return 0;
}

int rfid_event_loop(rfid_t* rfid, int events_mask, rfid_transponder_callback_t* transponder_callback, rfid_inventory_callback_t* inventory_callback, void* data)
{
  int retval = -1;
  rfid_events_context_t context;

  memset(&context, '\0', sizeof(rfid_events_context_t));

  context.events_mask = events_mask;
  context.transponder_callback = transponder_callback;
  context.inventory_callback = inventory_callback;
  context.data = data;

  transponders_grow(rfid, &context.current.transponders, &context.current.size);
  transponders_grow(rfid, &context.cache.transponders, &context.cache.size);

  retval = event_loop(rfid, &context);

  transponders_free(rfid, context.current.transponders, context.current.size);
  transponders_free(rfid, context.cache.transponders, context.cache.size);

  if(release_signals(rfid) < 0)
    return -1;

  return retval;
}

int rfid_write(rfid_t* rfid, rfid_transponder_t* transponder)
{
  rfid_reader_t* reader = rfid->reader;
  rfid_transponder_t* current_transponder = 0;
  int retval = -1;

  /* 5 milliseconds */
  struct timespec zzz = {
    0,
    5000000,
  };

  if(reader->transponder_alloc_f(reader, &current_transponder) < 0)
    goto finish;

  rfid->error = 0;

  while(1) {
    if(block_signals(rfid) < 0)
      goto finish;

    if(reader->transponder_present_f(reader, current_transponder) < 0) {

      if(reader->error_f(reader) == ENOENT) {
	/*
	 * No RFID transponder, try again.
	 */
	if(rfid->verbose == 1) fprintf(stderr, ".");

      } else if(reader->error_f(reader) == EAGAIN) {
	/*
	 * Recoverable error, try again.
	 */
	if(rfid->verbose == 1) fprintf(stderr, "!");

      } else {
	/*
	 * Unrecoverable error, abort.
	 */
	goto finish;
      }
    } else {
      /*
       * The RFID transponder is available, proceed.
       */
      break;
    }

    if(release_signals(rfid) < 0)
      goto finish;

    nanosleep(&zzz, 0);
  }

  if(!reader->transponder_null_f(reader, transponder) &&
     reader->transponder_cmp_f(reader, current_transponder, transponder)) {
    rfid->error = EXDEV;
    goto finish;
  }

  memcpy(current_transponder->data, transponder->data, transponder->data_length);
  current_transponder->data_length = transponder->data_length;

  if(reader->write_f(reader, 0, current_transponder->blocks, current_transponder) < 0)
    goto finish;

  if(reader->transponder_copy_f(reader, transponder, current_transponder) < 0)
    goto finish;
  
  retval = 0;

 finish:

  if(current_transponder) {
    if(reader->transponder_free_f(reader, current_transponder) < 0)
      return -1;
  }

  if(release_signals(rfid) < 0)
    return -1;

  return retval;
}

const char* rfid_version(void)
{
  return RFID_VERSION;
}
