/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <memory.h>
#include <getopt.h>

#include "rfid.h"

/*
 * Context of the commander.
 */
typedef struct {
  rfid_t* rfid;
  int verbose;
  int max_count;
  int iso;
} commander_t;

/*
 * Append rfid(3) error message to the error message.
 */
static void report_error(rfid_t* rfid, const char* message)
{
  char* str = rfid_strerror(rfid);
  fprintf(stderr, "%s: %s\n", message, str);
  free(str);
}

/*
 * Implements --run action when a RFID transponder is found.
 * Call system(3) on RFID transponder data.
 */
static int transponder_callback(rfid_t* rfid, int event, const rfid_transponder_t* transponder, void* data)
{
  commander_t* commander = (commander_t*)data;

  if(commander->max_count > 0)
    commander->max_count--;

  if(commander->verbose > 1) {
    char* transponder_description = rfid->reader->transponder_describe_f(rfid->reader, transponder);

    fprintf(stderr, "\
========================================\n\
%s",
	  transponder_description);
    free(transponder_description);
    {
      int i;
      for(i = 0; i < transponder->data_length; i++)
	fprintf(stderr, "%c(0x%02x)/", transponder->data[i], transponder->data[i]);
      fprintf(stderr, "\n");
    }
  }

  {
    int retval = system(transponder->data);
    if(commander->verbose)
      fprintf(stderr, "%s returned %d\n", transponder->data, retval);
  }

  if(commander->max_count >= 0)
    return commander->max_count == 0 ? RFID_CALLBACK_END : RFID_CALLBACK_OK;
  else
    return RFID_CALLBACK_OK;
}

/*
 * Implements --run, infinite loop reading commands from RFID transponders.
 */
static int run_daemon(commander_t* commander)
{
  if(rfid_event_loop(commander->rfid, RFID_EVENT_NEW, transponder_callback, 0, commander) < 0) {
    report_error(commander->rfid, "rfid_read");
    return -1;
  }

  return 0;
}

/*
 * Wait for a RFID transponder to be presented to the RFID reader
 * and store the arguments (white space separated) into it.
 */
static int add_command(commander_t* commander, int argc, char** argv)
{
  rfid_t* rfid = commander->rfid;
  rfid_transponder_t transponder;
  int i;

  transponder.data_length = 0;
  memset(transponder.data, '\0', RFID_TRANSPONDER_DATA_SIZE);
  for(i = 0; i < argc; i++) {
    int index = transponder.data_length;
    int length = strlen(argv[i]);
    /* Only add separator if not last argument */
    int space = (i + 1 < argc) ? 1 : 0;
    transponder.data_length += length + space;
    if(transponder.data_length > RFID_TRANSPONDER_DATA_SIZE) {
      fprintf(stderr, "\
The shell command is too long (%d bytes) to fit in the RFID transponder\n\
storage (maximum %d bytes).\n", transponder.data_length, RFID_TRANSPONDER_DATA_SIZE);
      return -1;
    }
    memcpy(transponder.data + index, argv[i], length);
    if(space)
      transponder.data[index + length] = ' ';
  }

  if(commander->verbose > 1)
    fprintf(stderr, "Attempting to write '%s' to a transponder ... ", transponder.data);
  
  if(rfid_write(rfid, &transponder) < 0) {
    report_error(rfid, "rfid_write");
    return -1;
  }

  if(commander->verbose > 1)
    fprintf(stderr, "done\n");

  return 0;
}

/*
 * Terse manual, see rfid_commander(3) for more.
 */
static void usage(const char* message)
{
  fprintf(stderr, "%s\n\n\
rfid_commander [--help] [--device special] [--verbose] [--iso] [--count n] [--run]\n\
rfid_commander [--help] [--device special] [--verbose] [--iso] [--add] cmd arg arg ...\n\
    --run              wait for a transponder and run the command\n\
                       memorized in it. Loop forever.\n\
    --count n          loop n times instead of looping forever.\n\
    --add              wait for a transponder and store the command\n\
                       given in argument in its memory.\n\
    --iso              interact with ISO-15693 conformant transponder.\n\
    --device special   dialog with RFID reader connected to special\n\
                       file <special> (default /dev/ttyS0)\n\
    --help             print this message\n\
    --verbose          increase verbosity level by 1\n\
", message);
}

/*
 * Run commands from RFID transponders or store commands into them.
 */
int main(int argc, char** argv) {
  int retval = 0;
  int c;
  int option_run = 0;
  int option_add = 0;
  rfid_t rfid;
  commander_t commander;

  memset(&rfid, '\0', sizeof(rfid_t));
  rfid.device = "/dev/ttyS0";

  commander.rfid = &rfid;
  commander.verbose = 0;
  commander.iso = 0;
  commander.max_count = -1;

  while (1) {
    int option_index = 0;
    static struct option long_options[] = {
      {"add", 0, 0, 'a'},
      {"run", 0, 0, 'r'},
      {"iso", 0, 0, 'i'},
      {"device", 1, 0, 'd'},
      {"count", 1, 0, 'c'},
      {"help", 0, 0, 'h'},
      {"verbose", 0, 0, 'v'},
      {0, 0, 0, 0}
    };

    c = getopt_long (argc, argv, "arivd:c:", long_options, &option_index);
    if (c == -1)
      break;

    switch (c) {
    case 'r':
      option_run++;
      break;

    case 'a':
      option_add++;
      break;

    case 'i':
      commander.iso++;
      break;

    case 'c':
      commander.max_count = atoi(optarg);
      break;

    case 'v':
      commander.verbose++;
      break;

    case 'h':
      usage("Usage help:");
      return 0;
      break;

    case 'd':
      rfid.device = strdup(optarg);
      break;

    case '?':
      break;

    default:
      fprintf(stderr, "?? getopt returned character code 0%o ??\n", c);
    }
  }

  if(option_run && option_add) {
    usage("Options --run and --add are mutually exclusive");
    return -1;
  }
  
  if(commander.max_count > 0 && !option_run) {
    usage("Options --count is only meaningfull with --run");
    return -1;
  }
  
  if(commander.iso)
    rfid.options |= RFID_OPTION_ISO15693;

  rfid.verbose = commander.verbose;

  if(commander.verbose > 2)
    rfid.options |= RFID_OPTION_DUMP_FRAMES;

  if(rfid_init(&rfid) < 0) {
    report_error(&rfid, "rfid_init");
    return -1;
  }
  
  if(option_run)
    retval = run_daemon(&commander);
  else if(option_add) {
    if (optind < argc) {
      retval = add_command(&commander, argc - optind, argv + optind);
    } else {
      usage("You must provide arguments when using --add");
      retval = -1;
    }
  } else
    usage("You must specify either --add or --run");

  rfid_end(&rfid);
      
  return retval;
}
