/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <s6350.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <signal.h>

static void report_error(rfid_reader_t* reader, const char* message)
{
  char* str = reader->strerror_f(reader);
  fprintf(stderr, "%s: %s\n", message, str);
  free(str);
}

static s6350_t* s6350;
static rfid_reader_t* reader;

static RETSIGTYPE alarm_f(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to establish communication with RFID reader at %s.\n\
FAILURE: Make sure a RFID reader is plugged in and turned on.\n\
\n\
", reader->device);
  exit(-1);
}

int main(int argc, char** argv)
{
  int timeout = 5;

  if(s6350_reader_alloc(&reader) < 0)
    return -1;
  s6350 = (s6350_t*)reader->private;

  if(argc > 1)
    reader->device = argv[1];
  else
    reader->device = "/dev/ttyS0";

  if(signal(SIGALRM, alarm_f) < 0) {
    perror("signal(SIGALRM)");
    return -1;
  }

  alarm(timeout);
  if(reader->init_f(reader) < 0) {
    report_error(reader, "init_f");
  }
  alarm(0);

  {
    s6350_transponder_t transponder;
    rfid_block_t block = { 0 };
   
    if(s6350_transponder_details(s6350, &transponder)) {
      if(reader->error_f(reader) == ENOENT) {
	fprintf(stderr, "\
\n\
FAILURE: The test program will attempt to read / write on a transponder,\n\
FAILURE: please put a transponder on the RFID reader.\n\
\n\
");
      }
      report_error(reader, "s6350_transponder_details");
      return -1;
    }

    block.data = (u_int8_t*)malloc(transponder.bytes_per_block);

    block.data[0] = block.data[1] = block.data[2] = block.data[3] = 'B';
    block.block_number = 0;

    if(s6350_write_block(s6350, &transponder, &block)) {
      report_error(reader, "s6350_write_block");
      return -1;
    }

    for(block.block_number = 0; block.block_number < transponder.blocks; block.block_number++) {
	
      if(s6350_read_block(s6350, &transponder, &block)) {
	report_error(reader, "s6350_read_block");
	return -1;
      }
    }

    free(block.data);

    {
      rfid_block_t blocks[32];
      int i;

      for(i = 0; i < 32; i++)
	blocks[i].data = (u_int8_t*)malloc(transponder.bytes_per_block);

      if(s6350_special_read_block(s6350, &transponder, 0xFF, blocks)) {
	report_error(reader, "s6350_special_read_block");
	return -1;
      }

      for(i = 0; i < 32; i++)
	free(blocks[i].data);

    }

  }
  
  reader->end_f(reader);
  reader->free_f(reader);
  
  return 0;
}
