/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include <rfid_error.h>
#include <rfid_reader.h>
#include <s6350.h>
#include <iso15693.h>

/*
 * Combine/split an int containing the command and the command flags
 */
#define ISO15693_COMMAND_COMBINE(command,flags) ((command)|((flags) << 8))
#define ISO15693_COMBINED_FLAGS(combined) ((combined >> 8) & 0xFF)
#define ISO15693_COMBINED_COMMAND(combined) (combined & 0xFF)

static int error_map[256] = {
  /* 0x00 */ RFID_ERROR_UNKNOWN_ERROR, 
  /* 0x01 */ RFID_ERROR_NOT_SUPPORTED, /* "The command is not supported, i.e. the request code is not recognized" */
  /* 0x02 */ RFID_ERROR_NOT_SUPPORTED, /* "The command is not recognized, for example: a format error occured" */
  /* 0x03 */ RFID_ERROR_READER_OPTION_NOT_SUPPORTED, /* "The command option is not supported" */
  /* 0x04 */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x05 */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x06 */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x07 */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x08 */ RFID_ERROR_UNKNOWN_ERROR, 
  /* 0x09 */ RFID_ERROR_UNKNOWN_ERROR, 
  /* 0x0a */ RFID_ERROR_UNKNOWN_ERROR, 
  /* 0x0b */ RFID_ERROR_UNKNOWN_ERROR, 
  /* 0x0c */ RFID_ERROR_UNKNOWN_ERROR, 
  /* 0x0d */ RFID_ERROR_UNKNOWN_ERROR, 
  /* 0x0e */ RFID_ERROR_UNKNOWN_ERROR, 
  /* 0x0f */ RFID_ERROR_UNKNOWN_ERROR, /* "Error with no information given or a specific error code is not supported" */
  /* 0x10 */ RFID_ERROR_NO_BLOCK, /* "The specified block is not available (doesn't exist)" */
  /* 0x11 */ RFID_ERROR_BLOCK_ALREADY_LOCKED, /* "The specified block is already locked and thus cannot be locked again" */
  /* 0x12 */ RFID_ERROR_BLOCK_READ_ONLY, /* "The specified block is locked and its content cannot be changed" */
  /* 0x13 */ RFID_ERROR_BLOCK_WRITE_FAILURE, /* "The specified block was not successfully programmed" */
  /* 0x14 */ RFID_ERROR_BLOCK_LOCK_FAILURE, /* "The specified block was not successfully locked" */
  /* 0x15 */ RFID_ERROR_TRANSPONDER_NOT_FOUND, /* not ISO-15693 "Transponder not found" */
  /* 0x16 */ RFID_ERROR_INVALID_FLAGS, /* not ISO-15693 "Invalid flags for command" */
  /* 0x17 */ RFID_ERROR_ISO15693_COLLISION, /* not ISO-15693 "Collision during inventory" */
  /* 0x18 */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x19 */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x1a */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x1b */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x1c */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x1d */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x1e */ RFID_ERROR_UNKNOWN_ERROR,
  /* 0x1f */ RFID_ERROR_UNKNOWN_ERROR,
};

#define clear_error(o) ((o)->error = 0)

/*
 * Return 0 if the transponders are equal, -1 if a < b, 1 if a > b.
 * Comparison is made on the UID of each transponder.
 */
static int iso15693_transponder_cmp(iso15693_transponder_t* a, iso15693_transponder_t* b)
{
  return memcmp(a->uid, b->uid, ISO15693_UID_SIZE);
}

/*
 * Return true if the UID of the transponder is all 0, return false otherwise.
 */
static int iso15693_transponder_null(iso15693_transponder_t* transponder)
{
  int i;

  for(i = 0; i < ISO15693_UID_SIZE; i++)
    if(transponder->uid[i])
      return 0;

  return 1;
}

/*
 * The Cyclic Redundancy Check (CRC) is calculated on all data
 * contained in a message, from the start of the flags through to the
 * end of data. This CRC is used from VCD to VICC and from VICC to
 * VCD.
 *
 *                              CRC Definition
 *  CRC type       Length    Polynomial	                 Direction  Preset  Residue
 *  ISO/IEC 13239  16 bits   X16 + X12 + X5 + 1 = '8408' Backward   'FFFF'  'F0B8'
 *
 * To add extra protection against shifting errors, a further
 * transformation is made on the calculated CRC. The value attached to
 * the message for transmission is the one's complement of the
 * calculated CRC. For checking of received messages the 2 CRC bytes
 * are included in the re-calculation. In this case, for the expected
 * value for the generated CRC the value of the residue is 'F0B8'.
 *
 * ISO/IEC 15693-3:2001 page 47.
 */

#define ISO15693_CRC16_PRESET		0xFFFF
#define ISO15693_CRC16_POLYNOMIAL	0x8408
#define ISO15693_CRC16_RESIDUE		0xF0B8

static u_int16_t crc16_compute(u_int8_t* buffer, int buffer_size)
{
  u_int16_t crc16 = ISO15693_CRC16_PRESET;
  int i;
  for (i = 0; i < buffer_size; i++) {
    int j;
    crc16 = crc16 ^ ((unsigned int)buffer[i]);
    for (j = 0; j < 8; j++) {
      if (crc16 & 0x0001)
	crc16 = (crc16 >> 1) ^ ISO15693_CRC16_POLYNOMIAL;
      else
	crc16 = (crc16 >> 1);
    }
  }

  return crc16;
}


void iso15693_crc16_set(u_int8_t* request, int request_length)
{
  u_int16_t crc16 = ~ ( crc16_compute(request, request_length - 2) );
  request[request_length - 2] = crc16 & 0xFF;
  request[request_length - 1] = (crc16 >> 8) & 0xFF;
}

int iso15693_crc16_check(u_int8_t* request, int request_length)
{
  u_int16_t residue = crc16_compute(request, request_length);
  return (residue & 0xFFFF) == ISO15693_CRC16_RESIDUE;
}

static int write_command(iso15693_t* iso15693, int command, u_int8_t* buffer, int buffer_size, const rfid_transponder_t* transponder)
{
  rfid_reader_t* driver = iso15693->reader;

  char* request = (char*)malloc(buffer_size + ISO15693_COMMAND_OVERHEAD);
  short request_length = buffer_size + ISO15693_COMMAND_OVERHEAD;
  int retval;

  request[ISO15693_FLAGS_INDEX] = ISO15693_COMBINED_FLAGS(command);
  request[ISO15693_COMMAND_INDEX] = ISO15693_COMBINED_COMMAND(command);

  memcpy(request + ISO15693_COMMAND_HEADER_SIZE, buffer, buffer_size);

  iso15693_crc16_set(request, request_length);
  
  if(iso15693->verbose > 1) {
    fprintf(stderr, "ISO-15693 command : ");
    rfid_dump_frame(request, request_length);
    fprintf(stderr, "\n");
  }

  if((retval = driver->iso15693_command_f(driver, request, request_length, transponder)) < 0)
    iso15693->error = driver->error_f(driver);

  free(request);

  return retval;
  
}

static int read_response(iso15693_t* iso15693, u_int8_t** bufferp, int* buffer_sizep, const rfid_transponder_t* transponder)
{
  rfid_reader_t* driver = iso15693->reader;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(driver->iso15693_response_f(driver, &buffer, &buffer_size, transponder) < 0)
    return -1;

  if(iso15693->verbose > 1) {
    fprintf(stderr, "ISO-15693 response: ");
    rfid_dump_frame(buffer, buffer_size);
    fprintf(stderr, "\n");
  }

  if(!iso15693_crc16_check(buffer, buffer_size)) {
    free(buffer);
    iso15693->error = RFID_ERROR_ISO15693_CHECKSUM;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  if(buffer[ISO15693_FLAGS_INDEX] & ISO15693_RESPONSE_FLAG_ERROR) {
    iso15693->error = error_map[buffer[ISO15693_ERROR_INDEX]];
    free(buffer);
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  buffer_size -= ISO15693_RESPONSE_OVERHEAD;
  memmove(buffer, buffer + ISO15693_RESPONSE_HEADER_SIZE, buffer_size);

  *bufferp = buffer;
  *buffer_sizep = buffer_size;

  return 0;
}

static int simple_command(iso15693_t* iso15693, u_int8_t command, const u_int8_t* uid)
{
  u_int8_t request[ISO15693_UID_SIZE];
  u_int8_t null_uid[ISO15693_UID_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0 };
  int request_length;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(memcmp(uid, null_uid, ISO15693_UID_SIZE)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(command,flags), request, request_length, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  if(read_response(iso15693, &buffer, &buffer_size, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  free(buffer);

  return 0;
}

int iso15693_inventory(iso15693_t* iso15693, iso15693_inventory_t* inventory, iso15693_inventory_result_t* result)
{
  u_int8_t* buffer;
  int buffer_size;
  u_int8_t request[ISO15693_AFI_SIZE + ISO15693_MASK_LENGTH_SIZE + ISO15693_MASK_VALUE_SIZE] = { 0 };
  int request_length = 0;
  int index = 0;
  int flags = ISO15693_COMMAND_FLAGS_INVENTORY |
    (inventory->one_slot ? ISO15693_COMMAND_FLAGS_NB_SLOTS : 0);

  clear_error(iso15693);

  /*
   * AFI field
   */
  if(inventory->afi) {
    request[index] = inventory->afi;
    index += ISO15693_AFI_SIZE;
    flags |= ISO15693_COMMAND_FLAGS_AFI;
  }

  /*
   * Mask length field.
   */
  if(inventory->mask_length > (ISO15693_UID_SIZE * 8)) {
    iso15693->error = RFID_ERROR_ISO15693_MASK_TOO_LARGE;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  request[index] = inventory->mask_length & 0xFF;
  index += ISO15693_MASK_LENGTH_SIZE;

  /*
   * Mask value field
   */
  if(inventory->mask_length > 0) {
    int nbytes = (inventory->mask_length >> 3) + 1;
    int i;
    for(i = 0; i < nbytes; i++) {
      request[index++] = inventory->mask_value[i];
    }
  }

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_INVENTORY,flags), request, request_length, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  if(read_response(iso15693, &buffer, &buffer_size, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  if(buffer_size < NOT_ISO15693_VALID_DATA_FLAGS_SIZE + NOT_ISO15693_COLLISION_FLAGS_SIZE) {
    free(buffer);
    iso15693->error = RFID_ERROR_SHORT_READ;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  {
    u_int8_t* pointer = buffer;
    int max_slot = (flags & ISO15693_COMMAND_FLAGS_NB_SLOTS) ?
      ISO15693_NB_SLOTS_ONE : ISO15693_NB_SLOTS_DEFAULT;
    int slot;

    result->nvalid = 0;

    result->valid = ((pointer[1] & 0xFF) << 8) | (pointer[0] & 0xFF);
    pointer += NOT_ISO15693_VALID_DATA_FLAGS_SIZE;

    result->collide = ((pointer[1] & 0xFF) << 8) | (pointer[0] & 0xFF);
    pointer += NOT_ISO15693_COLLISION_FLAGS_SIZE;

    for(slot = 0; slot < max_slot; slot++) {
      int bit_set = 1 << slot;
      if(!(result->collide & bit_set) && (result->valid & bit_set)) {
	result->nvalid++;

	pointer += ISO15693_FLAGS_SIZE;

	result->data[slot].dsfid = *pointer;
	pointer += ISO15693_DSFID_SIZE;

	memcpy(result->data[slot].uid, pointer, ISO15693_UID_SIZE);
	pointer += ISO15693_UID_SIZE;
      }
    }
  }

  free(buffer);

  return 0;
}

int iso15693_stay_quiet(iso15693_t* iso15693, u_int8_t* uid)
{
  u_int8_t request[ISO15693_UID_SIZE];
  int request_length = ISO15693_UID_SIZE;

  clear_error(iso15693);

  memcpy(request, uid, ISO15693_UID_SIZE);
  
  return write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_STAY_QUIET,ISO15693_COMMAND_FLAGS_ADDRESS), request, request_length, 0);
}

int iso15693_read_single_block(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* block)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  int want_security_status = block->security_status;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  if(want_security_status) {
    flags |= ISO15693_COMMAND_FLAGS_OPTION;
  }
  block->security_status = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  if(block->block_number >= transponder->blocks) {
    iso15693->error = RFID_ERROR_NO_BLOCK;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  request[index] = block->block_number & 0xFF;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_READ_SINGLE_BLOCK,flags), request, request_length, transponder) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  if(read_response(iso15693, &buffer, &buffer_size, transponder) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  index = 0;

  if(want_security_status) {
    block->security_status = buffer[index];
    index += ISO15693_BLOCK_SECURITY_STATUS_SIZE;
  }

  /*
   * The read single block command implicitly tells us the size of a
   * block. If the transponder has a wrong idea of this size, fix it.
   * This is a hack to cope with RFID readers that are unable to provide
   * the layout of RFID transponders.
   */
  if(transponder->bytes_per_block != buffer_size - index) {
    ((rfid_transponder_t*)transponder)->bytes_per_block = buffer_size - index;
    if(iso15693->verbose) fprintf(stderr, "fixed transponder block size to %d\n", transponder->bytes_per_block);
  }
  
  memcpy(block->data, buffer + index, transponder->bytes_per_block);

  free(buffer);

  return 0;
}

int iso15693_write_single_block(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* block)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_SIZE_MAX];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }
  
  if(block->block_number >= transponder->blocks) {
    iso15693->error = RFID_ERROR_NO_BLOCK;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  request[index] = block->block_number;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  memcpy(request + index, block->data, transponder->bytes_per_block);
  index += transponder->bytes_per_block;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_WRITE_SINGLE_BLOCK,flags), request, request_length, transponder) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  if(read_response(iso15693, &buffer, &buffer_size, transponder) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  free(buffer);

  return 0;
}

static int read_multiple_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* blocks, int blocks_length)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_NUMBER_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  int want_security_status = blocks[0].security_status;
  int block_start = blocks[0].block_number & 0xFF;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  if(want_security_status) {
    flags |= ISO15693_COMMAND_FLAGS_OPTION;
  }
  blocks[0].security_status = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  if(block_start >= transponder->blocks ||
     block_start + blocks_length > transponder->blocks) {
    iso15693->error = RFID_ERROR_NO_BLOCK;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  /*
   * Index of the first block to read.
   */
  request[index] = block_start;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  /*
   * Number of blocks to read. The actual number of blocks read
   * is incremented by one (so that 0-255 covers 1-256).
   * ISO/IEC 15693-3:2001 page 33.
   */
  request[index] = (blocks_length - 1) & 0xFF;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_READ_MULTIPLE_BLOCKS,flags), request, request_length, transponder) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  if(read_response(iso15693, &buffer, &buffer_size, transponder) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  {
    index = 0;
    int blocks_index = 0;

    for(blocks_index = 0; blocks_index < blocks_length; blocks_index++) {
      if(want_security_status) {
	blocks[blocks_index].block_number = block_start + index;
	blocks[blocks_index].security_status = buffer[index];
	index += ISO15693_BLOCK_SECURITY_STATUS_SIZE;
      }
      memcpy(blocks[blocks_index].data, buffer + index, transponder->bytes_per_block);
      index += transponder->bytes_per_block;
    }
  }

  free(buffer);

  return 0;
}

int iso15693_read_multiple_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* blocks, int blocks_length)
{
  /*
   * Split into multiple READ_MULTIPLE_BLOCKS if the request
   * would not fit in the input buffer of the underlying reader.
   */
  int first_block = blocks[0].block_number;
  int last_block = first_block + blocks_length;
  int want_security_status = blocks[0].security_status;
  int security_status_size = want_security_status ? ISO15693_BLOCK_SECURITY_STATUS_SIZE : 0;
  int block_increment = (iso15693->reader->buffer_in_max - (ISO15693_FLAGS_SIZE + ISO15693_CRC_SIZE)) / (security_status_size + transponder->bytes_per_block);
  int block_number;

  for(block_number = first_block; block_number < last_block; block_number += block_increment) {
    int length = block_number + block_increment > last_block ? last_block - block_number : block_increment;
    int index = block_number - first_block;

    blocks[index].block_number = block_number;
    blocks[index].security_status = want_security_status;

    if(read_multiple_blocks(iso15693, transponder, blocks + index, length) < 0) {
      RFID_ERROR_STACK(iso15693->verbose);
      return -1;
    }
  }

  return 0;
}

static int write_multiple_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* blocks, int blocks_length)
{
  u_int8_t* request;
  int request_length = ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_NUMBER_SIZE + blocks_length * transponder->bytes_per_block;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int block_start = blocks[0].block_number & 0xFF;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  request = (u_int8_t*)malloc(request_length);

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  if(block_start >= transponder->blocks ||
     block_start + blocks_length > transponder->blocks) {
    iso15693->error = RFID_ERROR_NO_BLOCK;
    free(request);
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  request[index] = block_start;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request[index] = blocks_length;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  {
    int i;
    for(i = 0; i < blocks_length; i++) {
      memcpy(request + index, blocks[i].data, transponder->bytes_per_block);
      index += transponder->bytes_per_block;
    }
  }

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_WRITE_MULTIPLE_BLOCKS,flags), request, request_length, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  free(request);

  if(read_response(iso15693, &buffer, &buffer_size, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  free(buffer);

  return 0;
}

int iso15693_write_multiple_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* blocks, int blocks_length)
{
  /*
   * Split into multiple WRITE_MULTIPLE_BLOCKS if the request
   * would not fit in the output buffer of the underlying reader.
   */
  int block_increment = (iso15693->reader->buffer_out_max - (ISO15693_FLAGS_SIZE + ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_CRC_SIZE)) / transponder->bytes_per_block;
  int first_block = blocks[0].block_number;
  int last_block = first_block + blocks_length;
  int block_number;

  for(block_number = first_block; block_number < last_block; block_number += block_increment) {
    int length = block_number + block_increment > last_block ? last_block - block_number : block_increment;
    int index = block_number - first_block;

    blocks[index].block_number = block_number;

    if(write_multiple_blocks(iso15693, transponder, blocks + index, length) < 0) {
      RFID_ERROR_STACK(iso15693->verbose);
      return -1;
    }
  }

  return 0;
}

int iso15693_lock_block(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* block)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request[index] = block->block_number;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_LOCK_BLOCK,flags), request, request_length, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  if(read_response(iso15693, &buffer, &buffer_size, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  free(buffer);

  return 0;
}

int iso15693_select(iso15693_t* iso15693, const u_int8_t* uid)
{
  clear_error(iso15693);

  return simple_command(iso15693, ISO15693_COMMAND_SELECT, uid);
}

int iso15693_reset_to_ready(iso15693_t* iso15693, const u_int8_t* uid)
{
  clear_error(iso15693);

  return simple_command(iso15693, ISO15693_COMMAND_RESET_TO_READY, uid);
}

int iso15693_write_afi(iso15693_t* iso15693, const rfid_transponder_t* transponder)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_AFI_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }
  /*  flags |= ISO15693_COMMAND_FLAGS_OPTION;*/

  request[index] = iso15693_transponder->afi;
  index += ISO15693_AFI_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_WRITE_AFI,flags), request, request_length, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  if(read_response(iso15693, &buffer, &buffer_size, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  free(buffer);

  return 0;
}

int iso15693_lock_afi(iso15693_t* iso15693, const u_int8_t* uid)
{
  clear_error(iso15693);

  return simple_command(iso15693, ISO15693_COMMAND_LOCK_AFI, uid);
}

int iso15693_write_dsfid(iso15693_t* iso15693, const rfid_transponder_t* transponder)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_AFI_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request[index] = iso15693_transponder->dsfid;
  index += ISO15693_DSFID_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_WRITE_DSFID,flags), request, request_length, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  if(read_response(iso15693, &buffer, &buffer_size, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  free(buffer);

  return 0;
}

int iso15693_lock_dsfid(iso15693_t* iso15693, const u_int8_t* uid)
{
  clear_error(iso15693);

  return simple_command(iso15693, ISO15693_COMMAND_LOCK_DSFID, uid);
}

#ifndef ISO15693_GET_SYSTEM_INFORMATION_FILE
#define ISO15693_GET_SYSTEM_INFORMATION_FILE "/var/lib/rfid/system_information.db"
#endif // ISO15693_GET_SYSTEM_INFORMATION_FILE

int iso15693_get_system_information(iso15693_t* iso15693, rfid_transponder_t* transponder)
{
  u_int8_t request[ISO15693_UID_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_GET_SYSTEM_INFORMATION,flags), request, request_length, 0) < 0) {
    /*
     * Plan B if not supported by reader or tag : lookup in
     * pre-defined database.
     */
    if((iso15693->error == RFID_ERROR_NOT_SUPPORTED) &&
       (flags & ISO15693_COMMAND_FLAGS_ADDRESS)) {
      DB* db = 0;
      DBC* dbc = 0;
      DBT key;
      DBT data;
      int error;
      char uid[32];

      db = iso15693->system_information;

      /*
       * Open database of pre-configured system information that copes
       * with transponders or readers not implementing
       * GET_SYSTEM_INFORMATION.
       */
      if(db == 0) {
	if((error = db_create(&iso15693->system_information, 0, 0))) {
	  if(iso15693->verbose) fprintf(stderr, "db_create: %s\n", db_strerror(error));
	  RFID_ERROR_STACK(iso15693->verbose);
	  return -1;
	}
	db = iso15693->system_information;

	if((error = db->open(db, 0, ISO15693_GET_SYSTEM_INFORMATION_FILE, 0, DB_BTREE, DB_RDONLY|DB_THREAD, 0))) {
	  if(iso15693->verbose) fprintf(stderr, "db->open(%s): %s\n", ISO15693_GET_SYSTEM_INFORMATION_FILE, db_strerror(error));
	  db->close(db, 0);
	  iso15693->system_information = 0;
	  RFID_ERROR_STACK(iso15693->verbose);
	  return -1;
	}

	if((error = db->cursor(db, 0, &iso15693->system_information_cursor, 0))) {
	  if(iso15693->verbose) fprintf(stderr, "db->cursor: %s\n", db_strerror(error));
	  db->close(db, 0);
	  iso15693->system_information = 0;
	  RFID_ERROR_STACK(iso15693->verbose);
	  return -1;
	}
	dbc = iso15693->system_information_cursor;
      }

      sprintf(uid, "%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x",
	      iso15693_transponder->uid[7],
	      iso15693_transponder->uid[6],
	      iso15693_transponder->uid[5],
	      iso15693_transponder->uid[4],
	      iso15693_transponder->uid[3],
	      iso15693_transponder->uid[2],
	      iso15693_transponder->uid[1],
	      iso15693_transponder->uid[0]);

      memset(&key, '\0', sizeof(DBT));
      key.data = uid;
      key.size = strlen(uid);
      memset(&data, '\0', sizeof(DBT));
      data.flags = DB_DBT_MALLOC;

      /*
       * Get the first database entry with a key larger than
       * the uid (typically e0.04.01.00.00.xx.xx.xx).
       */
      if((error = dbc->c_get(dbc, &key, &data, DB_SET_RANGE))) {
	if(iso15693->verbose) fprintf(stderr, "dbc->c_get(%s): %s\n", uid, db_strerror(error));
	RFID_ERROR_STACK(iso15693->verbose);
	return -1;
      }

      {
	int i;
	for(i = 0; i < key.size; i++)
	  if(uid[i] != ((char*)key.data)[i])
	    break;

	/* The uid does not match the mask. */
	if(((char*)key.data)[i] != 'x') {
	  if(iso15693->verbose) fprintf(stderr, "dbc->c_get(%s): not found\n", uid);
	  RFID_ERROR_STACK(iso15693->verbose);
	  return -1;
	}
      }

      /* Data is XXX.XX.COMMENT */
#define DATA_MIN_SIZE 7
      if(data.size < DATA_MIN_SIZE) {
	if(iso15693->verbose) fprintf(stderr, "dbc->c_get(%s): record length < 7, possible corruption (%.*s)\n", uid, data.size, (char*)data.data);
	RFID_ERROR_STACK(iso15693->verbose);
	return -1;
      }
      ((char*)data.data)[DATA_MIN_SIZE] = '\0';

      {
	int blocks;
	int bytes_per_block;
	if(sscanf(data.data, "%d.%d",
		  &blocks,
		  &bytes_per_block) != 2) {
	  if(iso15693->verbose) fprintf(stderr, "dbc->c_get(%s): failed to parse %.*s, possible corruption\n", uid, data.size, (char*)data.data);
	  RFID_ERROR_STACK(iso15693->verbose);
	  return -1;
	}
	transponder->blocks = blocks & 0xFF;
	transponder->bytes_per_block = bytes_per_block & 0xFF;
      }
      
      return 0;
    } else {
      RFID_ERROR_STACK(iso15693->verbose);
      return -1;
    }
  }

  if(read_response(iso15693, &buffer, &buffer_size, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  {
    u_int8_t* pointer = buffer;
    u_int8_t info;

    iso15693_transponder->info_flags = info = *pointer;
    pointer += ISO15693_INFO_FLAGS_SIZE;
    
    memcpy(iso15693_transponder->uid, pointer, ISO15693_UID_SIZE);
    pointer += ISO15693_UID_SIZE;
    
    if(info & ISO15693_INFO_FLAGS_DSFID) {
      iso15693_transponder->dsfid = *pointer;
      pointer += ISO15693_DSFID_SIZE;
    }

    if(info & ISO15693_INFO_FLAGS_AFI) {
      iso15693_transponder->afi = *pointer;
      pointer += ISO15693_AFI_SIZE;
    }

    if(info & ISO15693_INFO_FLAGS_VICC_MEMORY_SIZE) {
      iso15693_transponder->vicc_memory = ((pointer[1] & 0xFF) << 8) | (pointer[0] & 0xFF);
      pointer += ISO15693_VICC_MEMORY_SIZE_SIZE;
    }

    if(info & ISO15693_INFO_FLAGS_IC_REFERENCE) {
      iso15693_transponder->ic_ref = *pointer;
      pointer += ISO15693_IC_REFERENCE_SIZE;
    }
  }

  iso15693_transponder_vicc_blocks_get(transponder, transponder->blocks);
  iso15693_transponder_vicc_bytes_per_block_get(transponder, transponder->bytes_per_block);

  free(buffer);

  return 0;
}

static int get_multiple_block_security_status(iso15693_t* iso15693, const u_int8_t* uid, rfid_block_t* blocks, int blocks_length)
{
  u_int8_t null_uid[ISO15693_UID_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0 };
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_NUMBER_SIZE];
  int request_length;
  int index = 0;
  u_int8_t flags = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(iso15693);

  if(memcmp(uid, null_uid, ISO15693_UID_SIZE)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request[index] = blocks[0].block_number & 0xFF;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request[index] = (blocks_length - 1) & 0xFF;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_GET_MULTIPLE_BLOCK_SECURITY_STATUS,flags), request, request_length, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  if(read_response(iso15693, &buffer, &buffer_size, 0) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  if(buffer_size > blocks_length) {
    iso15693->error = RFID_ERROR_TOO_SMALL;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  for(index = 0; index < buffer_size; index++)
    blocks[index].security_status = buffer[index] & 0xFF;
  
  free(buffer);

  return 0;
}

int iso15693_get_multiple_block_security_status(iso15693_t* iso15693, const u_int8_t* uid, rfid_block_t* blocks, int blocks_length)
{
  /*
   * Split into multiple GET_MULTIPLE_BLOCK_SECURITY_STATUS if the response
   * would not fit in the input buffer of the underlying reader.
   */
  int block_increment = (iso15693->reader->buffer_in_max - (ISO15693_FLAGS_SIZE + ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_CRC_SIZE)) / ISO15693_BLOCK_SECURITY_STATUS_SIZE;
  int first_block = blocks[0].block_number;
  int last_block = first_block + blocks_length;
  int block_number;

  for(block_number = first_block; block_number < last_block; block_number += block_increment) {
    int length = block_number + block_increment > last_block ? last_block - block_number : block_increment;
    int index = block_number - first_block;

    blocks[index].block_number = block_number;

    if(get_multiple_block_security_status(iso15693, uid, blocks + index, length) < 0) {
      RFID_ERROR_STACK(iso15693->verbose);
      return -1;
    }
  }

  return 0;
}

int iso15693_response_size(rfid_reader_t* reader, char* request, int request_size, int* response_size, const rfid_transponder_t* transponder)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;

  switch(request[ISO15693_COMMAND_INDEX]) {
  case ISO15693_COMMAND_INVENTORY:
    *response_size = ISO15693_COMMAND_INVENTORY_RESPONSE_SIZE;
    break;
  case ISO15693_COMMAND_STAY_QUIET:
    *response_size = 0;
    break;
  case ISO15693_COMMAND_READ_SINGLE_BLOCK:
    {
      int block_security_status_size =
	(request[ISO15693_FLAGS_INDEX] & ISO15693_COMMAND_FLAGS_OPTION) ?
	ISO15693_BLOCK_SECURITY_STATUS_SIZE :
	0;
      if(!transponder) {
	iso15693->error = RFID_ERROR_TRANSPONDER_NULL;
	RFID_ERROR_STACK(iso15693->verbose);
	return -1;
      }
      *response_size = block_security_status_size + transponder->bytes_per_block;
    }
    break;
  case ISO15693_COMMAND_READ_MULTIPLE_BLOCKS:
    {
      int block_security_status_size =
	(request[ISO15693_FLAGS_INDEX] & ISO15693_COMMAND_FLAGS_OPTION) ?
	ISO15693_BLOCK_SECURITY_STATUS_SIZE :
	0;
      int uid_size = (request[ISO15693_FLAGS_INDEX] & ISO15693_COMMAND_FLAGS_ADDRESS) ? ISO15693_UID_SIZE : 0;
      int first_block_index = ISO15693_FLAGS_SIZE + ISO15693_COMMAND_SIZE + uid_size;
      int number_of_blocks_index = first_block_index + ISO15693_BLOCK_NUMBER_SIZE;
      if(!transponder) {
	iso15693->error = RFID_ERROR_TRANSPONDER_NULL;
	RFID_ERROR_STACK(iso15693->verbose);
	return -1;
      }
      *response_size = (block_security_status_size + transponder->bytes_per_block) * (request[number_of_blocks_index] + 1);
    }
    break;
  case ISO15693_COMMAND_WRITE_MULTIPLE_BLOCKS:
  case ISO15693_COMMAND_WRITE_SINGLE_BLOCK:
  case ISO15693_COMMAND_LOCK_BLOCK:
  case ISO15693_COMMAND_SELECT:
  case ISO15693_COMMAND_RESET_TO_READY:
  case ISO15693_COMMAND_WRITE_AFI:
  case ISO15693_COMMAND_LOCK_AFI:
  case ISO15693_COMMAND_WRITE_DSFID:
  case ISO15693_COMMAND_LOCK_DSFID:
    *response_size = 0;
    break;
  case ISO15693_COMMAND_GET_SYSTEM_INFORMATION:
    *response_size = ISO15693_COMMAND_GET_SYSTEM_INFORMATION_RESPONSE_SIZE;
    break;
  case ISO15693_COMMAND_GET_MULTIPLE_BLOCK_SECURITY_STATUS:
    {
      int uid_size = (request[ISO15693_FLAGS_INDEX] & ISO15693_COMMAND_FLAGS_ADDRESS) ? ISO15693_UID_SIZE : 0;
      int first_block_index = ISO15693_FLAGS_SIZE + ISO15693_COMMAND_SIZE + uid_size;
      int number_of_blocks_index = first_block_index + ISO15693_BLOCK_NUMBER_SIZE;
      *response_size = ISO15693_BLOCK_SECURITY_STATUS_SIZE * (request[number_of_blocks_index] + 1);
    }
    break;
  default:
    iso15693->error = RFID_ERROR_COMMAND_INVALID;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
    break;
  }

  if(request[ISO15693_COMMAND_INDEX] != ISO15693_COMMAND_STAY_QUIET)
    *response_size += ISO15693_FLAGS_SIZE + ISO15693_CRC_SIZE;

  return 0;
}

static const char* drv_version(struct rfid_reader* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;
  const char* driver_version = driver ? driver->version_f(driver) : "(no driver)";
  static char* version = 0;

  if(version == 0)
    version = malloc(64);

  
  sprintf(version, "%s/%s", "ISO-15693 " RFID_VERSION, driver_version);

  return version;
}

static void drv_verbose(struct rfid_reader* reader, int verbosity)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;

  iso15693->verbose = verbosity;
  if(driver)
    driver->verbose_f(driver, verbosity);
}

static int driver_load(iso15693_t* iso15693, const char* drivers)
{
  if(!iso15693->reader) {
    char* driver = strchr(drivers, '/');

    if(!driver) {
      iso15693->error = RFID_ERROR_ISO15693_MISSING_DRIVER;
      RFID_ERROR_STACK(iso15693->verbose);
      return -1;
    }
    driver++;
    if(rfid_alloc(driver, &iso15693->reader, iso15693->verbose) < 0) {
      iso15693->error = errno;
      RFID_ERROR_STACK(iso15693->verbose);
      return -1;
    }

    iso15693->reader->verbose_f(iso15693->reader, iso15693->verbose);
    iso15693->reader->io = iso15693->io;
  }

  return 0;
}

static int drv_probe(struct rfid_reader* reader, const char* drivers)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  iso15693->io = reader->io;
  int retval;

  strcpy(reader->drivers, drivers);
  if(driver_load(iso15693, drivers) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  iso15693->reader->iso15693_reader = reader;

  retval = iso15693->reader->probe_f(iso15693->reader, drivers);

  reader->id_string = iso15693->reader->id_string;
  reader->id = iso15693->reader->id;

  return retval;
}

static int drv_init(struct rfid_reader* reader, const char* drivers)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  iso15693->io = reader->io;

  strcpy(reader->drivers, drivers);
  if(driver_load(iso15693, drivers) < 0) {
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  iso15693->reader->iso15693_reader = reader;

  return iso15693->reader->init_f(iso15693->reader, drivers);
}

static void drv_end(struct rfid_reader* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;

  {
    DB* db = iso15693->system_information;
    DBC* dbc = iso15693->system_information_cursor;

    if(db) {
      dbc->c_close(dbc);
      iso15693->system_information_cursor = 0;
      db->close(db, 0);
      iso15693->system_information = 0;
    }
  }

  if(driver) {
    driver->end_f(driver);

    rfid_free(driver);
  }
}

static char* drv_strerror(struct rfid_reader* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;
  char* message = (char*)malloc(512);

  if(iso15693->error) {
    sprintf(message, "%d: %s", iso15693->error, rfid_internal_strerror(iso15693->error));
  } else if(driver->error_f(driver)) {
    char* error_string = driver->strerror_f(driver);
    sprintf(message, "%s", error_string);
    free(error_string);
  } else {
    sprintf(message, "Success");
  }

  return message;
}

static int drv_error(struct rfid_reader* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;

  if(iso15693->error) {
    return iso15693->error;
  } else {
    return driver->error_f(driver);
  }
}

static int drv_read(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_block_t blocks[number_of_blocks];
  int i;
  u_int8_t* pointer;
  int retval;

  if(block_start >= transponder->blocks ||
     block_start + number_of_blocks > transponder->blocks) {
    iso15693->error = RFID_ERROR_NO_BLOCK;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  transponder->data_length = transponder->blocks * transponder->bytes_per_block;
  transponder->data[transponder->data_length] = '\0'; /* for trailing NULL */

  pointer = transponder->data + block_start * transponder->bytes_per_block;
  
  for(i = 0; i < number_of_blocks; i++) {
    blocks[i].data = pointer;
    pointer += transponder->bytes_per_block;
  }

  blocks[0].block_number = block_start;
  blocks[0].security_status = 1;
      
  if((retval = iso15693_read_multiple_blocks(iso15693, transponder, blocks, number_of_blocks)) < 0) {
    if(iso15693->error == RFID_ERROR_ISO15693_CHECKSUM) {
      iso15693->error = RFID_ERROR_TRANSPONDER_NOT_FOUND;
      if(iso15693->verbose == 1) fprintf(stderr, "!");
    }
  } else {
    if(iso15693->verbose == 1) fprintf(stderr, "R");
  }

  return retval;
}

static int drv_write(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  u_int8_t* pointer;
  int i;

  if(block_start + number_of_blocks > transponder->blocks) {
    iso15693->error = RFID_ERROR_NO_BLOCK;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  pointer = transponder->data + block_start * transponder->bytes_per_block;
  
  for(i = 0; i < number_of_blocks; i++) {
    int retval;
    rfid_block_t block;
    block.data = pointer;
    block.block_number = block_start + i;
    block.security_status = 0;
    
    pointer += transponder->bytes_per_block;

    while((retval = iso15693_write_single_block(iso15693, transponder, &block)) < 0 &&
	  reader->error_f(reader) == RFID_ERROR_TRY_AGAIN) {
      if(iso15693->verbose == 1) fprintf(stderr, "!");
    }
    if(retval < 0) {
      RFID_ERROR_STACK(iso15693->verbose);
      return -1;
    }
    if(iso15693->verbose == 1) fprintf(stderr, "W");

  }

  if(iso15693->verbose == 1) fprintf(stderr, "\n");
  
  return 0;
}

/*
 * Implement the collision resolution algorithm implied by
 * the collision detection specifications described in 
 * ISO/IEC 15693-3:2001 page 16.
 *
 * In the following example, it is assumed that 
 * iso15693_inventory will use 16 slots. The ++++
 * designate the bits that will be permutated by the
 * iso15693_inventory, the ---- designate the bits 
 * that each RFID transponder UID must match during 
 * a single inventory request.
 *
 * INVENTORY 00000000000000000000000000000000 (mask = 0, mask_length = 0)
 *                             ++++
 * 00000000000000000000000000000000
 * 00000000000000000000000000000001   FOUND
 * 00000000000000000000000000000010
 * 00000000000000000000000000000011
 * 00000000000000000000000000000100
 * 00000000000000000000000000000101
 * 00000000000000000000000000000110   COLLIDE
 * 00000000000000000000000000000111
 * 00000000000000000000000000001000
 * 00000000000000000000000000001001   COLLIDE
 * 00000000000000000000000000001010
 * 00000000000000000000000000001011
 * 00000000000000000000000000001100
 * 00000000000000000000000000001101
 * 00000000000000000000000000001110
 * 00000000000000000000000000001111
 *
 * Because of 
 * 00000000000000000000000000000110   COLLIDE
 * recurse with 
 * INVENTORY 00000000000000000000000000000110 (mask = 0110, mask_length = 4)
 *                         ++++----
 * 00000000000000000000000000000110
 * 00000000000000000000000000010110
 * 00000000000000000000000000100110
 * 00000000000000000000000000110110
 * 00000000000000000000000001000110   FOUND
 * 00000000000000000000000001010110
 * 00000000000000000000000001100110
 * 00000000000000000000000001110110
 * 00000000000000000000000010000110
 * 00000000000000000000000010010110   COLLIDE
 * 00000000000000000000000010100110
 * 00000000000000000000000010110110
 * 00000000000000000000000011000110
 * 00000000000000000000000011010110
 * 00000000000000000000000011100110
 * 00000000000000000000000011110110
 *
 * Because of 
 * 00000000000000000000000010010110   COLLIDE
 * recurse with 
 * INVENTORY 00000000000000000000000010010110 (mask = 10010110, mask_length = 8)
 *                     ++++--------
 * 00000000000000000000000010010110
 * 00000000000000000000000110010110
 * 00000000000000000000001010010110
 * 00000000000000000000001110010110
 * 00000000000000000000010010010110
 * 00000000000000000000010110010110
 * 00000000000000000000011010010110   FOUND
 * 00000000000000000000011110010110
 * 00000000000000000000100010010110
 * 00000000000000000000100110010110
 * 00000000000000000000101010010110
 * 00000000000000000000101110010110   FOUND
 * 00000000000000000000110010010110
 * 00000000000000000000110110010110
 * 00000000000000000000111010010110
 * 00000000000000000000111110010110
 *
 * No collision at this point, the recursion that started because of
 * the first collision is now finished, we can proceed to the
 * second collision in the same way.
 */
int iso15693_inventory_logic(struct rfid_reader* reader, iso15693_inventory_t inventory, rfid_transponders_t* transponders)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;

  iso15693_inventory_result_t result;

  if(iso15693_inventory(iso15693, &inventory, &result) < 0) {
    transponders->length = 0;
    iso15693->error = iso15693->error;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }
  
  if(result.valid) {
    int i;
    int slot;
    int new_transponders_length = transponders->length + result.nvalid;

    if(new_transponders_length > transponders->size) {
      if(transponders->alloc) {
	int i;
	int old_size = transponders->size;
	transponders->size *= 2;
	transponders->list = (rfid_transponder_t**)realloc(transponders, (transponders->size * sizeof(rfid_transponders_t*)));
	for(i = old_size; i < transponders->size; i++)
	  reader->transponder_alloc_f(reader, &transponders->list[i]);
      } else {
	iso15693->error = RFID_ERROR_TOO_SMALL;
	transponders->length = 0;
	RFID_ERROR_STACK(iso15693->verbose);
	return -1;
      }
    }

    for(slot = 0, i = transponders->length; slot < ISO15693_NB_SLOTS_DEFAULT; slot++) {
      if(result.valid & (1 << slot))  {
	reader->transponder_clear_f(reader, transponders->list[i]);
	iso15693_transponder_uid_set(transponders->list[i], result.data[slot].uid);
	iso15693_transponder_dsfid_set(transponders->list[i], result.data[slot].dsfid);
	i++;
      }
    }

    transponders->length = new_transponders_length;
  }

  /*
   * For each collision, recurse and prepend the slot number
   * to the mask.
   */
  if(result.collide) {
    int slot;
    for(slot = 0; slot < ISO15693_NB_SLOTS_DEFAULT; slot++) {
      u_int8_t collide = (1 << slot);
      if(result.collide & collide)  {
	iso15693_inventory_t inventory_next = inventory;
	int index = inventory.mask_length / 8;
	inventory_next.mask_value[index] |= ((inventory.mask_length % 8) ? (slot << ISO15693_NB_SLOTS_DEFAULT_BITS) : slot) & 0xFF;
	inventory_next.mask_length += ISO15693_NB_SLOTS_DEFAULT_BITS;
	if(iso15693_inventory_logic(reader, inventory_next, transponders) < 0) {
	  RFID_ERROR_STACK(iso15693->verbose);
	  return -1;
	}
      }
    }
  }

  return 0;
}

static int drv_inventory(struct rfid_reader* reader, rfid_transponders_t* transponders)
{
  iso15693_inventory_t inventory;
  int retval;

  if(transponders->list == 0) {
    int i;
    transponders->list = (rfid_transponder_t**)malloc(ISO15693_NB_SLOTS_DEFAULT * sizeof(rfid_transponder_t*));
    transponders->size = ISO15693_NB_SLOTS_DEFAULT;
    transponders->alloc = 1;
    for(i = 0; i < transponders->size; i++)
      reader->transponder_alloc_f(reader, &transponders->list[i]);
  }

  transponders->length = 0;

  memset(&inventory, '\0', sizeof(iso15693_inventory_t));

  if((retval = iso15693_inventory_logic(reader, inventory, transponders)) < 0) {
    iso15693_t* iso15693 = (iso15693_t*)reader->private;
    if(iso15693->error == RFID_ERROR_ISO15693_CHECKSUM)
      iso15693->error = RFID_ERROR_TRY_AGAIN;
  }

  return retval;
}

static char* drv_transponder_describe(struct rfid_reader* reader, const rfid_transponder_t* transponder)
{
  char* str = malloc(512);
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;

  sprintf(str, "uid = 0x%02x%02x%02x%02x%02x%02x%02x%02x\n",
	  iso15693_transponder->uid[7],
	  iso15693_transponder->uid[6],
	  iso15693_transponder->uid[5],
	  iso15693_transponder->uid[4],
	  iso15693_transponder->uid[3],
	  iso15693_transponder->uid[2],
	  iso15693_transponder->uid[1],
	  iso15693_transponder->uid[0]
	  );

  if(iso15693_transponder->info_flags & ISO15693_INFO_FLAGS_DSFID)
    sprintf(str + strlen(str), "dsfid = 0x%02x\n", iso15693_transponder->dsfid);

  if(iso15693_transponder->info_flags & ISO15693_INFO_FLAGS_AFI)
    sprintf(str + strlen(str), "afi = 0x%02x\n", iso15693_transponder->afi);

  if(iso15693_transponder->info_flags & ISO15693_INFO_FLAGS_VICC_MEMORY_SIZE)
    sprintf(str + strlen(str), "vicc_memory = 0x%04x\n", iso15693_transponder->vicc_memory);

  if(iso15693_transponder->info_flags & ISO15693_INFO_FLAGS_IC_REFERENCE)
    sprintf(str + strlen(str), "ic_ref = 0x%02x\n", iso15693_transponder->ic_ref);

  sprintf(str + strlen(str), "blocks = %d\n", transponder->blocks);
  sprintf(str + strlen(str), "bytes_per_block = %d\n", transponder->bytes_per_block);
  
  return str;
}

static int drv_transponder_cmp(struct rfid_reader* reader, rfid_transponder_t* transponder_a, rfid_transponder_t* transponder_b)
{
  return iso15693_transponder_cmp((iso15693_transponder_t*)transponder_a->private,
			       (iso15693_transponder_t*)transponder_b->private);
}

static int drv_transponder_null(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  return iso15693_transponder_null((iso15693_transponder_t*)transponder->private);
}

static int drv_transponder_present(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  int retval;

  if((retval = iso15693_get_system_information(iso15693, transponder)) < 0) {
    if(iso15693->error == RFID_ERROR_ISO15693_CHECKSUM)
      iso15693->error = RFID_ERROR_TRANSPONDER_NOT_FOUND;
  }

  return retval;
}

static int drv_transponder_alloc(struct rfid_reader* reader, rfid_transponder_t** transponderp)
{
  rfid_transponder_t* transponder = (rfid_transponder_t*)malloc(sizeof(rfid_transponder_t));
  memset(transponder, '\0', sizeof(rfid_transponder_t));
  transponder->private = (iso15693_transponder_t*)malloc(sizeof(iso15693_transponder_t));
  memset(transponder->private, '\0', sizeof(iso15693_transponder_t));

  *transponderp = transponder;

  return 0;
}

static int drv_transponder_free(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  free(transponder->private);
  free(transponder);

  return 0;
}

static int drv_transponder_clear(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  iso15693_transponder_t* iso15693 = (iso15693_transponder_t*)transponder->private;
  memset(iso15693, '\0', sizeof(iso15693_transponder_t));
  memset(transponder, '\0', sizeof(rfid_transponder_t));
  transponder->private = iso15693;

  return 0;
}

static int drv_transponder_copy(struct rfid_reader* reader, rfid_transponder_t* to, rfid_transponder_t* from)
{
  iso15693_transponder_t* iso15693_to = (iso15693_transponder_t*)to->private;
  iso15693_transponder_t* iso15693_from = (iso15693_transponder_t*)from->private;

  memcpy(iso15693_to, iso15693_from, sizeof(iso15693_transponder_t));
  memcpy(to, from, sizeof(rfid_transponder_t));
  to->private = iso15693_to;

  return 0;
}

static int drv_transponder_id_get(struct rfid_reader* reader, const rfid_transponder_t* transponder, char** idp)
{
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;

  char* id = malloc(64);

  sprintf(id, "%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x",
	  iso15693_transponder->uid[7],
	  iso15693_transponder->uid[6],
	  iso15693_transponder->uid[5],
	  iso15693_transponder->uid[4],
	  iso15693_transponder->uid[3],
	  iso15693_transponder->uid[2],
	  iso15693_transponder->uid[1],
	  iso15693_transponder->uid[0]);

  *idp = id;

  return 0;
}

static int drv_transponder_id_set(struct rfid_reader* reader, rfid_transponder_t* transponder, char* id)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  unsigned int id0, id1, id2, id3, id4, id5, id6, id7;

  if(sscanf(id, "%x.%x.%x.%x.%x.%x.%x.%x",
	    &id7,
	    &id6,
	    &id5,
	    &id4,
	    &id3,
	    &id2,
	    &id1,
	    &id0) != 8) {
    iso15693->error = RFID_ERROR_PARSE;
    RFID_ERROR_STACK(iso15693->verbose);
    return -1;
  }

  iso15693_transponder->uid[7] = id7 & 0xFF;
  iso15693_transponder->uid[6] = id6 & 0xFF;
  iso15693_transponder->uid[5] = id5 & 0xFF;
  iso15693_transponder->uid[4] = id4 & 0xFF;
  iso15693_transponder->uid[3] = id3 & 0xFF;
  iso15693_transponder->uid[2] = id2 & 0xFF;
  iso15693_transponder->uid[1] = id1 & 0xFF;
  iso15693_transponder->uid[0] = id0 & 0xFF;

  return 0;
}

static int drv_free(rfid_reader_t* reader)
{
  free(reader->private);
  free(reader);
  return 0;
}

static int drv_alloc(rfid_reader_t** readerp)
{
  rfid_reader_t* reader = (rfid_reader_t*)malloc(sizeof(rfid_reader_t));
  iso15693_t* iso15693 = (iso15693_t*)malloc(sizeof(iso15693_t));
  memset(iso15693, '\0', sizeof(iso15693_t));

  reader->private = iso15693;

  reader->alloc_f = drv_alloc;
  reader->free_f = drv_free;

  reader->probe_f = drv_probe;
  reader->init_f = drv_init;
  reader->end_f = drv_end;

  reader->strerror_f = drv_strerror;
  reader->error_f = drv_error;
  reader->version_f = drv_version;
  reader->verbose_f = drv_verbose;

  reader->read_f = drv_read;
  reader->write_f = drv_write;
  reader->inventory_f = drv_inventory;
  reader->iso15693_command_f = 0;
  reader->iso15693_response_f = 0;

  reader->transponder_describe_f = drv_transponder_describe;
  reader->transponder_cmp_f = drv_transponder_cmp;
  reader->transponder_null_f = drv_transponder_null;
  reader->transponder_present_f = drv_transponder_present;
  reader->transponder_alloc_f = drv_transponder_alloc;
  reader->transponder_free_f = drv_transponder_free;
  reader->transponder_clear_f = drv_transponder_clear;
  reader->transponder_copy_f = drv_transponder_copy;
  reader->transponder_id_get_f = drv_transponder_id_get;
  reader->transponder_id_set_f = drv_transponder_id_set;

  *readerp = reader;

  return 0;
}

int iso15693_reader_alloc(rfid_reader_t** reader);

int iso15693_reader_alloc(rfid_reader_t** reader)
{
  return drv_alloc(reader);
}
