/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <rfid_error.h>
#include <s6350.h>
#include <iso15693.h>
#include <rfid.h>
#include <rfid_qsort.h>
#include <errno.h>
#include <ltdl.h>

static int block_signals(rfid_t* rfid)
{
  sigset_t sigset;

  if(sigemptyset(&rfid->sigset) < 0 ||
     sigemptyset(&sigset) < 0 ||
     sigaddset(&sigset, SIGINT) < 0 ||
     sigaddset(&sigset, SIGQUIT) < 0 ||
     sigaddset(&sigset, SIGALRM) < 0 ||
     sigaddset(&sigset, SIGTERM) < 0) {
    rfid->error = errno;
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }

  if(sigprocmask(SIG_BLOCK, &sigset, &rfid->sigset) < 0) {
    rfid->error = errno;
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }

  return 0;
}

static int release_signals(rfid_t* rfid)
{
  if(sigprocmask(SIG_SETMASK, &rfid->sigset, NULL) < 0) {
    rfid->error = errno;
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }

  if(sigemptyset(&rfid->sigset) < 0) {
    rfid->error = errno;
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }

  return 0;
}

typedef struct {
  const char** drivers;
  int drivers_length;
  const char** devices;
  int devices_length;

  rfid_reader_t** readers;
  int readers_length;
} rfid_probe_t;

static int rfid_probe(rfid_t* rfid, rfid_probe_t* probe)
{
  rfid_reader_t** probers;
  int retval = -1;
  int device_index;
  int driver_index;
  int found = 0;

  probers = (rfid_reader_t**)malloc(probe->drivers_length * sizeof(rfid_reader_t*));;
  probe->readers_length = 0;

  memset(probers, '\0', sizeof(rfid_reader_t*) * probe->drivers_length);

  for(driver_index = 0; driver_index < probe->drivers_length; driver_index++) {

    /*
     * We simply ignore drivers that cannot be loaded and we keep
     * going with the others.
     */
    if(rfid_alloc(probe->drivers[driver_index], &probers[driver_index], rfid->verbose) < 0) {
      
      if(errno != RFID_ERROR_TRANSPONDER_NOT_FOUND) {
	RFID_ERROR_STACK(rfid->verbose);
	goto err;
      }
    } else {
      if(rfid_io_alloc(&probers[driver_index]->io) < 0) {
	rfid->error = errno;
	RFID_ERROR_STACK(rfid->verbose);
	goto err;
      }

      probers[driver_index]->verbose_f(probers[driver_index], rfid->verbose);

      found++;
    }
  }

  if(!found) {
    rfid->error = RFID_ERROR_DRIVER_NOT_FOUND;
    RFID_ERROR_STACK(rfid->verbose);
    goto err;
  }

  for(device_index = 0; device_index < probe->devices_length; device_index++) {
    int driver_index;
    for(driver_index = 0; driver_index < probe->drivers_length; driver_index++) {
      if(probers[driver_index]) {
	rfid_reader_t* prober = probers[driver_index];

	rfid_io_init(prober->io, probe->devices[device_index]);

	if(prober->probe_f(prober, probe->drivers[driver_index]) < 0) {
	  if(prober->error_f(prober) == ENOENT) {
	    if(rfid->verbose) {
	      fprintf(stderr, "%s file not found (ignored)\n", probe->devices[device_index]);
	    }
	  } else {
	    if(prober->error_f(prober) != RFID_ERROR_READER_NOT_FOUND) {
	      rfid->error = prober->error_f(prober);
	      if(rfid->verbose) fprintf(stderr, "encountered and error while attempting to probe device %s with driver %s\n", probe->devices[device_index], probe->drivers[driver_index]);
	      RFID_ERROR_STACK(rfid->verbose);
	      goto err;
	    }
	    if(rfid->verbose) {
	      fprintf(stderr, "%s with %s: ", prober->io->device, prober->version_f(prober));
	      fprintf(stderr, "nothing\n");
	    }
	  }
	} else {
	  if(rfid->verbose) {
	    fprintf(stderr, "%s with %s: ", prober->io->device, prober->version_f(prober));
	    fprintf(stderr, "found RFID reader %s\n", prober->id_string);
	  }
	  if(prober->alloc_f(&probe->readers[probe->readers_length]) < 0) {
	    rfid->error = prober->error_f(prober);
	    RFID_ERROR_STACK(rfid->verbose);
	    goto err;
	  }
	  strcpy(probe->readers[probe->readers_length]->drivers, prober->drivers);
	  if(rfid_io_dup(prober->io, &probe->readers[probe->readers_length]->io) < 0) {
	    rfid->error = errno;
	    RFID_ERROR_STACK(rfid->verbose);
	    goto err;
	  }
	  probe->readers_length++;
	}
      }
    }
  }

  if(probe->readers_length > 0)
    retval = 0;
  else
    rfid->error = RFID_ERROR_READER_NOT_FOUND;

 err:

  /*
   * Release the probes.
   */
  for(driver_index = 0; driver_index < probe->drivers_length; driver_index++) {
    if(probers[driver_index]) {
      probers[driver_index]->end_f(probers[driver_index]);
      rfid_io_free(probers[driver_index]->io);
      rfid_free(probers[driver_index]);
    }
  }
  free(probers);

  if(retval < 0) {
    /*
     * Release the readers.
     */
    int i;
    for(i = 0; i < probe->readers_length; i++) {
      rfid_io_free(probe->readers[i]->io);
      probe->readers[i]->free_f(probe->readers[i]);
    }
    probe->readers_length = 0;
  }
  
  return retval;
}

int rfid_init(rfid_t* rfid, const char* driver, const char* device)
{
  int retval = -1;
  rfid_reader_t* reader = 0;
  const char* arg_drivers[1];
  const char* arg_devices[1];
  const char* default_devices[] = {
    "/dev/ttyS0",
    "/dev/ttyS1",
    "/dev/ttyS2",
    "/dev/ttyS3",
    "/dev/ttyUSB0",
    "/dev/ttyUSB1",
    "/dev/ttyUSB2",
    "/dev/ttyUSB3",
  };
  int devices_length = 8;
  const char** devices = default_devices;

  const char* default_drivers[] = {
    "iso15693/s6350",
    "iso15693/m2xxh",
    "s6350",
  };
  int drivers_length = 3;
  const char** drivers = default_drivers;

  rfid_reader_t* readers[8 * 3] = { 0, };

  rfid_probe_t probe;
  int i;

  probe.drivers = drivers;
  probe.drivers_length = drivers_length;
  probe.devices = devices;
  probe.devices_length = devices_length;
  probe.readers = readers;
  probe.readers_length = 0;

  if(driver) {
    arg_drivers[0] = driver;
    probe.drivers = arg_drivers;
    probe.drivers_length = 1;
  }

  if(device) {
    arg_devices[0] = device;
    probe.devices = arg_devices;
    probe.devices_length = 1;
  }
  

  rfid->error = 0;
  if(sigemptyset(&rfid->sigset) < 0)
    return -1;

  if(rfid_probe(rfid, &probe) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    goto err;
  }

  reader = rfid->reader = probe.readers[0];

  reader->verbose_f(reader, rfid->verbose);

  for(i = 1; i < probe.readers_length; i++) {
    probe.readers[i]->end_f(probe.readers[i]);
    rfid_io_free(probe.readers[i]->io);
    rfid_free(probe.readers[i]);
  }

  rfid->options |= RFID_OPTION_FREE_IO;

  if(block_signals(rfid) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    goto err;
  }
  if(reader->init_f(reader, reader->drivers) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    goto err;
  }

  retval = 0;

 err:
  if(release_signals(rfid) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }

  return retval;
}

rfid_transponder_t* rfid_transponder_alloc(rfid_t* rfid)
{
  rfid_reader_t* reader = rfid->reader;
  rfid_transponder_t* transponder = 0;

  reader->transponder_alloc_f(reader, &transponder);

  return transponder;
}

void rfid_transponder_free(rfid_t* rfid, rfid_transponder_t* transponder)
{
  rfid_reader_t* reader = rfid->reader;

  reader->transponder_free_f(reader, transponder);
}

char* rfid_transponder_describe(rfid_t* rfid, rfid_transponder_t* transponder)
{
  rfid_reader_t* reader = rfid->reader;

  return reader->transponder_describe_f(reader, transponder);
}

int rfid_transponder_id_get(rfid_t* rfid, const rfid_transponder_t* transponder, char** idp)
{
  rfid_reader_t* reader = rfid->reader;

  return reader->transponder_id_get_f(reader, transponder, idp);
}

int rfid_transponder_id_set(rfid_t* rfid, rfid_transponder_t* transponder, char* id)
{
  rfid_reader_t* reader = rfid->reader;

  return reader->transponder_id_set_f(reader, transponder, id);
}

void rfid_end(rfid_t* rfid)
{
  rfid->reader->end_f(rfid->reader);
  if(rfid->options & RFID_OPTION_FREE_IO)
    rfid_io_free(rfid->reader->io);
  rfid_free(rfid->reader);
}

char* rfid_strerror(rfid_t* rfid)
{
  rfid_reader_t* reader = rfid->reader;
  char* message = (char*)malloc(512);
  
  if(rfid->error) {
    sprintf(message, "%d: %s", rfid->error, rfid_internal_strerror(rfid->error));
  } else if(!reader) {
    sprintf(message, "unknow error");
  } else {
    char* error_string = reader->strerror_f(reader);
    sprintf(message, "%s", error_string);
    free(error_string);
  }

  return message;
}

int rfid_error(rfid_t* rfid)
{
  rfid_reader_t* reader = rfid->reader;

  if(rfid->error) {
    return rfid->error;
  } else {
    return reader->error_f(reader);
  }
}

static void transponders_free(rfid_t* rfid, rfid_transponders_t* transponders)
{
  if(transponders->size > 0) {
    int i;
    rfid_reader_t* reader = rfid->reader;

    for(i = 0; i < transponders->size; i++)
      reader->transponder_free_f(reader, transponders->list[i]);

    free(transponders->list);
  }
}

static int transponders_cmp(rfid_t* rfid, rfid_transponder_t** a, rfid_transponder_t** b)
{
  rfid_reader_t* reader = rfid->reader;
  return reader->transponder_cmp_f(reader, *a, *b);
}

static void transponders_sort(rfid_t* rfid, rfid_transponder_t** transponders, int transponders_length)
{
  rfid_qsort(transponders, transponders_length, sizeof(rfid_transponder_t*), (rfid_qsort_cmp)transponders_cmp, rfid);
}

typedef struct {
  int events_mask;
  rfid_transponder_callback_t* transponder_callback;
  rfid_inventory_callback_t* inventory_callback;
  void* data;
  
  rfid_transponders_t current;
  rfid_transponders_t cache;
} rfid_events_context_t;

static int transponders_diff(rfid_t* rfid, rfid_events_context_t* context, int* changedp)
{
  rfid_transponders_t* current = &context->current;
  rfid_transponders_t* cache = &context->cache;
  int events_mask = context->events_mask;
  rfid_transponder_callback_t* transponder_callback = context->transponder_callback;
  void* data = context->data;
  rfid_reader_t* reader = rfid->reader;

  int current_index;
  int cache_index;

  /*
   * Compare the cache with the current transponders.
   */
  for(current_index = 0, cache_index = 0;
      current_index < current->length || cache_index < cache->length;) {
    int retval;
    /*
     * -1 => current->list[i] is new on the RFID reader
     * 1 => cache->list[i] was removed from the RFID reader
     * 0 => current->list[i] is still on the RFID reader
     */
    int compare;

    if(current_index < current->length && cache_index < cache->length) {
      compare = reader->transponder_cmp_f(reader, current->list[current_index], cache->list[cache_index]);
    } else if(current_index < current->length) {
      /*
       * If current is longer than cache, what's left in 
       * current are RFID transponders that just entered
       * the range of the RFID reader.
       */
      compare = -1;
    } else {
      /*
       * If cache is longer than current, what's left in cache
       * describe RFID transponders that left the range of the RFID
       * reader.
       */
      compare = 1;
    }

    if(compare < 0) {
      (*changedp)++;
      if(rfid->verbose == 1) fprintf(stderr, "N");
      if(block_signals(rfid) < 0) {
	RFID_ERROR_STACK(rfid->verbose);
	return RFID_CALLBACK_ERROR;
      }
      /*
       * Fill meta information.
       */
      if(reader->transponder_present_f(reader, current->list[current_index]) < 0) {
	/*
	 * The state changed (the RFID transponder probably left the
	 * range of the RFID reader), try again.
	 */
	if(reader->error_f(reader) == RFID_ERROR_TRANSPONDER_NOT_FOUND)
	  rfid->error = RFID_ERROR_TRY_AGAIN;
	release_signals(rfid);
	RFID_ERROR_STACK(rfid->verbose);
	return RFID_CALLBACK_ERROR;
      }
      /*
       * Read data from the memory of the RFID transponder.
       */
      if(reader->read_f(reader, 0, current->list[current_index]->blocks, current->list[current_index]) < 0) {
	/*
	 * The state changed (the RFID transponder probably left the
	 * range of the RFID reader), try again.
	 */
	if(reader->error_f(reader) == RFID_ERROR_TRANSPONDER_NOT_FOUND)
	  rfid->error = RFID_ERROR_TRY_AGAIN;
	release_signals(rfid);
	RFID_ERROR_STACK(rfid->verbose);
	return RFID_CALLBACK_ERROR;
      }

      if(release_signals(rfid) < 0) {
	RFID_ERROR_STACK(rfid->verbose);
	return RFID_CALLBACK_ERROR;
      }

      if(events_mask & RFID_EVENT_NEW) {
	if((retval = (*transponder_callback)(rfid, RFID_EVENT_NEW, current->list[current_index], data)) != 0) {
	  if(retval < 0) RFID_ERROR_STACK(rfid->verbose);
	  return retval;
	}
      }
      current_index++;
    } else if(compare > 0) {
      (*changedp)++;
      if(rfid->verbose == 1) fprintf(stderr, "D");
      if(events_mask & RFID_EVENT_DELETE) {
	if((retval = (*transponder_callback)(rfid, RFID_EVENT_DELETE, cache->list[cache_index], data)) != 0) {
	  if(retval < 0) RFID_ERROR_STACK(rfid->verbose);
	  return retval;
	}
      }
      cache_index++;
    } else if(compare == 0) {
      /*
       * Copy whatever is known about the transponder from the cache instead
       * of requesting data with a new present/read.
       */
      reader->transponder_copy_f(reader, current->list[current_index], cache->list[cache_index]);
      if(rfid->verbose == 1) fprintf(stderr, "A");
      if(events_mask & RFID_EVENT_AGAIN) {
	if((retval = (*transponder_callback)(rfid, RFID_EVENT_AGAIN, current->list[current_index], data)) != 0) {
	  if(retval < 0) RFID_ERROR_STACK(rfid->verbose);
	  return retval;
	}
      }
      current_index++;
      cache_index++;
    }
  }

  return RFID_CALLBACK_OK;
}

static int event_loop(rfid_t* rfid, rfid_events_context_t* context)
{
  int retval = RFID_CALLBACK_OK;
  /* 5 milliseconds */
  struct timespec zzz = {
    0,
    5000000,
  };
  rfid_transponders_t* current = &context->current;
  rfid_transponders_t* cache = &context->cache;
  int events_mask = context->events_mask;
  rfid_inventory_callback_t* inventory_callback = context->inventory_callback;
  rfid_transponder_callback_t* transponder_callback = context->transponder_callback;
  rfid_reader_t* reader = rfid->reader;
  void* data = context->data;

  if((events_mask & (RFID_EVENT_NEW|RFID_EVENT_DELETE|RFID_EVENT_AGAIN)) &&
     !transponder_callback) {
    if(rfid->verbose) fprintf(stderr, "rfid_event_loop: events_mask set to one of RFID_EVENT_NEW, RFID_EVENT_DELETE or RFID_EVENT_AGAIN but transponder_callback argument is null\n");
    rfid->error = EINVAL;
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }

  if((events_mask & RFID_EVENT_INVENTORY) && !inventory_callback) {
    if(rfid->verbose) fprintf(stderr, "rfid_event_loop: events_mask set to one of RFID_EVENT_INVENTORY but inventory_callback argument is null\n");
    rfid->error = EINVAL;
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }
  
  rfid->error = 0;

  while(retval == RFID_CALLBACK_OK) {
    int changed = 0;

    if(block_signals(rfid) < 0) {
      RFID_ERROR_STACK(rfid->verbose);
      return -1;
    }

    if(reader->inventory_f(reader, current) < 0) {
      int inventory_error = reader->error_f(reader);
      if(release_signals(rfid) < 0) {
	RFID_ERROR_STACK(rfid->verbose);
	return -1;
      }
      /*
       * Restart if a recoverable error occurs.
       */
      if(inventory_error == RFID_ERROR_TRY_AGAIN) {
	if(rfid->verbose == 1) fprintf(stderr, "!");
	continue;
      }

      RFID_ERROR_STACK(rfid->verbose);
      return -1;
    }

    if(release_signals(rfid) < 0) {
      RFID_ERROR_STACK(rfid->verbose);
      return -1;
    }

    transponders_sort(rfid, current->list, current->length);

    if((retval = transponders_diff(rfid, context, &changed)) < 0) {
      /*
       * Recoverable error (typically the state changed in the middle of
       * the analysis of the current state).
       */
      if(rfid_error(rfid) == RFID_ERROR_TRY_AGAIN) {
	if(rfid->verbose == 1) fprintf(stderr, "!");
	rfid->error = 0;
	retval = RFID_CALLBACK_OK;
	continue;
      }
      /*
       * Everything else is a fatal error, either from the rfid
       * library or because a callback returned on error.
       */
      RFID_ERROR_STACK(rfid->verbose);
      return -1;
    }

    if((retval == RFID_CALLBACK_OK) && changed && (events_mask & RFID_EVENT_INVENTORY)) {
      if((retval = (*inventory_callback)(rfid, current->list, current->length, data)) < 0) {
	RFID_ERROR_STACK(rfid->verbose);
	return -1;
      }
    }
    
    if(rfid->verbose == 1) fprintf(stderr, "/");

    /*
     * Swap cache & current transponders. 
     */
    {
      rfid_transponders_t tmp;

      tmp = *cache;
      *cache = *current;
      *current = tmp;
    }

    /*
     * Short pause to avoid using all the CPU. More a safety measure
     * than really usefull since the speed of the serial line is more
     * than likely to introduce enough delay anyway.
     */
    nanosleep(&zzz, 0);
  }

  return 0;
}

int rfid_event_loop(rfid_t* rfid, int events_mask, rfid_transponder_callback_t* transponder_callback, rfid_inventory_callback_t* inventory_callback, void* data)
{
  int retval = -1;
  rfid_events_context_t context;

  memset(&context, '\0', sizeof(rfid_events_context_t));

  context.events_mask = events_mask;
  context.transponder_callback = transponder_callback;
  context.inventory_callback = inventory_callback;
  context.data = data;

  retval = event_loop(rfid, &context);

  transponders_free(rfid, &context.current);
  transponders_free(rfid, &context.cache);

  if(release_signals(rfid) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }

  return retval;
}

int rfid_write(rfid_t* rfid, rfid_transponder_t* transponder)
{
  rfid_reader_t* reader = rfid->reader;
  rfid_transponder_t* current_transponder = 0;
  int retval = -1;

  /* 5 milliseconds */
  struct timespec zzz = {
    0,
    5000000,
  };

  if(reader->transponder_alloc_f(reader, &current_transponder) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    goto err;
  }

  rfid->error = 0;

  while(1) {
    if(block_signals(rfid) < 0) {
      RFID_ERROR_STACK(rfid->verbose);
      goto err;
    }

    if(reader->transponder_present_f(reader, current_transponder) < 0) {

      if(reader->error_f(reader) == RFID_ERROR_TRANSPONDER_NOT_FOUND) {
	/*
	 * No RFID transponder, try again.
	 */
	if(rfid->verbose == 1) fprintf(stderr, ".");

      } else if(reader->error_f(reader) == RFID_ERROR_TRY_AGAIN) {
	/*
	 * Recoverable error, try again.
	 */
	if(rfid->verbose == 1) fprintf(stderr, "!");

      } else {
	/*
	 * Unrecoverable error, abort.
	 */
	RFID_ERROR_STACK(rfid->verbose);
	goto err;
      }
    } else {
      /*
       * The RFID transponder is available, proceed.
       */
      break;
    }

    if(release_signals(rfid) < 0) {
      RFID_ERROR_STACK(rfid->verbose);
      goto err;
    }

    nanosleep(&zzz, 0);
  }

  if(!reader->transponder_null_f(reader, transponder) &&
     reader->transponder_cmp_f(reader, current_transponder, transponder)) {
    rfid->error = RFID_ERROR_TRANSPONDER_NOT_FOUND;
    RFID_ERROR_STACK(rfid->verbose);
    goto err;
  }

  memcpy(current_transponder->data, transponder->data, transponder->data_length);
  current_transponder->data_length = transponder->data_length;

  if(reader->write_f(reader, 0, current_transponder->blocks, current_transponder) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    goto err;
  }

  if(reader->transponder_copy_f(reader, transponder, current_transponder) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    goto err;
  }
  
  retval = 0;

 err:

  if(current_transponder) {
    if(reader->transponder_free_f(reader, current_transponder) < 0) {
      RFID_ERROR_STACK(rfid->verbose);
      return -1;
    }
  }

  if(release_signals(rfid) < 0) {
    RFID_ERROR_STACK(rfid->verbose);
    return -1;
  }

  return retval;
}

const char* rfid_version(void)
{
  return "RFID "RFID_VERSION;
}
