/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include <rfid_error.h>
#include <rfid_io.h>

#ifdef POSIX_TERMIOS

#ifndef HAVE_CFMAKERAW
#define cfmakeraw(ptr) (ptr)->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR\
					 |IGNCR|ICRNL|IXON);\
                       (ptr)->c_oflag &= ~OPOST;\
                       (ptr)->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);\
                       (ptr)->c_cflag &= ~(CSIZE|PARENB);\
                       (ptr)->c_cflag |= CS8
#endif /* HAVE_CFMAKERAW */

#ifndef HAVE_CFSETSPEED
#if defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED)
#define cfsetspeed(t,speed) \
  (cfsetispeed(t,speed) || cfsetospeed(t,speed))
#else /* defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED) */
static int cfsetspeed(struct termios *t, int speed)
{
#ifdef HAVE_TERMIOS_CSPEED
	t->c_ispeed = speed;
	t->c_ospeed = speed;
#else /* HAVE_TERMIOS_CSPEED */
	t->c_cflag |= speed;
#endif /* HAVE_TERMIOS_CSPEED */
	return 0;
}
#endif /* defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED) */
#endif /* HAVE_CFSETSPEED */

#endif /* HAVE_CFMAKERAW */


int rfid_io_alloc(rfid_io_t** iop)
{
  *iop = (rfid_io_t*)malloc(sizeof(rfid_io_t));
  memset(*iop, '\0', sizeof(rfid_io_t));
  return 0;
}

int rfid_io_free(rfid_io_t* io)
{
  free(io);
  return 0;
}

int rfid_io_dup(rfid_io_t* from, rfid_io_t** top)
{
  if(rfid_io_alloc(top) < 0)
    return -1;
  memcpy(*top, from, sizeof(rfid_io_t));
  if(from->fd >= 0)
    (*top)->fd = dup(from->fd);

  return 0;
}

int rfid_io_init(rfid_io_t* io, const char* pathname)
{
  strcpy(io->device, pathname);
  io->speed = io->nbits = io->parity = io->stopbits = 0;
  io->lock = 0;
  io->fd = -1;
  io->timeout = 1000;
  return 0;
}

int rfid_io_set(rfid_io_t* io, int speed, int nbits, int parity, int stopbits, int lock)
{
  if(io->speed >= 0) io->speed = speed;
  if(io->nbits >= 0) io->nbits = nbits;
  if(io->parity >= 0) io->parity = parity;
  if(io->stopbits >= 0) io->stopbits = stopbits;
  io->lock = lock;
  return 0;
}

int rfid_io_end(rfid_io_t* io)
{
  if(io->fd >= 0) {
    close(io->fd);
    io->fd = -1;
  }

  return 0;
}

int rfid_io_open(rfid_io_t* io)
{
  if((io->fd = open(io->device, O_RDWR)) < 0) {
    /*
     * All errors that can be fixed by pluging in a RFID reader
     * should be mapped to RFID_ERROR_READER_NOT_FOUND.
     */
    if(errno == ENODEV)
      errno = RFID_ERROR_READER_NOT_FOUND;
    return -1;
  }

  if(io->lock) {
    if(lockf(io->fd, F_TLOCK, 0) < 0) {
      if(errno == EAGAIN) errno = RFID_ERROR_READER_BUSY;
      close(io->fd);
      return -1;
    }
  }

  return rfid_io_configure(io);
}

int rfid_io_configure(rfid_io_t* io)
{
  struct 	termios tcn;
  int i;

  if (!isatty(io->fd)) {
    close(io->fd);
    errno = RFID_ERROR_READER_NOT_FOUND;
    io->fd = -1;
    return -1;
  }

  tcgetattr(io->fd, &tcn);

  tcn.c_oflag	       &= ~OPOST;
  tcn.c_iflag	       &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP
			    |INLCR|IGNCR|ICRNL|IXON);
  tcn.c_cflag	       &= ~(CSIZE|CSTOPB|PARENB);
  tcn.c_lflag	       &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);

  tcn.c_iflag 	= IGNBRK | io->parity;
  tcn.c_cflag 	= CREAD | CLOCAL | io->nbits | io->stopbits | io->parity;

  (void) cfsetspeed(&tcn, io->speed);

  for (i = 0; i < 16; i++)
    tcn.c_cc[i] = 0;

  tcn.c_cc[VMIN] 	= 1;
  tcn.c_cc[VTIME] = 0;

  tcsetattr(io->fd, TCSANOW, &tcn);

  return 0;
}

/*
 * Read up to buffer_size bytes from input and times out after 1
 * second. This would not be necessary if the RFID reader reliably
 * sent the expected number of bytes. Unfortunately, when there is no
 * flow control on the serial line (and this is the case for
 * some readers), some messages may be incomplete.
 */
static int timeout_read(rfid_io_t* io, u_int8_t* buffer, int buffer_size)
{
  fd_set set;
  int fd = io->fd;
  int seconds = io->timeout / 1000;
  int microseconds = (io->timeout - seconds) * 1000;
  struct timeval timeout;
  int do_read;

  timeout.tv_sec = seconds;
  timeout.tv_usec = microseconds;

  FD_ZERO(&set);
  FD_SET(fd, &set);
  if((do_read = select(fd + 1, &set, NULL, NULL, &timeout)) < 0) {
    return -1;
  }

  if(do_read) {
    int available;
    int retval;

    if(ioctl(fd, FIONREAD, &available) < 0) {
      return -1;
    }

    if(available > buffer_size)
      available = buffer_size;

    if((retval = read(fd, buffer, available)) < 0) {
      return -1;
    } else {
      return retval;
    }
  } else {
    errno = RFID_ERROR_TRY_AGAIN;
    return -1;
  }
}

/*
 * Fill buffer with exactly buffer_size bytes.
 */
int rfid_io_read(rfid_io_t* io, u_int8_t* buffer, int buffer_size)
{
  int offset = 0;
  int read_bytes = 0;
  while(offset < buffer_size) {
    if((read_bytes = timeout_read(io, buffer + offset, buffer_size - offset)) < 0) {
      return -1;
    }
    offset += read_bytes;
  }

  return buffer_size;
}
