/*
 * Copyright (C) 2003 _INRIA_
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <memory.h>
#include <string.h>

#include <rfid.h>
#include <cgic.h>

#define DEVICE_SIZE 256

/*
 * Context of the rfid_cgi.
 */
typedef struct {
  rfid_t* rfid;
  int verbose;
  int max_count;
} rfid_cgi_t;

/*
 */
static void report_error(rfid_t* rfid, const char* message)
{
  char* str = rfid_strerror(rfid);
  fprintf(stderr, "%s: %s\n", message, str);
  fprintf(cgiOut, "ERROR %s: %s\n", message, str);
  free(str);
}

/*
 */
static int callback(rfid_t* rfid, const rfid_transponder_t* transponder, void* data)
{
  rfid_cgi_t* rfid_cgi = (rfid_cgi_t*)data;
  const s6350_transponder_t* info = &transponder->info;

  if(rfid_cgi->max_count > 0)
    rfid_cgi->max_count--;

  if(rfid_cgi->verbose) {
    fprintf(cgiOut, "\
# ========================================\n\
# transponder_id = 0x%02x%02x%02x%02x\n\
# manufacturer = 0x%02x\n\
# version_number = 0x%04x\n\
# blocks = %04d\n\
# bytes_per_block = %d\n\
",
	    info->transponder_id[3],
	    info->transponder_id[2],
	    info->transponder_id[1],
	    info->transponder_id[0],
	    info->manufacturer,
	    info->version_number,
	    info->blocks,
	    info->bytes_per_block
	    );

    {
      int i;
      fprintf(cgiOut, "# ");
      for(i = 0; i < transponder->data_length; i++)
	fprintf(cgiOut, "%c(0x%02x)/", transponder->data[i], transponder->data[i]);
      fprintf(cgiOut, "\n");
    }
  }

  fprintf(cgiOut, "%02x%02x%02x%02x.%02x.%04x\t%04d\t%s\n",
	  info->transponder_id[3],
	  info->transponder_id[2],
	  info->transponder_id[1],
	  info->transponder_id[0],
	  info->manufacturer,
	  info->version_number,
	  info->blocks * info->bytes_per_block,
	  transponder->data);

  fflush(cgiOut);

  if(rfid_cgi->max_count >= 0)
    return rfid_cgi->max_count == 0 ? RFID_CALLBACK_END : RFID_CALLBACK_OK;
  else
    return RFID_CALLBACK_OK;
}

/*
 * Implements read=1, infinite loop reading commands from RFID transponders.
 */
static int read_daemon(rfid_cgi_t* rfid_cgi)
{
  if(rfid_read(rfid_cgi->rfid, callback, rfid_cgi) < 0) {
    report_error(rfid_cgi->rfid, "rfid_read");
    return -1;
  }

  return 0;
}

/*
 * Wait for a RFID transponder to be presented to the RFID reader
 * and store the arguments (white space separated) into it.
 */
static int write_command(rfid_cgi_t* rfid_cgi, char* data)
{
  rfid_t* rfid = rfid_cgi->rfid;
  rfid_transponder_t transponder;

  memset(transponder.data, '\0', RFID_TRANSPONDER_DATA_SIZE);
  memcpy(transponder.data, data, RFID_TRANSPONDER_DATA_SIZE);
  transponder.data_length = RFID_TRANSPONDER_DATA_SIZE;

  if(rfid_cgi->verbose) {
    fprintf(cgiOut, "# Attempting to write '%s' to a transponder ... ", transponder.data);
    fflush(cgiOut);
  }
  
  if(rfid_write(rfid, &transponder) < 0) {
    report_error(rfid, "rfid_write");
    return -1;
  }

  if(rfid_cgi->verbose) {
    fprintf(cgiOut, "done\n");
    fflush(cgiOut);
  }

  return read_daemon(rfid_cgi);
}

/*
 * Terse manual, see rfid_rfid_cgi(3) for more.
 */
static void usage(const char* message)
{
  char* usage = "\
# usage:\n\
#\n\
# rfid-cgi?[help=<n>&][device=<special>&][verbose=1&][read=<n>]\n\
# rfid-cgi?[help=<n>&][device=<special>&][verbose=1&][write=<data>]\n\
#    read=<n>           Wait for a transponder and display its content.\n\
#                       Loop <n> times.\n\
#    write=<data>       Wait for a transponder and store the data\n\
#                       given in argument in its memory.\n\
#    device=<special>   Dialog with RFID reader connected to special\n\
#                       file <special>. Use %2F instead of /\n\
#                       (default %2Fdev%2FttyS0).\n\
#    verbose=1          Set verbosity level to 1 (default 0).\n\
#    help=1             Print this message.\n\
#\n\
#    When read=<n> is set, each tag is reported using the following format,\n\
#    one per line. Newlines are forbiden and will break the format.\n\
#    \n\
#    IIIIIIII.MM.VVVV<tab>BBBB<tab>CCCCCCC...\n\
#    \n\
#    Where:\n\
#    \n\
#    IIIIIIII  is a 8 digit transponder identification in hexadecimal.\n\
#    MM        is a 2 digit manufacturer identification in hexadecimal.\n\
#    VVVV      is a 4 digit version number in hexadecimal.\n\
#    BBBB      is a is the size of the memory of the transponder as\n\
#              4 digit decimal number\n\
#    CCCCC.... is the content of the memory of the transponder in characters\n\
#              until a newline is found.\n\
#    <tab>     is a tabulation used as a separator.\n\
#\n\
";
  fprintf(stderr, "usage: %s\n", message);
  fprintf(cgiOut, "# %s\n%s", message, usage);
}

/*
 */
int cgiMain() {
  int retval = 0;
  int option_read = 0;
  int option_write = 0;
  char data[RFID_TRANSPONDER_DATA_SIZE + 1];
  rfid_t rfid;
  rfid_cgi_t rfid_cgi;

  memset(&rfid, '\0', sizeof(rfid_t));
  rfid.device = (char*)malloc(DEVICE_SIZE+1);

  rfid_cgi.rfid = &rfid;

  cgiHeaderContentType("text/plain");
  fflush(cgiOut);

  cgiFormInteger("verbose", &rfid_cgi.verbose, 0);

  {
    int dummy;
    if(cgiFormInteger("help", &dummy, 0) == cgiFormSuccess) {
      usage("Help");
      return 0;
    }
  }
  
  if(cgiFormString("device", rfid.device, DEVICE_SIZE) != cgiFormSuccess)
      strcpy(rfid.device, "/dev/ttyS0");

  if(cgiFormInteger("read", &rfid_cgi.max_count, 1) == cgiFormSuccess) {
    option_read++;
  }

  if(cgiFormString("write", data, RFID_TRANSPONDER_DATA_SIZE) == cgiFormSuccess) {
    option_write++;
  }

  if(option_read && option_write) {
    usage("Options read=<n> and write=<data> are mutually exclusive");
    return -1;
  }
  
  if(rfid_init(&rfid) < 0) {
    report_error(&rfid, "rfid_init");
    return -1;
  }
  
  if(option_read)
    retval = read_daemon(&rfid_cgi);
  else if(option_write) {
    retval = write_command(&rfid_cgi, data);
  } else
    usage("You must specify either write=<data> or read=1");

  rfid_end(&rfid);

  return retval;
}

