/*
 * Copyright (C) 2003 _INRIA_
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <memory.h>
#include <errno.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/ioctl.h>

#include <s6350.h>

#define S6350_RF_CARRIER_ON 0xFF
#define S6350_RF_CARRIER_OFF 0x00

#define S6350_REQUEST_HEADER_SIZE 7
#define S6350_REQUEST_TRAILER_SIZE 2
#define S6350_REQUEST_OVERHEAD S6350_REQUEST_HEADER_SIZE + S6350_REQUEST_TRAILER_SIZE

#define S6350_HEADER_SOF 0
#define S6350_HEADER_LENGTH_LSB 1
#define S6350_HEADER_LENGTH_MSB 2
#define S6350_HEADER_NODE_ADDRESS_LSB 3
#define S6350_HEADER_NODE_ADDRESS_MSB 4
#define S6350_HEADER_COMMAND_FLAGS 5
#define S6350_HEADER_COMMAND 6

#define S6350_ERROR_DATA 7

#define S6350_VALUE_SOF 0x01

/*
 * Combine/split an int containing the command and the command flags
 */
#define S6350_COMMAND_COMBINE(command,flags) ((command)|((flags) << 8))
#define S6350_COMBINED_FLAGS(combined) ((combined >> 8) & 0xFF)
#define S6350_COMBINED_COMMAND(combined) (combined & 0xFF)

#define S6350_COMMAND_FLAGS_ADDRESS		0x10
#define S6350_COMMAND_FLAGS_ERROR		0x10

#define S6350_COMMAND_READ_BLOCK		0x02
#define S6350_COMMAND_WRITE_BLOCK		0x03
#define S6350_COMMAND_LOCK_BLOCK		0x04
#define S6350_COMMAND_READ_TRANSPONDER_DETAILS	0x05
#define S6350_COMMAND_SPECIAL_READ_BLOCK	0x0F
#define S6350_COMMAND_INITIATE_FLASH_LOADER	0xD0
#define S6350_COMMAND_SEND_DATA_TO_FLASH	0xD8
#define S6350_COMMAND_READER_VERSION		0xF0
#define S6350_COMMAND_READER_INPUTS		0xF1
#define S6350_COMMAND_WRITE_READER_OUTPUT	0xF2
#define S6350_COMMAND_RF_CARRIER		0xF4
#define S6350_COMMAND_BAUD_RATE_CONFIGURATION	0xFF

#define clear_error(o) ((o)->error = (o)->s6350_error = 0)
/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.1.5 BCC 
 */
static u_int16_t bcc_compute(u_int8_t* buffer, int buffer_size)
{
  int i;
  u_int8_t lrc = 0;
  for(i = 0; i < buffer_size; i++) {
    lrc ^= buffer[i];
  }
  return lrc | ((~lrc) << 8);
}

static void bcc_set(u_int8_t* buffer, int buffer_size)
{
  u_int16_t bcc = bcc_compute(buffer, buffer_size - 2);
  buffer[buffer_size - 1] = (bcc >> 8) & 0xFF;
  buffer[buffer_size - 2] = bcc & 0xFF;
}

static int bcc_check(u_int8_t* buffer, int buffer_size)
{
  u_int16_t bcc = bcc_compute(buffer, buffer_size - 2);
  return buffer[buffer_size - 1] == ((bcc >> 8) & 0xFF) &&
    buffer[buffer_size - 2] == (bcc & 0xFF);
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.1.1 Request Packet Format (Host to Reader)
 */
static int write_request(s6350_t* s6350, int command, u_int8_t* buffer, int buffer_size)
{
  char* request = (char*)malloc(buffer_size + S6350_REQUEST_OVERHEAD);
  short request_length = buffer_size + S6350_REQUEST_OVERHEAD;
  int wrote;

  request[S6350_HEADER_SOF] = 0x01;
  request[S6350_HEADER_LENGTH_LSB] = request_length & 0xFF;
  request[S6350_HEADER_LENGTH_MSB] = (request_length >> 8) & 0xFF;
  request[S6350_HEADER_NODE_ADDRESS_LSB] = 0;
  request[S6350_HEADER_NODE_ADDRESS_MSB] = 0;
  request[S6350_HEADER_COMMAND_FLAGS] = S6350_COMBINED_FLAGS(command);
  request[S6350_HEADER_COMMAND] = S6350_COMBINED_COMMAND(command);

  memcpy(request + S6350_REQUEST_HEADER_SIZE, buffer, buffer_size);

  bcc_set(request, request_length);

  if((wrote = write(s6350->fd, request, request_length)) != request_length)
    s6350->error = errno;
  else
    s6350->error = 0;

  free(request);

  return s6350->error < 0 ? -1 : 0;
}

/*
 * Read up to buffer_size bytes from input and times out after 1
 * second. This would not be necessary if the RFID reader reliably
 * sent the expected number of bytes. Unfortunately, when there is no
 * flow control on the serial line, some messages may be incomplete.
 */
static int timeout_read(s6350_t* s6350, char* buffer, int buffer_size)
{
  fd_set set;
  int fd = s6350->fd;
  /* 1 second */
  struct timeval timeout = {
    1,
    0
  };
  int do_read;

  FD_ZERO(&set);
  FD_SET(fd, &set);
  if((do_read = select(fd + 1, &set, NULL, NULL, &timeout)) < 0) {
    s6350->error = errno;
    return -1;
  }

  if(do_read) {
    int available;
    int retval;

    if(ioctl(fd, FIONREAD, &available) < 0) {
      s6350->error = errno;
      return -1;
    }

    if(available > buffer_size)
      available = buffer_size;

    if((retval = read(fd, buffer, available)) < 0) {
      s6350->error = errno;
      return -1;
    } else {
      return retval;
    }
  } else {
    s6350->error = EAGAIN;
    return -1;
  }
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.1.2 Response Packet Format (Reader to Host)
 * 3.1.4 Command Flags Response
 */
static int read_answer(s6350_t* s6350, u_int8_t** bufferp, int* buffer_sizep)
{
  char answer_header[S6350_REQUEST_HEADER_SIZE];
  char* buffer;
  int read_bytes;
  u_int16_t length;
  int to_read;
  int offset = 0;

  *bufferp = 0;
  *buffer_sizep = 0;

  /*
   * Wait for the Start Of Frame (SOF) to appear in the input stream.
   * Most of the time it's the first byte but there is no guarantee.
   */
  while((read_bytes = timeout_read(s6350, answer_header, 1)) >= 0 &&
	answer_header[0] != S6350_VALUE_SOF)
    ;
  if(read_bytes < 0) {
    return -1;
  }
  offset++;
  
  /*
   * Read the rest of the header, the Start Of Frame (SOF) is already
   * in the buffer.
   */
  while(offset < S6350_REQUEST_HEADER_SIZE) {
    if((read_bytes = timeout_read(s6350, answer_header + offset, S6350_REQUEST_HEADER_SIZE - offset)) < 0) {
      return -1;
    }
    offset += read_bytes;
  }

  /*
   * Extract the total length of the frame
   */
  length = answer_header[S6350_HEADER_LENGTH_LSB] |
    (answer_header[S6350_HEADER_LENGTH_MSB] << 8);

  /*
   * Sanity checks to prevent memory corruption if message
   * is corrupted.
   */
  if(length < S6350_REQUEST_OVERHEAD) {
    s6350->error = EINVAL;
    return -1;
  }
  /*
   * Sanity checks to detect corrupted headers.
   */
  if(answer_header[S6350_HEADER_NODE_ADDRESS_LSB] != 0x00 ||
     answer_header[S6350_HEADER_NODE_ADDRESS_MSB] != 0x00) {
    s6350->error = EAGAIN;
    return -1;
  }

  to_read = length - S6350_REQUEST_HEADER_SIZE;

  buffer = (char*)malloc(length);
  memcpy(buffer, answer_header, S6350_REQUEST_HEADER_SIZE);

  offset = 0;
  while(offset < to_read) {
    if((read_bytes = timeout_read(s6350, buffer + S6350_REQUEST_HEADER_SIZE + offset, to_read - offset)) < 0) {
      free(buffer);
      return -1;
    }
    offset += read_bytes;
  }

  /*
   * Check consistency of the frame content.
   */
  if(!bcc_check(buffer, length)) {
    free(buffer);
    s6350->error = EFAULT;
    return -1;
  }

  /*
   * If the transponder returns on error, abort.
   * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
   * 3.1.4 Command Flags Response
   */
  if(buffer[S6350_HEADER_COMMAND_FLAGS] == S6350_COMMAND_FLAGS_ERROR) {
    s6350->s6350_error = buffer[S6350_ERROR_DATA];
    s6350->error = EIO;
    free(buffer);
    return -1;
  }

  memmove(buffer, buffer + S6350_REQUEST_HEADER_SIZE, to_read - S6350_REQUEST_TRAILER_SIZE);
  
  *bufferp = buffer;
  *buffer_sizep = to_read - S6350_REQUEST_TRAILER_SIZE;

  s6350->error = 0;

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * Appendix B Error Codes
 */
static char error_invalid[] = "Invalid s6350 error number";
static char* error_message[] = {
  /* 0x00 */ error_invalid, 
  /* 0x01 */ "Transponder not found",
  /* 0x02 */ "Command not supported",
  /* 0x03 */ "Packet BCC invalid",
  /* 0x04 */ "Packet flags invalid for command",
  /* 0x05 */ "General write failure",
  /* 0x06 */ "Write failure due to locked block",
  /* 0x07 */ "Transponder does not support function",
  /* 0x08 */ error_invalid, 
  /* 0x09 */ error_invalid, 
  /* 0x0a */ error_invalid, 
  /* 0x0b */ error_invalid, 
  /* 0x0c */ error_invalid, 
  /* 0x0d */ error_invalid, 
  /* 0x0e */ error_invalid, 
  /* 0x0f */ "Undefined error" 
};

const char* s6350_error(s6350_t* s6350)
{
  if(s6350->s6350_error > 0x00 && s6350->s6350_error <= 0x0F) {
    return error_message[s6350->s6350_error];
  } else {
    return error_invalid;
  }
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.1 Tag-it HF Command Definitions
 *    page 20: Read Transponder Details (05hex)
 */

#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_TRANSPONDER_ID 0
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_MANUFACTURER 4
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_VERSION_NUMBER0 5
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_VERSION_NUMBER1 6
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_BLOCKS 7
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_BYTES_PER_BLOCK 8

int s6350_transponder_details(s6350_t* s6350, s6350_transponder_t* transponder)
{
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  memset(transponder, '\0', sizeof(s6350_transponder_t));

  if(write_request(s6350, S6350_COMMAND_READ_TRANSPONDER_DETAILS, "", 0) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  memcpy(transponder->transponder_id,
	 buffer + S6350_READ_TRANSPONDER_DETAILS_ANSWER_TRANSPONDER_ID,
	 S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE);
  transponder->manufacturer = buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_MANUFACTURER];
  transponder->version_number =
    buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_VERSION_NUMBER0] |
    (buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_VERSION_NUMBER1] << 8);
  transponder->blocks = buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_BLOCKS];
  transponder->bytes_per_block = buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_BYTES_PER_BLOCK];
  
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.1 Tag-it HF Command Definitions
 *    page 21: Special Read Block Command (0Fhex)
 */

#define S6350_SPECIAL_READ_BLOCK_ANSWER_TRANSPONDER_ID 0
#define S6350_SPECIAL_READ_BLOCK_ANSWER_DATA S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE

int s6350_special_read_block(s6350_t* s6350, const s6350_transponder_t* transponder, u_int8_t block_numbers, s6350_block_t* blocks)
{
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  if(write_request(s6350, S6350_COMMAND_SPECIAL_READ_BLOCK, &block_numbers, sizeof(u_int8_t)) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  if(buffer_size < S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE) {
    s6350->error = EBADMSG;
    free(buffer);
    return -1;
  }

  /*
   * Expect the returned transponder_id to be equal to the transponder_id of
   * the transponder record. It makes sure the transponder record obtained
   * from read_transponder_details matches the transponder answering the
   * special_read_block request. This is the only way to figure out the size of
   * the data block. 
   */
  if(memcmp(transponder->transponder_id,
	    buffer + S6350_SPECIAL_READ_BLOCK_ANSWER_TRANSPONDER_ID,
	    S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE)) {
    s6350->error = EAGAIN;
    free(buffer);
    return -1;
  }

  {
    int offset;
    int i;
    int data_length = transponder->bytes_per_block + sizeof(u_int8_t) * 2;
    for(i = 0, offset = S6350_SPECIAL_READ_BLOCK_ANSWER_DATA;
	offset < buffer_size;
	i++, offset += data_length) {
      memcpy(blocks[i].data, buffer + offset, transponder->bytes_per_block);
      blocks[i].lock_status = buffer[offset + transponder->bytes_per_block];
      blocks[i].block_number = buffer[offset + transponder->bytes_per_block + sizeof(u_int8_t)];
    }
  }
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.1 Tag-it HF Command Definitions
 *    page 19: Read Block Command (02hex)
 */

#define S6350_READ_BLOCK_REQUEST_ADDRESS 0
#define S6350_READ_BLOCK_REQUEST_BLOCK_NUMBER 4

int s6350_read_block(s6350_t* s6350, const s6350_transponder_t* transponder, s6350_block_t* block)
{
  u_int8_t request[S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE + sizeof(u_int8_t)];
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  if(block->block_number > transponder->blocks) {
    s6350->error = E2BIG;
    return -1;
  }
  
  memcpy(request + S6350_READ_BLOCK_REQUEST_ADDRESS,
	 transponder->transponder_id,
	 S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE);
  request[S6350_READ_BLOCK_REQUEST_BLOCK_NUMBER] = block->block_number;

  if(write_request(s6350, S6350_COMMAND_COMBINE(S6350_COMMAND_READ_BLOCK, S6350_COMMAND_FLAGS_ADDRESS), request, sizeof(request)) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  memcpy(block->data, buffer, transponder->bytes_per_block);
  block->lock_status = buffer[transponder->bytes_per_block];
  block->block_number = buffer[transponder->bytes_per_block + sizeof(u_int8_t)];
  
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.1 Tag-it HF Command Definitions
 *    page 19: Write Block Command (03hex)
 */

#define S6350_WRITE_BLOCK_REQUEST_ADDRESS 0
#define S6350_WRITE_BLOCK_REQUEST_BLOCK_NUMBER 4
#define S6350_WRITE_BLOCK_REQUEST_DATA 5

int s6350_write_block(s6350_t* s6350, const s6350_transponder_t* transponder, const s6350_block_t* block)
{
  u_int8_t* request;
  int request_size;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  request_size = S6350_WRITE_BLOCK_REQUEST_DATA + transponder->bytes_per_block;
  request = (u_int8_t*)malloc(request_size);
  memcpy(request + S6350_WRITE_BLOCK_REQUEST_ADDRESS,
	 transponder->transponder_id,
	 S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE);
  request[S6350_WRITE_BLOCK_REQUEST_BLOCK_NUMBER] = block->block_number;
  memcpy(request + S6350_WRITE_BLOCK_REQUEST_DATA, block->data, transponder->bytes_per_block);

  if(write_request(s6350, S6350_COMMAND_COMBINE(S6350_COMMAND_WRITE_BLOCK, S6350_COMMAND_FLAGS_ADDRESS), request, request_size) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;
  
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.2 Miscellaneous Commands
 *    page 23: Reader Version Command (F0hex)
 */

#define S6350_READER_ANSWER_VERSION_MINOR 0
#define S6350_READER_ANSWER_VERSION_MAJOR 1
#define S6350_READER_ANSWER_VERSION_TYPE 2

int s6350_reader_version(s6350_t* s6350, s6350_reader_version_t* reader_version)
{
  u_int8_t* buffer = 0;
  int buffer_size = 0;
  
  clear_error(s6350);

  if(write_request(s6350, S6350_COMMAND_READER_VERSION, "", 0) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  reader_version->major = buffer[S6350_READER_ANSWER_VERSION_MAJOR];
  reader_version->minor = buffer[S6350_READER_ANSWER_VERSION_MINOR];
  reader_version->reader_type = buffer[S6350_READER_ANSWER_VERSION_TYPE];
  
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.2 Miscellaneous Commands
 *    page 24: RF Carrier on/off Command (F4hex)
 */

#define S6350_RF_CARRIER_ANSWER_STATUS 0

#define S6350_RF_CARRIER_SUCCESS 0x00

int s6350_rf_carrier(s6350_t* s6350, u_int8_t on_off)
{
  char request[1];
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  request[0] = on_off;

  if(write_request(s6350, S6350_COMMAND_RF_CARRIER, request, 1) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  if(buffer[S6350_RF_CARRIER_ANSWER_STATUS] != S6350_RF_CARRIER_SUCCESS) {
    free(buffer);
    s6350->error = ECANCELED;
    return -1;
  }
  
  free(buffer);

  return 0;
}

#ifdef POSIX_TERMIOS

#ifndef HAVE_CFMAKERAW
#define cfmakeraw(ptr) (ptr)->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR\
					 |IGNCR|ICRNL|IXON);\
                       (ptr)->c_oflag &= ~OPOST;\
                       (ptr)->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);\
                       (ptr)->c_cflag &= ~(CSIZE|PARENB);\
                       (ptr)->c_cflag |= CS8
#endif /* HAVE_CFMAKERAW */

#ifndef HAVE_CFSETSPEED
#if defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED)
#define cfsetspeed(t,speed) \
  (cfsetispeed(t,speed) || cfsetospeed(t,speed))
#else /* defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED) */
static int cfsetspeed(struct termios *t, int speed)
{
#ifdef HAVE_TERMIOS_CSPEED
	t->c_ispeed = speed;
	t->c_ospeed = speed;
#else /* HAVE_TERMIOS_CSPEED */
	t->c_cflag |= speed;
#endif /* HAVE_TERMIOS_CSPEED */
	return 0;
}
#endif /* defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED) */
#endif /* HAVE_CFSETSPEED */

#endif /* HAVE_CFMAKERAW */

static int calcrate(int baudrate)
{
#ifdef B300
	if (baudrate == 300)
		return B300;
#endif
#ifdef B1200
	if (baudrate == 1200)
		return B1200;
#endif
#ifdef B2400
	if (baudrate == 2400)
		return B2400;
#endif
#ifdef B4800
	if (baudrate == 4800)
		return B4800;
#endif
#ifdef B9600
	if (baudrate == 9600)
		return B9600;
#endif
#ifdef B19200
	else if (baudrate == 19200)
		return B19200;
#endif
#ifdef B38400
	else if (baudrate == 38400)
		return B38400;
#endif
#ifdef B57600
	else if (baudrate == 57600)
		return B57600;
#endif
#ifdef B115200
	else if (baudrate == 115200)
		return B115200;
#endif
#ifdef B230400
	else if (baudrate == 230400)
		return B230400;
#endif
#ifdef B460800
	else if (baudrate == 460800)
		return B460800;
#endif
	else {
		return -1;	/* invalid baud rate */
	}
}

/*
 * Open serial line with default setting of 57600, 8bits, no parity.
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 1.2 Programming Interface
 */
int s6350_init(s6350_t* s6350)
{
  clear_error(s6350);

  if(!s6350->device) {
    s6350->fd = -1;
    s6350->error = EINVAL;
    return -1;
  }
  
  if((s6350->fd = open(s6350->device, O_RDWR)) < 0) {
    s6350->error = errno;
    return -1;
  }

  /*
   * Init serial line
   */
  {
	struct 	termios tcn;
	int i;

	if (!isatty(s6350->fd)) {
	  s6350->error = ENOTTY;
	  close(s6350->fd);
	  s6350->fd = -1;
	  return -1;
	}

#ifdef POSIX_TERMIOS
	/* Set the tty to raw and to the correct speed */
	tcgetattr(s6350->fd, &tcn);

	tcn.c_oflag 	= 0;
	tcn.c_iflag 	= IGNBRK | IGNPAR;
	tcn.c_cflag 	= CREAD | CLOCAL | CS8;

	(void) cfsetspeed(&tcn, calcrate(57600));

	tcn.c_lflag = NOFLSH;

	cfmakeraw(&tcn);

	for (i = 0; i < 16; i++)
		tcn.c_cc[i] = 0;

	tcn.c_cc[VMIN] 	= 1;
	tcn.c_cc[VTIME] = 0;

	tcsetattr(s6350->fd, TCSANOW, &tcn);
#else /* POSIX_TERMIOS */
	/* Set the tty to raw and to the correct speed */
	ioctl(s6350->fd, TIOCGETP, &tcn);

	data->tco = tcn;

	tcn.sg_flags = RAW;
	tcn.sg_ispeed = calcrate(57600);
	tcn.sg_ospeed = calcrate(57600);

	ioctl(s6350->fd, TIOCSETN, &tcn);
#endif /* POSIX_TERMIOS */
  }

  /*
   * Set speed
   */
  {
#ifdef POSIX_TERMIOS
	struct 	termios tcn;

	tcgetattr(s6350->fd, &tcn);

	tcn.c_cflag 	= CREAD | CLOCAL | CS8;
	(void) cfsetspeed(&tcn, calcrate(57600));

	tcsetattr(s6350->fd, TCSADRAIN, &tcn);

#else /* POSIX_TERMIOS */
	struct sgttyb tcn;

	ioctl(s6350->fd, TIOCGETP, &tcn);

	tcn.sg_ispeed 	= calcrate(57600);
	tcn.sg_ospeed 	= calcrate(57600);

	ioctl(s6350->fd, TIOCSETN, &tcn);
#endif /* POSIX_TERMIOS */
  }

  /*
   * Sanity check on transponder. If the flash memory is not loaded,
   * we won't go anywhere.
   */
  {
    s6350_reader_version_t reader_version;    
    if(s6350_reader_version(s6350, &reader_version) >= 0) {
      if(reader_version.reader_type != S6350_READER_VERSION_TYPE_LOADED) {
	s6350->error = ENOEXEC;
	return -1;
      }
    }
  }

  /*
   * Make sure carrier is turned on.
   */
  if(s6350_rf_carrier(s6350, S6350_RF_CARRIER_ON) < 0)
    return -1;
  
  return 0;
}

void s6350_end(s6350_t* s6350)
{
  if(s6350) {
    if(s6350->fd > 0)
      close(s6350->fd);
  }
}

const char* s6350_version(void)
{
  return S6350_VERSION;
}
