/*
 * Copyright (C) 2003 _INRIA_
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <s6350.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <signal.h>

static void report_error(s6350_t* s6350, const char* message)
{
  fprintf(stderr, "Device %s\n", s6350->device);
  if(s6350->error == EIO) {
    fprintf(stderr, "%s: S6350 Error: %s\n", message, s6350_error(s6350));
  } else {
    fprintf(stderr, "0x%02x: %s\n", s6350->error, strerror(s6350->error));
  }
}

static s6350_t s6350;

static RETSIGTYPE alarm_f(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to establish communication with RFID reader at %s.\n\
FAILURE: Make sure a RFID reader is plugged in and turned on.\n\
\n\
", s6350.device);
  exit(-1);
}

int main(int argc, char** argv)
{

  if(argc > 1)
    s6350.device = argv[1];
  else
    s6350.device = "/dev/ttyS0";

  if(signal(SIGALRM, alarm_f) < 0) {
    perror("signal(SIGALRM)");
    return -1;
  }

  alarm(5);
  if(s6350_init(&s6350) < 0) {
    report_error(&s6350, "s6350_init");
  }
  alarm(0);

  {
    s6350_transponder_t transponder;
    s6350_block_t block = { 0 };
   
    if(s6350_transponder_details(&s6350, &transponder)) {
      if(s6350.s6350_error == S6350_ERROR_TRANSPONDER_NOT_FOUND) {
	fprintf(stderr, "\
\n\
FAILURE: The test program will attempt to read / write on a transponder,\n\
FAILURE: please put a transponder on the RFID reader.\n\
\n\
");
      }
      report_error(&s6350, "s6350_transponder_details");
      return -1;
    }

    block.data = (u_int8_t*)malloc(transponder.bytes_per_block);

    block.data[0] = block.data[1] = block.data[2] = block.data[3] = 'B';
    block.block_number = 0;

    if(s6350_write_block(&s6350, &transponder, &block)) {
      report_error(&s6350, "s6350_write_block");
      return -1;
    }

    for(block.block_number = 0; block.block_number < transponder.blocks; block.block_number++) {
	
      if(s6350_read_block(&s6350, &transponder, &block)) {
	report_error(&s6350, "s6350_read_block");
	return -1;
      }
    }

    {
      s6350_block_t blocks[32];
      int i;

      for(i = 0; i < 32; i++)
	blocks[i].data = (u_int8_t*)malloc(transponder.bytes_per_block);

      if(s6350_special_read_block(&s6350, &transponder, 0xFF, blocks)) {
	report_error(&s6350, "s6350_special_read_block");
	return -1;
      }
    }
  }
  
  s6350_end(&s6350);
  
  return 0;
}
