/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <rfid_reader.h>
#include <s6350.h>
#include <iso15693.h>

/*
 * Combine/split an int containing the command and the command flags
 */
#define ISO15693_COMMAND_COMBINE(command,flags) ((command)|((flags) << 8))
#define ISO15693_COMBINED_FLAGS(combined) ((combined >> 8) & 0xFF)
#define ISO15693_COMBINED_COMMAND(combined) (combined & 0xFF)

static int error_map[256] = {
  /* 0x00 */ ENOSYS, 
  /* 0x01 */ ENOTSUP, /* "The command is not supported, i.e. the request code is not recognized" */
  /* 0x02 */ EMFILE, /* "The command is not recognized, for example: a format error occured" */
  /* 0x03 */ EISDIR, /* "The command option is not supported" */
  /* 0x04 */ ENOSYS,
  /* 0x05 */ ENOSYS,
  /* 0x06 */ ENOSYS,
  /* 0x07 */ ENOSYS,
  /* 0x08 */ ENOSYS, 
  /* 0x09 */ ENOSYS, 
  /* 0x0a */ ENOSYS, 
  /* 0x0b */ ENOSYS, 
  /* 0x0c */ ENOSYS, 
  /* 0x0d */ ENOSYS, 
  /* 0x0e */ ENOSYS, 
  /* 0x0f */ EDOM, /* "Error with no information given or a specific error code is not supported" */
  /* 0x10 */ ENODEV, /* "The specified block is not available (doesn't exist)" */
  /* 0x11 */ ENOLCK, /* "The specified block is already locked and thus cannot be locked again" */
  /* 0x12 */ EPERM, /* "The specified block is locked and its content cannot be changed" */
  /* 0x13 */ ENOMEM, /* "The specified block was not successfully programmed" */
  /* 0x14 */ ESRCH, /* "The specified block was not successfully locked" */
  /* 0x15 */ ENOENT, /* not ISO-15693 "Transponder not found" */
  /* 0x16 */ ENOEXEC, /* not ISO-15693 "Invalid flags for command" */
  /* 0x17 */ ENOSYS,
  /* 0x18 */ ENOSYS,
  /* 0x19 */ ENOSYS,
  /* 0x1a */ ENOSYS,
  /* 0x1b */ ENOSYS,
  /* 0x1c */ ENOSYS,
  /* 0x1d */ ENOSYS,
  /* 0x1e */ ENOSYS,
  /* 0x1f */ ENOSYS,
};

static u_int16_t crc16_compute(u_int8_t* buffer, int buffer_size)
{
  /* NOT IMPLEMENTED */
  return 0;
}


void iso15693_crc16_set(u_int8_t* request, int request_length)
{
  /* NOT IMPLEMENTED */
  u_int16_t crc16 = crc16_compute(request, request_length);
  request[request_length - 1] = crc16 & 0xFF;
  request[request_length - 2] = (crc16 >> 8) & 0xFF;
}

int iso15693_crc16_check(u_int8_t* request, int request_length)
{
  /* NOT IMPLEMENTED */
  return 1;
}

static int write_command(iso15693_t* iso15693, int command, u_int8_t* buffer, int buffer_size)
{
  rfid_reader_t* driver = iso15693->reader;

  char* request = (char*)malloc(buffer_size + ISO15693_COMMAND_OVERHEAD);
  short request_length = buffer_size + ISO15693_COMMAND_OVERHEAD;
  int retval;

  request[ISO15693_FLAGS_INDEX] = ISO15693_COMBINED_FLAGS(command);
  request[ISO15693_COMMAND_INDEX] = ISO15693_COMBINED_COMMAND(command);

  memcpy(request + ISO15693_COMMAND_HEADER_SIZE, buffer, buffer_size);

  iso15693_crc16_set(request, request_length);
  
  if(iso15693->verbose > 1) {
    fprintf(stderr, "ISO-15693 command : ");
    rfid_dump_frame(request, request_length);
    fprintf(stderr, "\n");
  }

  retval = driver->iso15693_command_f(driver, request, request_length);

  free(request);

  return retval;
  
}

static int read_response(iso15693_t* iso15693, u_int8_t** bufferp, int* buffer_sizep)
{
  rfid_reader_t* driver = iso15693->reader;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(driver->iso15693_response_f(driver, &buffer, &buffer_size) < 0)
    return -1;

  if(iso15693->verbose > 1) {
    fprintf(stderr, "ISO-15693 response: ");
    rfid_dump_frame(buffer, buffer_size);
    fprintf(stderr, "\n");
  }

  if(!iso15693_crc16_check(buffer, buffer_size)) {
    free(buffer);
    iso15693->error = EFAULT;
    return -1;
  }

  if(buffer[ISO15693_FLAGS_INDEX] & ISO15693_RESPONSE_FLAG_ERROR) {
    iso15693->error = error_map[buffer[ISO15693_ERROR_INDEX]];
    free(buffer);
    return -1;
  }

  buffer_size -= ISO15693_RESPONSE_OVERHEAD;
  memmove(buffer, buffer + ISO15693_RESPONSE_HEADER_SIZE, buffer_size);

  *bufferp = buffer;
  *buffer_sizep = buffer_size;

  return 0;
}

static int simple_command(iso15693_t* iso15693, u_int8_t command, const u_int8_t* uid)
{
  u_int8_t request[ISO15693_UID_SIZE];
  u_int8_t null_uid[ISO15693_UID_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0 };
  int request_length;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(!memcmp(uid, null_uid, ISO15693_UID_SIZE)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(command,flags), request, request_length) < 0)
    return -1;

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  free(buffer);

  return 0;
}

int iso15693_inventory(iso15693_t* iso15693, iso15693_inventory_t* inventory, iso15693_inventory_result_t** resultsp, int* results_lengthp)
{
  u_int8_t* buffer;
  int buffer_size;
  u_int8_t request[ISO15693_AFI_SIZE + ISO15693_MASK_LENGTH_SIZE + ISO15693_MASK_VALUE_SIZE] = { 0 };
  int request_length = 0;
  int index = 0;
  int flags = ISO15693_COMMAND_FLAGS_INVENTORY;

  /*
   * AFI field
   */
  if(inventory->afi) {
    request[index] = inventory->afi;
    index += ISO15693_AFI_SIZE;
    flags |= ISO15693_COMMAND_FLAGS_AFI;
  }

  /*
   * Mask length field
   */
  if((inventory->mask_length >> 3) > ISO15693_MASK_LENGTH_SIZE) {
    iso15693->error = E2BIG;
    return -1;
  }
  request[index] = inventory->mask_length & 0xFF;
  index += ISO15693_MASK_LENGTH_SIZE;

  /*
   * Mask value field
   */
  {
    int nbytes = inventory->mask_length >> 3;
    int i;
    for(i = 0; i < nbytes; i++) {
      request[index++] = inventory->mask_value[i];
    }
  }

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_INVENTORY,flags), request, request_length) < 0)
    return -1;
  
  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  /*
   * In theory the total length of the result should be
   * 9 bytes (dsfid + uid). However we expect the underlying
   * driver to return a set of answers of the form (flags + dsfid + uid).
   */
  int results_length = buffer_size / 10;
  iso15693_inventory_result_t* results = 0;

  if(buffer_size % 10) {
    free(buffer);
    iso15693->error = EFAULT;
    return -1;
  }

  if(results_length) {
    int i;
    u_int8_t* pointer = buffer;
    results = (iso15693_inventory_result_t*)malloc(sizeof(iso15693_inventory_result_t) * results_length);

    for(i = 0; i < results_length; i++) {
      pointer += ISO15693_FLAGS_SIZE;

      results[i].dsfid = *pointer;
      pointer += ISO15693_DSFID_SIZE;

      memcpy(results[i].uid, pointer, ISO15693_UID_SIZE);
      pointer += ISO15693_UID_SIZE;
    }
  }

  free(buffer);

  *resultsp = results;
  *results_lengthp = results_length;

  return 0;
}

int iso15693_stay_quiet(iso15693_t* iso15693, u_int8_t* uid)
{
  u_int8_t request[ISO15693_UID_SIZE];
  int request_length = ISO15693_UID_SIZE;

  memcpy(request, uid, ISO15693_UID_SIZE);
  
  return write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_STAY_QUIET,ISO15693_COMMAND_FLAGS_ADDRESS), request, request_length);
}

int iso15693_read_single_block(iso15693_t* iso15693, rfid_transponder_t* transponder, rfid_block_t* block)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  int want_security_status = block->security_status;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(want_security_status) {
    flags |= ISO15693_COMMAND_FLAGS_OPTION;
  }
  block->security_status = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  if(block->block_number > transponder->blocks) {
    iso15693->error = E2BIG;
    return -1;
  }

  request[index] = block->block_number & 0xFF;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_READ_SINGLE_BLOCK,flags), request, request_length) < 0)
    return -1;

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  index = 0;

  if(want_security_status) {
    block->security_status = buffer[index];
    index += ISO15693_BLOCK_SECURITY_STATUS_SIZE;
  }

  /*
   * The read single block command implicitly tells us the size of a
   * block.  If the transponder has a wrong idea of this size, fix it.
   */
  if(transponder->bytes_per_block != buffer_size - index) {
    transponder->bytes_per_block = buffer_size - index;
    if(iso15693->verbose) fprintf(stderr, "fixed transponder block size to %d\n", transponder->bytes_per_block);
  }
  
  memcpy(block->data, buffer + index, transponder->bytes_per_block);

  free(buffer);

  return 0;
}

int iso15693_write_single_block(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* block)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_SIZE_MAX];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  if(block->block_number > transponder->blocks) {
    iso15693->error = E2BIG;
    return -1;
  }

  request[index] = block->block_number;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  memcpy(request + index, block->data, transponder->bytes_per_block);
  index += transponder->bytes_per_block;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_WRITE_SINGLE_BLOCK,flags), request, request_length) < 0)
    return -1;

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  free(buffer);

  return 0;
}

int iso15693_read_multiple_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* blocks, int blocks_length)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_NUMBER_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  int want_security_status = blocks[0].security_status;
  int block_start = blocks[0].block_number & 0xFF;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(want_security_status) {
    flags |= ISO15693_COMMAND_FLAGS_OPTION;
  }
  blocks[0].security_status = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  if(block_start > transponder->blocks ||
     block_start + blocks_length > transponder->blocks) {
    iso15693->error = E2BIG;
    return -1;
  }

  /*
   * Index of the first block to read.
   */
  request[index] = block_start;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  /*
   * Number of blocks to read. The actual number of blocks read
   * is incremented by one (so that 0-255 covers 1-256).
   */
  request[index] = (blocks_length - 1) & 0xFF;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_READ_MULTIPLE_BLOCKS,flags), request, request_length) < 0)
    return -1;

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  {
    index = 0;
    int blocks_index = 0;

    for(blocks_index = 0; blocks_index < blocks_length; blocks_index++) {
      if(want_security_status) {
	blocks[blocks_index].block_number = block_start + index;
	blocks[blocks_index].security_status = buffer[index];
	index += ISO15693_BLOCK_SECURITY_STATUS_SIZE;
      }
      memcpy(blocks[blocks_index].data, buffer + index, transponder->bytes_per_block);
      index += transponder->bytes_per_block;
    }
  }

  free(buffer);

  return 0;
}

int iso15693_write_multiple_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* blocks, int blocks_length)
{
  u_int8_t* request;
  int request_length = ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE + ISO15693_BLOCK_NUMBER_SIZE + blocks_length * transponder->bytes_per_block;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int block_start = blocks[0].block_number & 0xFF;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  request = (u_int8_t*)malloc(request_length);

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  if(block_start > transponder->blocks ||
     block_start + blocks_length > transponder->blocks) {
    iso15693->error = E2BIG;
    free(request);
    return -1;
  }

  request[index] = block_start;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request[index] = blocks_length;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  {
    int i;
    for(i = 0; i < blocks_length; i++) {
      memcpy(request + index, blocks[i].data, transponder->bytes_per_block);
      index += transponder->bytes_per_block;
    }
  }
  
  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_WRITE_MULTIPLE_BLOCKS,flags), request, request_length) < 0)
    return -1;

  free(request);

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  free(buffer);

  return 0;
}

int iso15693_lock_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* block)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_BLOCK_NUMBER_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request[index] = block->block_number;
  index += ISO15693_BLOCK_NUMBER_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_LOCK_BLOCK,flags), request, request_length) < 0)
    return -1;

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  free(buffer);

  return 0;
}

int iso15693_select(iso15693_t* iso15693, const u_int8_t* uid)
{
  return simple_command(iso15693, ISO15693_COMMAND_SELECT, uid);
}

int iso15693_reset_to_ready(iso15693_t* iso15693, const u_int8_t* uid)
{
  return simple_command(iso15693, ISO15693_COMMAND_RESET_TO_READY, uid);
}

int iso15693_write_afi(iso15693_t* iso15693, const rfid_transponder_t* transponder)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_AFI_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request[index] = iso15693_transponder->afi;
  index += ISO15693_AFI_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_WRITE_AFI,flags), request, request_length) < 0)
    return -1;

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  free(buffer);

  return 0;
}

int iso15693_lock_afi(iso15693_t* iso15693, const u_int8_t* uid)
{
  return simple_command(iso15693, ISO15693_COMMAND_LOCK_AFI, uid);
}

int iso15693_write_dsfid(iso15693_t* iso15693, const rfid_transponder_t* transponder)
{
  u_int8_t request[ISO15693_UID_SIZE + ISO15693_AFI_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request[index] = iso15693_transponder->dsfid;
  index += ISO15693_DSFID_SIZE;

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_WRITE_DSFID,flags), request, request_length) < 0)
    return -1;

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  free(buffer);

  return 0;
}

int iso15693_lock_dsfid(iso15693_t* iso15693, const u_int8_t* uid)
{
  return simple_command(iso15693, ISO15693_COMMAND_LOCK_DSFID, uid);
}

int iso15693_get_system_information(iso15693_t* iso15693, rfid_transponder_t* transponder)
{
  u_int8_t request[ISO15693_UID_SIZE];
  int request_length;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  u_int8_t flags = 0;
  int index = 0;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  if(!iso15693_transponder_null(iso15693_transponder)) {
    flags |= ISO15693_COMMAND_FLAGS_ADDRESS;
    memcpy(request, iso15693_transponder->uid, ISO15693_UID_SIZE);
    index += ISO15693_UID_SIZE;
  }

  request_length = index;

  if(write_command(iso15693, ISO15693_COMMAND_COMBINE(ISO15693_COMMAND_GET_SYSTEM_INFORMATION,flags), request, request_length) < 0)
    return -1;

  if(read_response(iso15693, &buffer, &buffer_size) < 0)
    return -1;

  {
    u_int8_t* pointer = buffer;
    u_int8_t info;

    iso15693_transponder->info_flags = info = *pointer;
    pointer += ISO15693_INFO_FLAGS_SIZE;
    
    memcpy(iso15693_transponder->uid, pointer, ISO15693_UID_SIZE);
    pointer += ISO15693_UID_SIZE;
    
    if(info & ISO15693_INFO_FLAGS_DSFID) {
      iso15693_transponder->dsfid = *pointer;
      pointer += ISO15693_DSFID_SIZE;
    }

    if(info & ISO15693_INFO_FLAGS_AFI) {
      iso15693_transponder->afi = *pointer;
      pointer += ISO15693_AFI_SIZE;
    }

    if(info & ISO15693_INFO_FLAGS_VICC_MEMORY_SIZE) {
      iso15693_transponder->vicc_memory = ((pointer[1] & 0xFF) << 8) | (pointer[0] & 0xFF);
      pointer += ISO15693_VICC_MEMORY_SIZE_SIZE;
    }

    if(info & ISO15693_INFO_FLAGS_IC_REFERENCE) {
      iso15693_transponder->ic_ref = *pointer;
      pointer += ISO15693_IC_REFERENCE_SIZE;
    }
  }

  iso15693_transponder_vicc_blocks_get(transponder, transponder->blocks);
  iso15693_transponder_vicc_bytes_per_block_get(transponder, transponder->bytes_per_block);

  free(buffer);

  return 0;
}

int iso15693_transponder_cmp(iso15693_transponder_t* a, iso15693_transponder_t* b)
{
  return memcmp(a->uid, b->uid, ISO15693_UID_SIZE);
}

int iso15693_transponder_null(iso15693_transponder_t* transponder)
{
  int i;

  for(i = 0; i < ISO15693_UID_SIZE; i++)
    if(transponder->uid[i])
      return 0;

  return 1;
}

static const char* drv_version(struct rfid_reader* reader)
{
  return RFID_VERSION;
}

static void drv_verbose(struct rfid_reader* reader, int verbosity)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;

  iso15693->verbose = verbosity;
}

static int drv_init(struct rfid_reader* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  iso15693->device = reader->device;
  iso15693->error = 0;
  iso15693->flags = 0;

  if(s6350_reader_alloc(&iso15693->reader) < 0)
    return -1;

  iso15693->reader->parent = reader;
  iso15693->reader->device = iso15693->device;

  if(iso15693->reader->init_f(iso15693->reader) < 0)
    return -1;

  return 0;
}

static void drv_end(struct rfid_reader* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;

  return driver->end_f(driver);
}

static char* drv_strerror(struct rfid_reader* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;
  char* message = (char*)malloc(512);

  if(iso15693->error) {
    sprintf(message, "%d: %s", iso15693->error, strerror(iso15693->error));
  } else if(driver->error_f(driver)) {
    char* error_string = driver->strerror_f(driver);
    sprintf(message, "%s", error_string);
    free(error_string);
  } else {
    sprintf(message, "Success");
  }

  return message;
}

static int drv_error(struct rfid_reader* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;

  if(iso15693->error < 0) {
    return iso15693->error;
  } else {
    return driver->error_f(driver);
  }
}

static int drv_read(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_block_t blocks[number_of_blocks];
  int i;
  u_int8_t* pointer;

  if(block_start > transponder->blocks ||
     block_start + number_of_blocks > transponder->blocks) {
    iso15693->error = E2BIG;
    return -1;
  }

  transponder->data_length = number_of_blocks * transponder->bytes_per_block;
  pointer = transponder->data + block_start * transponder->bytes_per_block;
  
  for(i = 0; i < number_of_blocks; i++) {
    blocks[i].data = pointer;
    pointer += transponder->bytes_per_block;
  }

  blocks[0].block_number = block_start;
  blocks[0].security_status = 1;
      
  return iso15693_read_multiple_blocks(iso15693, transponder, blocks, number_of_blocks);
}

static int drv_write(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  u_int8_t* pointer;
  int i;

  if(block_start + number_of_blocks > transponder->blocks) {
    iso15693->error = EFBIG;
    return -1;
  }

  pointer = transponder->data + block_start * transponder->bytes_per_block;
  
  for(i = 0; i < number_of_blocks; i++) {
    int retval;
    rfid_block_t block;
    block.data = pointer;
    block.block_number = block_start + i;
    block.security_status = 0;
    
    pointer += transponder->bytes_per_block;

    while((retval = iso15693_write_single_block(iso15693, transponder, &block)) < 0 &&
	    reader->error_f(reader) == EAGAIN) {
      if(iso15693->verbose == 1) fprintf(stderr, "!");
    }
    if(retval < 0)
      return -1;
    if(iso15693->verbose == 1) fprintf(stderr, "W");

  }

  if(iso15693->verbose == 1) fprintf(stderr, "\n");
  
  return 0;
}

static  int drv_inventory(struct rfid_reader* reader, rfid_transponder_t*** transpondersp, int* transponders_lengthp, int* transponders_sizep)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;

  iso15693_inventory_result_t* results = 0;
  int results_length = 0;
  iso15693_inventory_t inventory;
  int retval = -1;
  int i;

  rfid_transponder_t** transponders = 0;

  memset(&inventory, '\0', sizeof(iso15693_inventory_t));
  *transponders_lengthp = 0;

  if(iso15693_inventory(iso15693, &inventory, &results, &results_length) < 0)
    goto finish;

  /* No transponders is not an error, it's an empty list. */
  if(results_length < 1) {
    retval = 0;
    goto finish;
  }

  if(*transpondersp != 0) {
    if(*transponders_sizep < results_length) {
      iso15693->error = E2BIG;
      goto finish;
    }

    transponders = *transpondersp;
  } else {
    transponders = (rfid_transponder_t**)malloc(results_length * sizeof(rfid_transponder_t*));
    memset(transponders, '\0', results_length * sizeof(rfid_transponder_t**));
    for(i = 0; i < results_length; i++) {
      reader->transponder_alloc_f(reader, transponders + i);
    }
  }

  for(i = 0; i < results_length; i++) {
    reader->transponder_clear_f(reader, transponders[i]);
    iso15693_transponder_uid_set(transponders[i], results[i].uid);
    iso15693_transponder_dsfid_set(transponders[i], results[i].dsfid);
  }

  retval = 0;

  if(*transpondersp == 0) {
    *transpondersp = transponders;
    *transponders_sizep = results_length;
  }
  *transponders_lengthp = results_length;

 finish:
  if(results)
    free(results);
  
  return retval;
}

static char* drv_transponder_describe(struct rfid_reader* reader, const rfid_transponder_t* transponder)
{
  char* str = malloc(512);
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;

  sprintf(str, "uid = 0x%02x%02x%02x%02x%02x%02x%02x%02x\n",
	  iso15693_transponder->uid[7],
	  iso15693_transponder->uid[6],
	  iso15693_transponder->uid[5],
	  iso15693_transponder->uid[4],
	  iso15693_transponder->uid[3],
	  iso15693_transponder->uid[2],
	  iso15693_transponder->uid[1],
	  iso15693_transponder->uid[0]
	  );

  if(iso15693_transponder->info_flags & ISO15693_INFO_FLAGS_DSFID)
    sprintf(str + strlen(str), "dsfid = 0x%02x\n", iso15693_transponder->dsfid);

  if(iso15693_transponder->info_flags & ISO15693_INFO_FLAGS_AFI)
    sprintf(str + strlen(str), "afi = 0x%02x\n", iso15693_transponder->afi);

  if(iso15693_transponder->info_flags & ISO15693_INFO_FLAGS_VICC_MEMORY_SIZE)
    sprintf(str + strlen(str), "vicc_memory = 0x%04x\n", iso15693_transponder->vicc_memory);

  if(iso15693_transponder->info_flags & ISO15693_INFO_FLAGS_IC_REFERENCE)
    sprintf(str + strlen(str), "ic_ref = 0x%02x\n", iso15693_transponder->ic_ref);

  sprintf(str + strlen(str), "blocks = %d\n", transponder->blocks);
  sprintf(str + strlen(str), "bytes_per_block = %d\n", transponder->bytes_per_block);
  
  return str;
}

static int drv_transponder_cmp(struct rfid_reader* reader, rfid_transponder_t* transponder_a, rfid_transponder_t* transponder_b)
{
  return iso15693_transponder_cmp((iso15693_transponder_t*)transponder_a->private,
			       (iso15693_transponder_t*)transponder_b->private);
}

static int drv_transponder_null(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  return iso15693_transponder_null((iso15693_transponder_t*)transponder->private);
}

static int drv_transponder_present(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;

  return iso15693_get_system_information(iso15693, transponder);

#if 0
  rfid_reader_t* driver = iso15693->reader;

  iso15693_inventory_result_t* results = 0;
  int results_length = 0;
  iso15693_inventory_t inventory;
  int retval = -1;

  memset(&inventory, '\0', sizeof(iso15693_inventory_t));

  if(iso15693_inventory(iso15693, &inventory, &results, &results_length) < 0)
    goto finish;

  if(results_length < 1) {
    iso15693->error = ENOENT;
    goto finish;
  }

  iso15693_transponder_uid_set(transponder, results[0].uid);
  iso15693_transponder_dsfid_set(transponder, results[0].dsfid);

  retval = 0;

 finish:
  if(results)
    free(results);
  
  return retval;
#endif
}

static int drv_transponder_alloc(struct rfid_reader* reader, rfid_transponder_t** transponderp)
{
  rfid_transponder_t* transponder = (rfid_transponder_t*)malloc(sizeof(rfid_transponder_t));
  memset(transponder, '\0', sizeof(rfid_transponder_t));
  transponder->private = (iso15693_transponder_t*)malloc(sizeof(iso15693_transponder_t));
  memset(transponder->private, '\0', sizeof(iso15693_transponder_t));

  *transponderp = transponder;

  return 0;
}

static int drv_transponder_free(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  free(transponder->private);
  free(transponder);

  return 0;
}

static int drv_transponder_clear(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  iso15693_transponder_t* iso15693 = (iso15693_transponder_t*)transponder->private;
  memset(iso15693, '\0', sizeof(iso15693_transponder_t));
  memset(transponder, '\0', sizeof(rfid_transponder_t));
  transponder->private = iso15693;

  return 0;
}

static int drv_transponder_copy(struct rfid_reader* reader, rfid_transponder_t* transponder_a, rfid_transponder_t* transponder_b)
{
  iso15693_transponder_t* iso15693_a = (iso15693_transponder_t*)transponder_a->private;
  iso15693_transponder_t* iso15693_b = (iso15693_transponder_t*)transponder_b->private;

  memcpy(iso15693_a, iso15693_b, sizeof(iso15693_transponder_t));
  memcpy(transponder_a, transponder_b, sizeof(rfid_transponder_t));
  transponder_a->private = iso15693_a;

  return 0;
}

static int drv_transponder_id_get(struct rfid_reader* reader, rfid_transponder_t* transponder, char** idp)
{
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;

  char* id = malloc(64);

  sprintf(id, "%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x",
	  iso15693_transponder->uid[7],
	  iso15693_transponder->uid[6],
	  iso15693_transponder->uid[5],
	  iso15693_transponder->uid[4],
	  iso15693_transponder->uid[3],
	  iso15693_transponder->uid[2],
	  iso15693_transponder->uid[1],
	  iso15693_transponder->uid[0]);

  *idp = id;

  return 0;
}

static int drv_transponder_id_set(struct rfid_reader* reader, rfid_transponder_t* transponder, char* id)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  iso15693_transponder_t* iso15693_transponder = (iso15693_transponder_t*)transponder->private;
  unsigned int id0, id1, id2, id3, id4, id5, id6, id7;

  if(sscanf(id, "%x.%x.%x.%x.%x.%x.%x.%x",
	    &id7,
	    &id6,
	    &id5,
	    &id4,
	    &id3,
	    &id2,
	    &id1,
	    &id0) != 8) {
    iso15693->error = ECANCELED;
    return -1;
  }

  iso15693_transponder->uid[7] = id7 & 0xFF;
  iso15693_transponder->uid[6] = id6 & 0xFF;
  iso15693_transponder->uid[5] = id5 & 0xFF;
  iso15693_transponder->uid[4] = id4 & 0xFF;
  iso15693_transponder->uid[3] = id3 & 0xFF;
  iso15693_transponder->uid[2] = id2 & 0xFF;
  iso15693_transponder->uid[1] = id1 & 0xFF;
  iso15693_transponder->uid[0] = id0 & 0xFF;

  return 0;
}

static int drv_free(rfid_reader_t* reader)
{
  iso15693_t* iso15693 = (iso15693_t*)reader->private;
  rfid_reader_t* driver = iso15693->reader;

  driver->free_f(driver);

  free(reader->private);
  free(reader);
  return 0;
}

static int drv_alloc(rfid_reader_t** readerp)
{
  rfid_reader_t* reader = (rfid_reader_t*)malloc(sizeof(rfid_reader_t));
  iso15693_t* iso15693 = (iso15693_t*)malloc(sizeof(iso15693_t));
  memset(iso15693, '\0', sizeof(iso15693_t));

  reader->private = iso15693;

  reader->alloc_f = drv_alloc;
  reader->free_f = drv_free;

  reader->init_f = drv_init;
  reader->end_f = drv_end;

  reader->strerror_f = drv_strerror;
  reader->error_f = drv_error;
  reader->version_f = drv_version;
  reader->verbose_f = drv_verbose;

  reader->read_f = drv_read;
  reader->write_f = drv_write;
  reader->inventory_f = drv_inventory;
  reader->iso15693_command_f = 0;
  reader->iso15693_response_f = 0;

  reader->transponder_describe_f = drv_transponder_describe;
  reader->transponder_cmp_f = drv_transponder_cmp;
  reader->transponder_null_f = drv_transponder_null;
  reader->transponder_present_f = drv_transponder_present;
  reader->transponder_alloc_f = drv_transponder_alloc;
  reader->transponder_free_f = drv_transponder_free;
  reader->transponder_clear_f = drv_transponder_clear;
  reader->transponder_copy_f = drv_transponder_copy;
  reader->transponder_id_get_f = drv_transponder_id_get;
  reader->transponder_id_set_f = drv_transponder_id_set;

  *readerp = reader;

  return 0;
}

int iso15693_reader_alloc(rfid_reader_t** reader)
{
  return drv_alloc(reader);
}
