/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifndef _RFID_READER_H_
#define _RFID_READER_H_

#include <sys/types.h>

/*
 * Assume that a RFID transponder can hold a maximum of 8192 bytes,
 * according to ISO-15693 definition (256 blocks of 256 bits = 64Kbits).
 */
#define RFID_TRANSPONDER_DATA_SIZE 8192
/*
 * Size of a string holding hexadecimal representation of 
 * a RFID transponder identity, with separators and trailing 
 * NULL.
 */
#define RFID_TRANSPONDER_HEX_ID_SIZE 64

/*
 * RFID transponder identity and data.
 */
typedef struct {
  u_int8_t blocks;
  u_int8_t bytes_per_block;
  u_int8_t data[RFID_TRANSPONDER_DATA_SIZE + 1]; /* + 1 for trailing NULL */
  int data_length;			/* Length of data in the
					   data field. */
  void* private;
} rfid_transponder_t;

/*
 * Chunk of data used to read/write data from/to the RFID transponder.
 */
typedef struct {
  /*
   * Out
   */
  u_int8_t* data;		/* Data buffer whose length must be
				   rfid_transponder_t.bytes_per_block. */
  /*
   * In/Out
   */
  u_int8_t security_status;	/* If 1 block is locked, if 0 block is
				   not locked. */
  u_int8_t block_number;	/* Position of the data block in the
				   transponder, in the range
				   [0,rfid_transponder_t.blocks[.*/
} rfid_block_t;

typedef struct rfid_reader {
  char* device;
  void* private;
  void* parent;

  int (*alloc_f)(struct rfid_reader** reader);
  int (*free_f)(struct rfid_reader* reader);

  int (*init_f)(struct rfid_reader* reader);
  void (*end_f)(struct rfid_reader* reader);

  char* (*strerror_f)(struct rfid_reader* reader);
  int (*error_f)(struct rfid_reader* reader);

  const char* (*version_f)(struct rfid_reader* reader);
  void (*verbose_f)(struct rfid_reader* reader, int verbosity);

  int (*read_f)(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder);
  int (*write_f)(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder);
  int (*inventory_f)(struct rfid_reader* reader, rfid_transponder_t*** transpondersp, int* transponders_lengthp, int* transponders_sizep);

  int (*iso15693_command_f)(struct rfid_reader* reader, u_int8_t* data, int data_length);
  int (*iso15693_response_f)(struct rfid_reader* reader, u_int8_t** datap, int* data_lengthp);

  char* (*transponder_describe_f)(struct rfid_reader* reader, const rfid_transponder_t* transponder);
  int (*transponder_cmp_f)(struct rfid_reader* reader, rfid_transponder_t* transponder_a, rfid_transponder_t* transponder_b);
  int (*transponder_null_f)(struct rfid_reader* reader, rfid_transponder_t* transponder);
  int (*transponder_present_f)(struct rfid_reader* reader, rfid_transponder_t* transponder);
  int (*transponder_alloc_f)(struct rfid_reader* reader, rfid_transponder_t** transponder);
  int (*transponder_free_f)(struct rfid_reader* reader, rfid_transponder_t* transponder);
  int (*transponder_copy_f)(struct rfid_reader* reader, rfid_transponder_t* transponder_a, rfid_transponder_t* transponder_b);
  int (*transponder_clear_f)(struct rfid_reader* reader, rfid_transponder_t* transponder);
  int (*transponder_id_get_f)(struct rfid_reader* reader, rfid_transponder_t* transponder, char** idp);
  int (*transponder_id_set_f)(struct rfid_reader* reader, rfid_transponder_t* transponder, char* id);
} rfid_reader_t;

void rfid_dump_frame(u_int8_t* frame, int frame_length);

#endif /* _RFID_READER_H_ */
