/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <memory.h>
#include <errno.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/ioctl.h>

#include <rfid_reader.h>
#include <iso15693.h>
#include <s6350.h>

/*
 * Assume that a RFID transponder can hold a maximum of 32 bytes.
 */
#define S6350_TRANSPONDER_DATA_SIZE 32
/*
 * Size of a string holding hexadecimal representation of 
 * a RFID transponder identity, with separators and trailing 
 * NULL.
 */
#define S6350_TRANSPONDER_HEX_ID_SIZE 32

#define S6350_RF_CARRIER_ON 0xFF
#define S6350_RF_CARRIER_OFF 0x00

#define S6350_REQUEST_HEADER_SIZE 7
#define S6350_REQUEST_TRAILER_SIZE 2
#define S6350_REQUEST_OVERHEAD S6350_REQUEST_HEADER_SIZE + S6350_REQUEST_TRAILER_SIZE

#define S6350_HEADER_SOF 0
#define S6350_HEADER_LENGTH_LSB 1
#define S6350_HEADER_LENGTH_MSB 2
#define S6350_HEADER_NODE_ADDRESS_LSB 3
#define S6350_HEADER_NODE_ADDRESS_MSB 4
#define S6350_HEADER_COMMAND_FLAGS 5
#define S6350_HEADER_COMMAND 6

#define S6350_ERROR_DATA 7

#define S6350_VALUE_SOF 0x01

/*
 * Combine/split an int containing the command and the command flags
 */
#define S6350_COMMAND_COMBINE(command,flags) ((command)|((flags) << 8))
#define S6350_COMBINED_FLAGS(combined) ((combined >> 8) & 0xFF)
#define S6350_COMBINED_COMMAND(combined) (combined & 0xFF)

#define S6350_COMMAND_FLAGS_ADDRESS		0x10
#define S6350_COMMAND_FLAGS_ERROR		0x10

#define S6350_COMMAND_READ_BLOCK		0x02
#define S6350_COMMAND_WRITE_BLOCK		0x03
#define S6350_COMMAND_LOCK_BLOCK		0x04
#define S6350_COMMAND_READ_TRANSPONDER_DETAILS	0x05
#define S6350_COMMAND_SPECIAL_READ_BLOCK	0x0F
#define S6350_COMMAND_ISO15693			0x60
#define S6350_COMMAND_INITIATE_FLASH_LOADER	0xD0
#define S6350_COMMAND_SEND_DATA_TO_FLASH	0xD8
#define S6350_COMMAND_READER_VERSION		0xF0
#define S6350_COMMAND_READER_INPUTS		0xF1
#define S6350_COMMAND_WRITE_READER_OUTPUT	0xF2
#define S6350_COMMAND_RF_CARRIER		0xF4
#define S6350_COMMAND_BAUD_RATE_CONFIGURATION	0xFF

#define clear_error(o) ((o)->error = 0)

static int error_map[256] = {
  /* 0x00 */ ENOSYS, /* "Invalid error code" */
  /* 0x01 */ ENOENT, /* "Transponder not found" */
  /* 0x02 */ EPERM, /* "Command not supported" */
  /* 0x03 */ EPIPE, /* "Packet BCC invalid" */
  /* 0x04 */ ERANGE, /* "Packet flags invalid for command" */
  /* 0x05 */ ENXIO, /* "General write failure" */
  /* 0x06 */ ENOLCK, /* "Write failure due to locked block" */
  /* 0x07 */ ENOTSUP, /* "Transponder does not support function" */
  /* 0x08 */ ENOSYS, 
  /* 0x09 */ ENOSYS, 
  /* 0x0a */ ENOSYS, 
  /* 0x0b */ ENOSYS, 
  /* 0x0c */ ENOSYS, 
  /* 0x0d */ ENOSYS, 
  /* 0x0e */ ENOSYS, 
  /* 0x0f */ ESRCH 
};

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.1.5 BCC 
 */
static u_int16_t bcc_compute(u_int8_t* buffer, int buffer_size)
{
  int i;
  u_int8_t lrc = 0;
  for(i = 0; i < buffer_size; i++) {
    lrc ^= buffer[i];
  }
  return lrc | ((~lrc) << 8);
}

static void bcc_set(u_int8_t* buffer, int buffer_size)
{
  u_int16_t bcc = bcc_compute(buffer, buffer_size - 2);
  buffer[buffer_size - 1] = (bcc >> 8) & 0xFF;
  buffer[buffer_size - 2] = bcc & 0xFF;
}

static int bcc_check(u_int8_t* buffer, int buffer_size)
{
  u_int16_t bcc = bcc_compute(buffer, buffer_size - 2);
  return buffer[buffer_size - 1] == ((bcc >> 8) & 0xFF) &&
    buffer[buffer_size - 2] == (bcc & 0xFF);
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.1.1 Request Packet Format (Host to Reader)
 */
static int write_request(s6350_t* s6350, int command, u_int8_t* buffer, int buffer_size)
{
  char* request = (char*)malloc(buffer_size + S6350_REQUEST_OVERHEAD);
  short request_length = buffer_size + S6350_REQUEST_OVERHEAD;
  int wrote;

  request[S6350_HEADER_SOF] = 0x01;
  request[S6350_HEADER_LENGTH_LSB] = request_length & 0xFF;
  request[S6350_HEADER_LENGTH_MSB] = (request_length >> 8) & 0xFF;
  request[S6350_HEADER_NODE_ADDRESS_LSB] = 0;
  request[S6350_HEADER_NODE_ADDRESS_MSB] = 0;
  request[S6350_HEADER_COMMAND_FLAGS] = S6350_COMBINED_FLAGS(command);
  request[S6350_HEADER_COMMAND] = S6350_COMBINED_COMMAND(command);

  memcpy(request + S6350_REQUEST_HEADER_SIZE, buffer, buffer_size);

  bcc_set(request, request_length);

  if(s6350->verbose > 1) {
    fprintf(stderr, "S6350 request : ");
    rfid_dump_frame(request, request_length);
    fprintf(stderr, "\n");
  }
    
  if((wrote = write(s6350->fd, request, request_length)) != request_length)
    s6350->error = errno;
  else
    s6350->error = 0;

  free(request);

  return s6350->error < 0 ? -1 : 0;
}

/*
 * Read up to buffer_size bytes from input and times out after 1
 * second. This would not be necessary if the RFID reader reliably
 * sent the expected number of bytes. Unfortunately, when there is no
 * flow control on the serial line, some messages may be incomplete.
 */
static int timeout_read(s6350_t* s6350, char* buffer, int buffer_size)
{
  fd_set set;
  int fd = s6350->fd;
  /* 1 second */
  struct timeval timeout = {
    1,
    0
  };
  int do_read;

  FD_ZERO(&set);
  FD_SET(fd, &set);
  if((do_read = select(fd + 1, &set, NULL, NULL, &timeout)) < 0) {
    s6350->error = errno;
    return -1;
  }

  if(do_read) {
    int available;
    int retval;

    if(ioctl(fd, FIONREAD, &available) < 0) {
      s6350->error = errno;
      return -1;
    }

    if(available > buffer_size)
      available = buffer_size;

    if((retval = read(fd, buffer, available)) < 0) {
      s6350->error = errno;
      return -1;
    } else {
      return retval;
    }
  } else {
    s6350->error = EAGAIN;
    return -1;
  }
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.1.2 Response Packet Format (Reader to Host)
 * 3.1.4 Command Flags Response
 */
static int read_answer(s6350_t* s6350, u_int8_t** bufferp, int* buffer_sizep)
{
  char answer_header[S6350_REQUEST_HEADER_SIZE];
  char* buffer;
  int read_bytes;
  u_int16_t length;
  int to_read;
  int offset = 0;

  *bufferp = 0;
  *buffer_sizep = 0;

  /*
   * Wait for the Start Of Frame (SOF) to appear in the input stream.
   * Most of the time it's the first byte but there is no guarantee.
   */
  while((read_bytes = timeout_read(s6350, answer_header, 1)) >= 0 &&
	answer_header[0] != S6350_VALUE_SOF)
    ;
  if(read_bytes < 0) {
    return -1;
  }
  offset++;
  
  /*
   * Read the rest of the header, the Start Of Frame (SOF) is already
   * in the buffer.
   */
  while(offset < S6350_REQUEST_HEADER_SIZE) {
    if((read_bytes = timeout_read(s6350, answer_header + offset, S6350_REQUEST_HEADER_SIZE - offset)) < 0) {
      return -1;
    }
    offset += read_bytes;
  }

  /*
   * Extract the total length of the frame
   */
  length = answer_header[S6350_HEADER_LENGTH_LSB] |
    (answer_header[S6350_HEADER_LENGTH_MSB] << 8);

  /*
   * Sanity checks to prevent memory corruption if message
   * is corrupted.
   */
  if(length < S6350_REQUEST_OVERHEAD) {
    s6350->error = EINVAL;
    return -1;
  }
  /*
   * Sanity checks to detect corrupted headers.
   */
  if(answer_header[S6350_HEADER_NODE_ADDRESS_LSB] != 0x00 ||
     answer_header[S6350_HEADER_NODE_ADDRESS_MSB] != 0x00) {
    s6350->error = EAGAIN;
    return -1;
  }

  to_read = length - S6350_REQUEST_HEADER_SIZE;

  buffer = (char*)malloc(length);
  memcpy(buffer, answer_header, S6350_REQUEST_HEADER_SIZE);

  offset = 0;
  while(offset < to_read) {
    if((read_bytes = timeout_read(s6350, buffer + S6350_REQUEST_HEADER_SIZE + offset, to_read - offset)) < 0) {
      free(buffer);
      return -1;
    }
    offset += read_bytes;
  }

  if(s6350->verbose > 1) {
    fprintf(stderr, "S6350 answer: ");
    rfid_dump_frame(buffer, length);
    fprintf(stderr, "\n");
  }

  /*
   * Check consistency of the frame content.
   */
  if(!bcc_check(buffer, length)) {
    free(buffer);
    s6350->error = EFAULT;
    return -1;
  }

  /*
   * If the transponder returns on error, abort.
   * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
   * 3.1.4 Command Flags Response
   */
  if(buffer[S6350_HEADER_COMMAND_FLAGS] == S6350_COMMAND_FLAGS_ERROR) {
    s6350->error = error_map[buffer[S6350_ERROR_DATA] & 0xFF];
    free(buffer);
    return -1;
  }

  memmove(buffer, buffer + S6350_REQUEST_HEADER_SIZE, to_read - S6350_REQUEST_TRAILER_SIZE);
  
  *bufferp = buffer;
  *buffer_sizep = to_read - S6350_REQUEST_TRAILER_SIZE;

  s6350->error = 0;

  return 0;
}

char* s6350_strerror(s6350_t* s6350)
{
  char* error_string = strerror(s6350->error);
  char* message = malloc(strlen(error_string) + 32);
  sprintf(message, "%d: %s", s6350->error, error_string);
  return message;
}

int s6350_error(s6350_t* s6350)
{
  return s6350->error;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.1 Tag-it HF Command Definitions
 *    page 20: Read Transponder Details (05hex)
 */

#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_TRANSPONDER_ID	0
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_MANUFACTURER	4
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_VERSION_NUMBER0	5
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_VERSION_NUMBER1	6
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_BLOCKS		7
#define S6350_READ_TRANSPONDER_DETAILS_ANSWER_BYTES_PER_BLOCK	8

int s6350_transponder_details(s6350_t* s6350, s6350_transponder_t* transponder)
{
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  memset(transponder, '\0', sizeof(s6350_transponder_t));

  if(write_request(s6350, S6350_COMMAND_READ_TRANSPONDER_DETAILS, "", 0) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  memcpy(transponder->transponder_id,
	 buffer + S6350_READ_TRANSPONDER_DETAILS_ANSWER_TRANSPONDER_ID,
	 S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE);
  transponder->manufacturer = buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_MANUFACTURER];
  transponder->version_number =
    buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_VERSION_NUMBER0] |
    (buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_VERSION_NUMBER1] << 8);
  transponder->blocks = buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_BLOCKS];
  transponder->bytes_per_block = buffer[S6350_READ_TRANSPONDER_DETAILS_ANSWER_BYTES_PER_BLOCK];
  
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.1 Tag-it HF Command Definitions
 *    page 21: Special Read Block Command (0Fhex)
 */

#define S6350_SPECIAL_READ_BLOCK_ANSWER_TRANSPONDER_ID 0
#define S6350_SPECIAL_READ_BLOCK_ANSWER_DATA S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE

int s6350_special_read_block(s6350_t* s6350, const s6350_transponder_t* transponder, u_int8_t block_numbers, rfid_block_t* blocks)
{
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  if(write_request(s6350, S6350_COMMAND_SPECIAL_READ_BLOCK, &block_numbers, sizeof(u_int8_t)) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  if(buffer_size < S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE) {
    s6350->error = EBADMSG;
    free(buffer);
    return -1;
  }

  /*
   * Expect the returned transponder_id to be equal to the transponder_id of
   * the transponder record. It makes sure the transponder record obtained
   * from read_transponder_details matches the transponder answering the
   * special_read_block request. This is the only way to figure out the size of
   * the data block. 
   */
  if(memcmp(transponder->transponder_id,
	    buffer + S6350_SPECIAL_READ_BLOCK_ANSWER_TRANSPONDER_ID,
	    S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE)) {
    s6350->error = EAGAIN;
    free(buffer);
    return -1;
  }

  {
    int offset;
    int i;
    int data_length = transponder->bytes_per_block + sizeof(u_int8_t) * 2;
    for(i = 0, offset = S6350_SPECIAL_READ_BLOCK_ANSWER_DATA;
	offset < buffer_size;
	i++, offset += data_length) {
      memcpy(blocks[i].data, buffer + offset, transponder->bytes_per_block);
      blocks[i].security_status = buffer[offset + transponder->bytes_per_block];
      blocks[i].block_number = buffer[offset + transponder->bytes_per_block + sizeof(u_int8_t)];
    }
  }
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.1 Tag-it HF Command Definitions
 *    page 19: Read Block Command (02hex)
 */

#define S6350_READ_BLOCK_REQUEST_ADDRESS 0
#define S6350_READ_BLOCK_REQUEST_BLOCK_NUMBER 4

int s6350_read_block(s6350_t* s6350, const s6350_transponder_t* transponder, rfid_block_t* block)
{
  u_int8_t request[S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE + sizeof(u_int8_t)];
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  if(block->block_number > transponder->blocks) {
    s6350->error = E2BIG;
    return -1;
  }
  
  memcpy(request + S6350_READ_BLOCK_REQUEST_ADDRESS,
	 transponder->transponder_id,
	 S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE);
  request[S6350_READ_BLOCK_REQUEST_BLOCK_NUMBER] = block->block_number;

  if(write_request(s6350, S6350_COMMAND_COMBINE(S6350_COMMAND_READ_BLOCK, S6350_COMMAND_FLAGS_ADDRESS), request, sizeof(request)) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  memcpy(block->data, buffer, transponder->bytes_per_block);
  block->security_status = buffer[transponder->bytes_per_block];
  block->block_number = buffer[transponder->bytes_per_block + sizeof(u_int8_t)];
  
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.1 Tag-it HF Command Definitions
 *    page 19: Write Block Command (03hex)
 */

#define S6350_WRITE_BLOCK_REQUEST_ADDRESS 0
#define S6350_WRITE_BLOCK_REQUEST_BLOCK_NUMBER 4
#define S6350_WRITE_BLOCK_REQUEST_DATA 5

int s6350_write_block(s6350_t* s6350, const s6350_transponder_t* transponder, const rfid_block_t* block)
{
  u_int8_t* request;
  int request_size;
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  request_size = S6350_WRITE_BLOCK_REQUEST_DATA + transponder->bytes_per_block;
  request = (u_int8_t*)malloc(request_size);
  memcpy(request + S6350_WRITE_BLOCK_REQUEST_ADDRESS,
	 transponder->transponder_id,
	 S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE);
  request[S6350_WRITE_BLOCK_REQUEST_BLOCK_NUMBER] = block->block_number;
  memcpy(request + S6350_WRITE_BLOCK_REQUEST_DATA, block->data, transponder->bytes_per_block);

  if(write_request(s6350, S6350_COMMAND_COMBINE(S6350_COMMAND_WRITE_BLOCK, S6350_COMMAND_FLAGS_ADDRESS), request, request_size) < 0) {
    free(request);
    return -1;
  }

  free(request);

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;
  
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.2 Miscellaneous Commands
 *    page 23: Reader Version Command (F0hex)
 */

#define S6350_READER_ANSWER_VERSION_MINOR 0
#define S6350_READER_ANSWER_VERSION_MAJOR 1
#define S6350_READER_ANSWER_VERSION_TYPE 2

int s6350_reader_version(s6350_t* s6350, s6350_reader_version_t* reader_version)
{
  u_int8_t* buffer = 0;
  int buffer_size = 0;
  
  clear_error(s6350);

  if(write_request(s6350, S6350_COMMAND_READER_VERSION, "", 0) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  reader_version->major = buffer[S6350_READER_ANSWER_VERSION_MAJOR];
  reader_version->minor = buffer[S6350_READER_ANSWER_VERSION_MINOR];
  reader_version->reader_type = buffer[S6350_READER_ANSWER_VERSION_TYPE];
  
  free(buffer);

  return 0;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 3.2.2 Miscellaneous Commands
 *    page 24: RF Carrier on/off Command (F4hex)
 */

#define S6350_RF_CARRIER_ANSWER_STATUS 0

#define S6350_RF_CARRIER_SUCCESS 0x00

int s6350_rf_carrier(s6350_t* s6350, u_int8_t on_off)
{
  char request[1];
  u_int8_t* buffer = 0;
  int buffer_size = 0;

  clear_error(s6350);

  request[0] = on_off;

  if(write_request(s6350, S6350_COMMAND_RF_CARRIER, request, 1) < 0)
    return -1;

  if(read_answer(s6350, &buffer, &buffer_size) < 0)
    return -1;

  if(buffer[S6350_RF_CARRIER_ANSWER_STATUS] != S6350_RF_CARRIER_SUCCESS) {
    free(buffer);
    s6350->error = ECANCELED;
    return -1;
  }
  
  free(buffer);

  return 0;
}

#ifdef POSIX_TERMIOS

#ifndef HAVE_CFMAKERAW
#define cfmakeraw(ptr) (ptr)->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR\
					 |IGNCR|ICRNL|IXON);\
                       (ptr)->c_oflag &= ~OPOST;\
                       (ptr)->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);\
                       (ptr)->c_cflag &= ~(CSIZE|PARENB);\
                       (ptr)->c_cflag |= CS8
#endif /* HAVE_CFMAKERAW */

#ifndef HAVE_CFSETSPEED
#if defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED)
#define cfsetspeed(t,speed) \
  (cfsetispeed(t,speed) || cfsetospeed(t,speed))
#else /* defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED) */
static int cfsetspeed(struct termios *t, int speed)
{
#ifdef HAVE_TERMIOS_CSPEED
	t->c_ispeed = speed;
	t->c_ospeed = speed;
#else /* HAVE_TERMIOS_CSPEED */
	t->c_cflag |= speed;
#endif /* HAVE_TERMIOS_CSPEED */
	return 0;
}
#endif /* defined(HAVE_CFSETISPEED) && defined(HAVE_CFSETOSPEED) */
#endif /* HAVE_CFSETSPEED */

#endif /* HAVE_CFMAKERAW */

static int calcrate(int baudrate)
{
#ifdef B300
	if (baudrate == 300)
		return B300;
#endif
#ifdef B1200
	if (baudrate == 1200)
		return B1200;
#endif
#ifdef B2400
	if (baudrate == 2400)
		return B2400;
#endif
#ifdef B4800
	if (baudrate == 4800)
		return B4800;
#endif
#ifdef B9600
	if (baudrate == 9600)
		return B9600;
#endif
#ifdef B19200
	else if (baudrate == 19200)
		return B19200;
#endif
#ifdef B38400
	else if (baudrate == 38400)
		return B38400;
#endif
#ifdef B57600
	else if (baudrate == 57600)
		return B57600;
#endif
#ifdef B115200
	else if (baudrate == 115200)
		return B115200;
#endif
#ifdef B230400
	else if (baudrate == 230400)
		return B230400;
#endif
#ifdef B460800
	else if (baudrate == 460800)
		return B460800;
#endif
	else {
		return -1;	/* invalid baud rate */
	}
}

/*
 * Open serial line with default setting of 57600, 8bits, no parity.
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * 1.2 Programming Interface
 */
int s6350_init(s6350_t* s6350)
{
  clear_error(s6350);
  s6350->iso15693_command_size = 128;
  s6350->iso15693_command = malloc(s6350->iso15693_command_size);
  s6350->iso15693_command_length = 0;

  if(!s6350->device) {
    s6350->fd = -1;
    s6350->error = EINVAL;
    return -1;
  }

  if((s6350->fd = open(s6350->device, O_RDWR)) < 0) {
    s6350->error = errno;
    return -1;
  }

  if(lockf(s6350->fd, F_TLOCK, 0) < 0) {
    if(errno == EAGAIN)
      s6350->error = EEXIST;
    else
      s6350->error = errno;
    close(s6350->fd);
    return -1;
  }
  
  /*
   * Init serial line
   */
  {
	struct 	termios tcn;
	int i;

	if (!isatty(s6350->fd)) {
	  s6350->error = ENOTTY;
	  close(s6350->fd);
	  s6350->fd = -1;
	  return -1;
	}

#ifdef POSIX_TERMIOS
	/* Set the tty to raw and to the correct speed */
	tcgetattr(s6350->fd, &tcn);

	tcn.c_oflag 	= 0;
	tcn.c_iflag 	= IGNBRK | IGNPAR;
	tcn.c_cflag 	= CREAD | CLOCAL | CS8;

	(void) cfsetspeed(&tcn, calcrate(57600));

	tcn.c_lflag = NOFLSH;

	cfmakeraw(&tcn);

	for (i = 0; i < 16; i++)
		tcn.c_cc[i] = 0;

	tcn.c_cc[VMIN] 	= 1;
	tcn.c_cc[VTIME] = 0;

	tcsetattr(s6350->fd, TCSANOW, &tcn);
#else /* POSIX_TERMIOS */
	/* Set the tty to raw and to the correct speed */
	ioctl(s6350->fd, TIOCGETP, &tcn);

	data->tco = tcn;

	tcn.sg_flags = RAW;
	tcn.sg_ispeed = calcrate(57600);
	tcn.sg_ospeed = calcrate(57600);

	ioctl(s6350->fd, TIOCSETN, &tcn);
#endif /* POSIX_TERMIOS */
  }

  /*
   * Set speed
   */
  {
#ifdef POSIX_TERMIOS
	struct 	termios tcn;

	tcgetattr(s6350->fd, &tcn);

	tcn.c_cflag 	= CREAD | CLOCAL | CS8;
	(void) cfsetspeed(&tcn, calcrate(57600));

	tcsetattr(s6350->fd, TCSADRAIN, &tcn);

#else /* POSIX_TERMIOS */
	struct sgttyb tcn;

	ioctl(s6350->fd, TIOCGETP, &tcn);

	tcn.sg_ispeed 	= calcrate(57600);
	tcn.sg_ospeed 	= calcrate(57600);

	ioctl(s6350->fd, TIOCSETN, &tcn);
#endif /* POSIX_TERMIOS */
  }

  /*
   * Sanity check on transponder. If the flash memory is not loaded,
   * we won't go anywhere.
   */
  {
    s6350_reader_version_t reader_version;    
    if(s6350_reader_version(s6350, &reader_version) >= 0) {
      if(reader_version.reader_type != S6350_READER_VERSION_TYPE_LOADED) {
	s6350->error = ENOEXEC;
	return -1;
      }
    }
  }

  /*
   * Make sure carrier is turned on.
   */
  if(s6350_rf_carrier(s6350, S6350_RF_CARRIER_ON) < 0)
    return -1;
  
  return 0;
}

/*
 * Return 0 if a and b are equal, return non 0 otherwise.
 */
int s6350_transponder_cmp(s6350_transponder_t* a, s6350_transponder_t* b)
{
  return memcmp(a->transponder_id, b->transponder_id, sizeof(u_int8_t) * S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE) ||
    a->manufacturer != b->manufacturer ||
    a->version_number != b->version_number;
}

/*
 * Return 1 if the transponder object is not set, 0 otherwise.
 */
int s6350_transponder_null(s6350_transponder_t* transponder)
{
  return
    transponder->transponder_id[3] == 0 &&
    transponder->transponder_id[2] == 0 &&
    transponder->transponder_id[1] == 0 &&
    transponder->transponder_id[0] == 0 &&
    transponder->manufacturer == 0 &&
    transponder->version_number == 0;
}

void s6350_end(s6350_t* s6350)
{
  if(s6350) {
    if(s6350->fd > 0)
      close(s6350->fd);
    if(s6350->iso15693_command)
      free(s6350->iso15693_command);
  }
}

const char* s6350_version(void)
{
  return S6350_VERSION;
}

/*
 * rfid_reader driver wrappers.
 */
static int drv_init(struct rfid_reader* reader)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  s6350->device = reader->device;
  return s6350_init(s6350);
}

static void drv_end(struct rfid_reader* reader)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  return s6350_end(s6350);
}

static char* drv_strerror(struct rfid_reader* reader)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  return s6350_strerror(s6350);
}

static int drv_error(struct rfid_reader* reader)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  return s6350_error(s6350);
}

static const char* drv_version(struct rfid_reader* reader)
{
  return S6350_VERSION;
}

static void drv_verbose(struct rfid_reader* reader, int verbosity)
{
  s6350_t* s6350 = (s6350_t*)reader->private;

  s6350->verbose = verbosity;
}

static int drv_read(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  rfid_block_t* blocks;
  int block_map = 0;
  int i;

  if(block_start) {
    s6350->error = E2BIG;
    return -1;
  }

  if(number_of_blocks > transponder->blocks) {
    s6350->error = EBADF;
    return -1;
  }

  transponder->data_length = number_of_blocks * transponder->bytes_per_block;
  if(transponder->data_length > S6350_TRANSPONDER_DATA_SIZE) {
    transponder->data_length = 0;
    s6350->error = EFBIG;
    return -1;
  }
  memset(transponder->data, '\0', S6350_TRANSPONDER_DATA_SIZE + 1); /* + 1 for trailing NULL */

  blocks = (rfid_block_t*)malloc(sizeof(rfid_block_t) * number_of_blocks);

  block_map = ((1 << number_of_blocks) - 1) & 0xFF;
  
  for(i = 0; i < number_of_blocks; i++)
    blocks[i].data = &transponder->data[i * transponder->bytes_per_block];
  
  if(s6350_special_read_block(s6350, (s6350_transponder_t*)transponder->private, block_map, blocks) < 0) {
    free(blocks);
    return -1;
  }

  free(blocks);
  
  return 0;
}

static int drv_write(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  u_int8_t* pointer;
  int i;

  if(block_start + number_of_blocks > transponder->blocks) {
    s6350->error = EFBIG;
    return -1;
  }

  pointer = transponder->data + block_start * transponder->bytes_per_block;
  
  for(i = 0; i < number_of_blocks; i++) {
    int retval;
    rfid_block_t block;
    block.data = pointer;
    block.block_number = block_start + i;
    block.security_status = 0;
    
    pointer += transponder->bytes_per_block;

    while((retval = s6350_write_block(s6350, (s6350_transponder_t*)transponder->private, &block)) < 0 &&
	    reader->error_f(reader) == EAGAIN) {
      if(s6350->verbose == 1) fprintf(stderr, "!");
    }
    if(retval < 0)
      return -1;
    if(s6350->verbose == 1) fprintf(stderr, "W");

  }

  if(s6350->verbose == 1) fprintf(stderr, "\n");
  
  return 0;
#if 0
  s6350_t* s6350 = (s6350_t*)reader->private;

  if(block_start) {
    s6350->error = E2BIG;
    return -1;
  }

  if(number_of_blocks > transponder->blocks) {
    s6350->error = EBADF;
    return -1;
  }

  if(transponder->data_length > number_of_blocks * transponder->blocks) {
    s6350->error = EFBIG;
    return -1;
  }

  {
    int block_number;
    int retval;
    int offset;
    u_int8_t* block_data = (u_int8_t*)malloc(transponder->bytes_per_block * number_of_blocks);
  
    memset(block_data, '\0', transponder->bytes_per_block * number_of_blocks);
    memcpy(block_data, transponder->data, transponder->data_length);
      
    for(block_number = 0, offset = 0; block_number < number_of_blocks; block_number++, offset += transponder->bytes_per_block) {
      rfid_block_t block;

      block.data = block_data + offset;
      block.block_number = block_number;
      block.security_status = 0;

      while((retval = s6350_write_block(s6350, (s6350_transponder_t*)transponder->private, &block)) < 0 &&
	    s6350->error == EAGAIN) {
	if(s6350->verbose == 1) fprintf(stderr, "!");
      }
      if(retval < 0) {
	free(block_data);
	return -1;
      }
      if(s6350->verbose == 1) fprintf(stderr, "W");
    }
      
    free(block_data);
    if(s6350->verbose == 1) fprintf(stderr, "\n");
  }
  
  return 0;
#endif
}

static  int drv_inventory(struct rfid_reader* reader, rfid_transponder_t*** transpondersp, int* transponders_lengthp, int* transponders_sizep)
{
  rfid_transponder_t* transponder = 0;
  rfid_transponder_t** transponders = 0;

  reader->transponder_alloc_f(reader, &transponder);
  
  if(reader->transponder_present_f(reader, transponder) < 0) {
    reader->transponder_free_f(reader, transponder);
    return -1;
  }

  transponders = (rfid_transponder_t**)malloc(sizeof(rfid_transponder_t*));
  transponders[0] = transponder;

  *transpondersp = transponders;
  *transponders_lengthp = 1;
  *transponders_sizep = 1;

  return 0;
}

#define S6350_ISO15693_CONFIGURATION_BYTE_SIZE 1
/*
 * Data Coding Mode: Bit 0 of the configuration byte is used to set
 * the Data Coding Mode. When set high the reader is configured for
 * Data Coding Mode 1/4; when set low the reader is configured for
 * Data Coding Mode 1/256.
 */
#define S6350_ISO15693_DATA_CODING_MODE 0x01
/*
 * Modulation depth: Bit 4 of the configuration byte is used to set
 * Moduldation Depth. When set high the reader is configured for 100%
 * Modulation Depth, when set low the reader will operate at 10% to
 * 30% (with a 20% nominal setting) Modulation Depth.
 */
#define S6350_ISO15693_MODULATION_DEPTH 0x10

static int drv_iso15693_command(struct rfid_reader* reader, u_int8_t* data, int data_length)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  /*
   * These two flags MUST bet set otherwise the reader issues a 
   * "Packet flags invalid for command" error. The exact reason why
   * it is necessary is not documented and there are no hints in the
   * documentation. 
   * 
   * It may be the case that the following note makes this need obvious
   * for people with intimate knowledge of the product or RFID technologies
   * in general:
   *
   * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
   * 3.2.3.2 Request/Response packet format for ISO/IEC 15693-3
   * Note: The Reader RF Physical Layer is defined as:
   * Reader to Transponder : 10% - 30% (nominally set at 20%) or
   * 100% modulation (set by software) - ASK. Data Coding Mode : 1/4
   * or 1/256. Transponder to Reader: FSK / Fast Data Rate.
   *
   * One could probably infer from this note that
   * ISO15693_COMMAND_FLAGS_DATA_RATE is needed (because transponder
   * to reader is said to be in "Fast Data Rate"). The FSK acronyme (I
   * have no clue what it refers to and the documentation is silent on
   * the subject) may imply that ISO15693_COMMAND_FLAGS_SUB_CARRIER is
   * also needed.
   * 
   * In the best case the documentation is obscure, in the worst case
   * it's completly missing this important requirement.
   * 
   */
  data[ISO15693_FLAGS_INDEX] |= ISO15693_COMMAND_FLAGS_SUB_CARRIER|ISO15693_COMMAND_FLAGS_DATA_RATE;
  /*
   * Record the command sent for later use by drv_iso15693_response.
   * It must be done before any s6350 specific frame re-mastering so
   * that the frame saved matches ISO-15693 specifications.
   */
  {
    while(s6350->iso15693_command_size < data_length) {
      s6350->iso15693_command_size *= 2;
      s6350->iso15693_command = realloc(s6350->iso15693_command, s6350->iso15693_command_size);
    }
    
    memcpy(s6350->iso15693_command, data, data_length);
    s6350->iso15693_command_length = data_length;
  }
  
  switch(data[ISO15693_COMMAND_INDEX]) {
  /*
   * List all unsupported commands for which an emulation is provided
   * by the drv_iso15693_response function. Do nothing at this point.
   */
  case ISO15693_COMMAND_GET_SYSTEM_INFORMATION:
    break;

  default:
    /*
     * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
     * 3.2.3.2 Request/Response Packet Format for ISO/IEC 15693-3 (page
     * 28) "SOF, CRC16 and EOF fields must not be included in the
     * message data packet" Note that SOF & EOF are not data fields (0/1
     * bits) but a signal specification defined in ISO/IEC 15693-2.
     */
    data_length -= ISO15693_COMMAND_TRAILER_SIZE;
    /*
     * Shift request one byte to leave room for configuration byte.
     * This will override the CRC16 but we don't want it anyhow.
     */
    memmove(data + S6350_ISO15693_CONFIGURATION_BYTE_SIZE, data, data_length);
    /*
     * Not mandatory but significantly improves the reliability of the
     * communications. If not set, the reader randomly fails to see a
     * transponder, even if it's the only one available and at the
     * closest possible distance from the antena.
     */
    data[0] = S6350_ISO15693_DATA_CODING_MODE|S6350_ISO15693_MODULATION_DEPTH;
    data_length += S6350_ISO15693_CONFIGURATION_BYTE_SIZE;

    if(write_request(s6350, S6350_COMMAND_ISO15693, data, data_length) < 0)
      return -1;
    break;
  }
  
  return 0;
}

static int emulate_iso15693_get_system_information(struct rfid_reader* reader, u_int8_t* command, int command_length, u_int8_t** datap, int* data_lengthp)
{
  int retval = -1;
  u_int8_t* data = malloc(ISO15693_INFO_FLAGS_SIZE +
			  ISO15693_UID_SIZE +
			  ISO15693_DSFID_SIZE +
			  ISO15693_AFI_SIZE +
			  ISO15693_VICC_MEMORY_SIZE_SIZE +
			  ISO15693_IC_REFERENCE_SIZE +
			  ISO15693_RESPONSE_TRAILER_SIZE
			  );
  int data_length = 0;

  rfid_transponder_t* transponder = 0;
  rfid_reader_t* iso15693_reader = reader->parent;
  iso15693_t* iso15693 = (iso15693_t*)iso15693_reader->private;
  iso15693_inventory_result_t* inventory_results = 0;

  if(iso15693_reader->transponder_alloc_f(reader, &transponder))
    goto finish;

  /*
   * CAVEAT: the DSFID is not filled. Should send an INVENTORY in
   * both cases.
   */
  if(command[ISO15693_FLAGS_INDEX] & ISO15693_COMMAND_FLAGS_ADDRESS) {
    /*
     * We target a specific transponder, therefore the uid is provided to
     * us in the command.
     */
    iso15693_transponder_uid_set(transponder, command + ISO15693_DATA_INDEX);
  } else {
    /*
     * We target the first available transponder, look for it.
     */
    int results_length = 0;
    iso15693_inventory_t inventory;

    memset(&inventory, '\0', sizeof(iso15693_inventory_t));

    if(iso15693_inventory(iso15693, &inventory, &inventory_results, &results_length) < 0)
      goto finish;

    if(results_length < 1)
      goto finish;

    iso15693_transponder_uid_set(transponder, inventory_results[0].uid);
    iso15693_transponder_dsfid_set(transponder, inventory_results[0].dsfid);
  }

  /*
   * Fill transponder->bytes_per_block
   *
   * Read the first block : the bytes_per_block will be fixed
   * according to the size of the returned data.
   */
  {
    rfid_block_t block;

    block.data = transponder->data;
    block.block_number = 0;
    block.security_status = 0;

    transponder->blocks = ISO15693_BLOCK_COUNT_MAX;
    transponder->bytes_per_block = ISO15693_BLOCK_SIZE_MAX;
  
    if(iso15693_read_single_block(iso15693, transponder, &block) < 0)
      goto finish;
  }

  /*
   * Fill transponder->blocks
   *
   * This moronic ISO based API provides no way to figure out the
   * actual number of blocks the transponder can handle. Be very
   * conservative.  One could also map the product identification to a
   * size, statically, to accurately compensate this lack of
   * information.
   */
  transponder->blocks = 16;

  /*
   * Fill VICC memory size.
   */
  iso15693_transponder_vicc_blocks_set(transponder, transponder->blocks);
  iso15693_transponder_vicc_bytes_per_block_set(transponder, transponder->bytes_per_block);

  /*
   * NO SUPPORT for DSFID, AFI, IC REF for now.
   */

  /*
   * Hand made answer frame, based on the data collected so far.
   */

  {
    u_int8_t* pointer = data;
    u_int16_t vicc_memory;

    iso15693_transponder_vicc_memory_get(transponder, vicc_memory);

    *pointer = 0;
    pointer += ISO15693_FLAGS_SIZE;

    *pointer = ISO15693_INFO_FLAGS_VICC_MEMORY_SIZE & 0xFF;
    pointer += ISO15693_INFO_FLAGS_SIZE;

    iso15693_transponder_uid_get(transponder, pointer);
    pointer += ISO15693_UID_SIZE;

    pointer[0] = vicc_memory & 0xFF;
    pointer[1] = (vicc_memory >> 8) & 0xFF;
    pointer += ISO15693_VICC_MEMORY_SIZE_SIZE;

    data_length = pointer - data;
  }

  *datap = data;
  *data_lengthp = data_length;
  
  retval = 0;

 finish:
  if(inventory_results)
    free(inventory_results);

  if(transponder)
    reader->transponder_free_f(reader, transponder);

  if(retval < 0) {
    if(data)
      free(data);

    *datap = 0;
    *data_lengthp = 0;

    /*
     * Whatever went wrong at this stage results in "No transponder found".
     * Since we are emulating a ISO-15693 primitive, returning other
     * codes would only lead to confusion.
     */
    iso15693->error = ENOENT;
  }
  
  return retval;
}

/*
 * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
 * The Error Byte (ISO Response Data Byte 0), page 29
 */
#define S6350_ISO15693_ERROR_TRANSPONDER_NOT_FOUND	0x01
#define S6350_ISO15693_ERROR_COMMAND_NOT_SUPPORTED	0x02
#define S6350_ISO15693_ERROR_INVALID_FLAGS		0x04

static int drv_iso15693_response(struct rfid_reader* reader, u_int8_t** datap, int* data_lengthp)
{
#if ISO15693_RESPONSE_OVERHEAD > S6350_REQUEST_OVERHEAD
  fprintf(stderr, "PANIC: ISO15693_RESPONSE_OVERHEAD > S6350_REQUEST_OVERHEAD");
  abort();
#else /* ISO15693_RESPONSE_OVERHEAD > S6350_REQUEST_OVERHEAD */
  s6350_t* s6350 = (s6350_t*)reader->private;
  u_int8_t* data;
  int data_length;

  /*
   * Each unsupported command can be emulated at this point.
   */
  switch(s6350->iso15693_command[ISO15693_COMMAND_INDEX]) {

  case ISO15693_COMMAND_GET_SYSTEM_INFORMATION:
    if(emulate_iso15693_get_system_information(reader, s6350->iso15693_command, s6350->iso15693_command_length, &data, &data_length) < 0)
      return -1;
    break;

  default:
    if(read_answer(s6350, &data, &data_length) < 0)
      return -1;
  }

  /*
   * With the exception of INVENTORY, all responses start with
   * a flags field holding the possible error status. 
   */
  if(s6350->iso15693_command[ISO15693_COMMAND_INDEX] != ISO15693_COMMAND_INVENTORY &&
     data[ISO15693_FLAGS_INDEX] & ISO15693_RESPONSE_FLAG_ERROR) {
    /*
     * If an error occurs map proprietary error code to ISO-15693
     * error codes or extended error codes defined in iso15693.h.
     */
    switch(data[ISO15693_ERROR_INDEX]) {
    case S6350_ISO15693_ERROR_TRANSPONDER_NOT_FOUND:
      data[ISO15693_ERROR_INDEX] = ISO15693_ERROR_TRANSPONDER_NOT_FOUND;
      break;
    case S6350_ISO15693_ERROR_COMMAND_NOT_SUPPORTED:
      data[ISO15693_ERROR_INDEX] = ISO15693_ERROR_NOT_SUPPORTED;
      break;
    case S6350_ISO15693_ERROR_INVALID_FLAGS:
      data[ISO15693_ERROR_INDEX] = ISO15693_ERROR_INVALID_FLAGS;
      break;
    }
  } else {

    /*
     * If the command was successfull, per command re-mastering of the
     * response frame to conform to ISO-15693 or emulation of unsupported
     * commands, where possible.
     */
    switch(s6350->iso15693_command[ISO15693_COMMAND_INDEX]) {
    case ISO15693_COMMAND_INVENTORY:
      /*
       * Trash valid data flags and collision flags.
       * And build a fake flags field for consistency.
       * http://www.ti.com/tiris/docs/manuals/refmanuals/RI-STU-TRDCrefGuide.pdf
       * 3.2.3.3 Mandatory Commands, page 30
       */
#define S6350_ISO15693_VALID_DATA_FLAGS_SIZE		2
#define S6350_ISO15693_COLLISION_FLAGS_SIZE		2
      {
	int length =
	  S6350_ISO15693_VALID_DATA_FLAGS_SIZE
	  + S6350_ISO15693_COLLISION_FLAGS_SIZE;

	data_length -= length;
	memmove(data + ISO15693_FLAGS_SIZE, data + length, data_length);

	/*
	 * Apparently the S6350 assumes an INVENTORY can't
	 * fail. Awesome.
	 */
	data[ISO15693_FLAGS_INDEX] = 0x00;
	data_length += ISO15693_FLAGS_SIZE;
      }
      break;
    }
  }
  
  /*
   * S6350 does not return a proper ISO15693 frame (truncated
   * CRC trailer). Restore it for proper processing by caller.
   */
  data_length += ISO15693_RESPONSE_TRAILER_SIZE;
  iso15693_crc16_set(data, data_length);

  *datap = data;
  *data_lengthp = data_length;

  return 0;
#endif /* ISO15693_RESPONSE_OVERHEAD > S6350_REQUEST_OVERHEAD */
}

static char* drv_transponder_describe(struct rfid_reader* reader, const rfid_transponder_t* transponder)
{
  char* str = malloc(512);
  s6350_transponder_t* s6350_transponder = (s6350_transponder_t*)transponder->private;

  sprintf(str, "\
transponder_id = 0x%02x%02x%02x%02x\n\
manufacturer = 0x%02x\n\
version_number = 0x%04x\n\
blocks = %d\n\
bytes_per_block = %d\n\
",
	  s6350_transponder->transponder_id[3],
	  s6350_transponder->transponder_id[2],
	  s6350_transponder->transponder_id[1],
	  s6350_transponder->transponder_id[0],
	  s6350_transponder->manufacturer,
	  s6350_transponder->version_number,
	  s6350_transponder->blocks,
	  s6350_transponder->bytes_per_block
	  );

  return str;
}

static int drv_transponder_cmp(struct rfid_reader* reader, rfid_transponder_t* transponder_a, rfid_transponder_t* transponder_b)
{
  return s6350_transponder_cmp((s6350_transponder_t*)transponder_a->private,
			       (s6350_transponder_t*)transponder_b->private);
}

static int drv_transponder_null(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  return s6350_transponder_null((s6350_transponder_t*)transponder->private);
}

static int drv_transponder_present(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  s6350_transponder_t* s6350_transponder = (s6350_transponder_t*)transponder->private;
  if(s6350_transponder_details(s6350, s6350_transponder) < 0)
    return -1;

  transponder->blocks = s6350_transponder->blocks;
  transponder->bytes_per_block = s6350_transponder->bytes_per_block;

  return 0;
}

static int drv_transponder_alloc(struct rfid_reader* reader, rfid_transponder_t** transponderp)
{
  rfid_transponder_t* transponder = (rfid_transponder_t*)malloc(sizeof(rfid_transponder_t));

  memset(transponder, '\0', sizeof(rfid_transponder_t));
  transponder->private = (s6350_transponder_t*)malloc(sizeof(s6350_transponder_t));
  memset(transponder->private, '\0', sizeof(s6350_transponder_t));

  *transponderp = transponder;

  return 0;
}

static int drv_transponder_free(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  free(transponder->private);
  free(transponder);

  return 0;
}

static int drv_transponder_clear(struct rfid_reader* reader, rfid_transponder_t* transponder)
{
  s6350_transponder_t* s6350 = (s6350_transponder_t*)transponder->private;
  memset(s6350, '\0', sizeof(s6350_transponder_t));
  memset(transponder, '\0', sizeof(rfid_transponder_t));
  transponder->private = s6350;

  return 0;
}

static int drv_transponder_copy(struct rfid_reader* reader, rfid_transponder_t* transponder_a, rfid_transponder_t* transponder_b)
{
  s6350_transponder_t* s6350_a = (s6350_transponder_t*)transponder_a->private;
  s6350_transponder_t* s6350_b = (s6350_transponder_t*)transponder_b->private;

  memcpy(s6350_a, s6350_b, sizeof(s6350_transponder_t));
  memcpy(transponder_a, transponder_b, sizeof(rfid_transponder_t));
  transponder_a->private = s6350_a;

  return 0;
}

static int drv_transponder_id_get(struct rfid_reader* reader, rfid_transponder_t* transponder, char** idp)
{
  s6350_transponder_t* s6350_transponder = (s6350_transponder_t*)transponder->private;

  char* id = malloc(32);

  sprintf(id, "%02x.%02x.%02x.%02x.%02x.%04x",
	  s6350_transponder->transponder_id[3],
	  s6350_transponder->transponder_id[2],
	  s6350_transponder->transponder_id[1],
	  s6350_transponder->transponder_id[0],
	  s6350_transponder->manufacturer,
	  s6350_transponder->version_number);

  *idp = id;

  return 0;
}

static int drv_transponder_id_set(struct rfid_reader* reader, rfid_transponder_t* transponder, char* id)
{
  s6350_t* s6350 = (s6350_t*)reader->private;
  s6350_transponder_t* s6350_transponder = (s6350_transponder_t*)transponder->private;
  unsigned int id0, id1, id2, id3, manufacturer, version_number;

  if(sscanf(id, "%x.%x.%x.%x.%x.%x",
	    &id3,
	    &id2,
	    &id1,
	    &id0,
	    &manufacturer,
	    &version_number) != 6) {
    s6350->error = ECANCELED;
    return -1;
  }

  s6350_transponder->transponder_id[3] = id3 & 0xFF;
  s6350_transponder->transponder_id[2] = id2 & 0xFF;
  s6350_transponder->transponder_id[1] = id1 & 0xFF;
  s6350_transponder->transponder_id[0] = id0 & 0xFF;
  s6350_transponder->manufacturer = manufacturer & 0xFF;
  s6350_transponder->version_number = version_number & 0xFF; 

  return 0;
}

static int drv_free(rfid_reader_t* reader)
{
  free(reader->private);
  free(reader);
  return 0;
}

static int drv_alloc(rfid_reader_t** readerp)
{
  rfid_reader_t* reader = (rfid_reader_t*)malloc(sizeof(rfid_reader_t));
  s6350_t* s6350 = (s6350_t*)malloc(sizeof(s6350_t));
  memset(s6350, '\0', sizeof(s6350_t));

  reader->private = s6350;

  reader->alloc_f = drv_alloc;
  reader->free_f = drv_free;

  reader->init_f = drv_init;
  reader->end_f = drv_end;

  reader->strerror_f = drv_strerror;
  reader->error_f = drv_error;
  reader->version_f = drv_version;
  reader->verbose_f = drv_verbose;

  reader->read_f = drv_read;
  reader->write_f = drv_write;
  reader->inventory_f = drv_inventory;
  reader->iso15693_command_f = drv_iso15693_command;
  reader->iso15693_response_f = drv_iso15693_response;

  reader->transponder_describe_f = drv_transponder_describe;
  reader->transponder_cmp_f = drv_transponder_cmp;
  reader->transponder_null_f = drv_transponder_null;
  reader->transponder_present_f = drv_transponder_present;
  reader->transponder_alloc_f = drv_transponder_alloc;
  reader->transponder_free_f = drv_transponder_free;
  reader->transponder_clear_f = drv_transponder_clear;
  reader->transponder_copy_f = drv_transponder_copy;
  reader->transponder_id_get_f = drv_transponder_id_get;
  reader->transponder_id_set_f = drv_transponder_id_set;

  *readerp = reader;

  return 0;
}

int s6350_reader_alloc(rfid_reader_t** reader)
{
  return drv_alloc(reader);
}
