/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

/*
 * For a detailed description of the interface see s6350(3)
 */

#ifndef _S6350_H_
#define _S6350_H_

#include <sys/types.h>
#include <errno.h>

#include <rfid_reader.h>

/*
 * Handler holding context for a specific RFID device and used by all
 * s6350 function.
 */
typedef struct {
  int fd;			/* File descriptor that must be a tty. */
  int error;			/* Last error, see errno(3) and
				   s6350(3) for meaning. */
  char* device;			/* Path name of the device used to
				   acquire fd. */
  u_int8_t* iso15693_command;    /* Last ISO-15693 command sent. */
  int iso15693_command_length;    /* Length of command stored in iso15693_command_length. */
  int iso15693_command_size;    /* Total number of allocated bytes. */
  int verbose;
} s6350_t;

#define S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE 4

/*
 * Identity and data layout of a RFID transponder.
 */
typedef struct {
  /* Identity of the transponder, least significant bytes first. */
  u_int8_t transponder_id[S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE];
  u_int8_t manufacturer;
  u_int16_t version_number;
  u_int8_t blocks;		/* Number of blocks in the
				   transponder, each of them has
				   bytes_per_block bytes. */
  u_int8_t bytes_per_block;	/* Number of bytes in each block. */
} s6350_transponder_t;

/*
 * Identity and status of the RFID reader.
 */
typedef struct {
  u_int8_t major;		/* Major firmware version number. */
  u_int8_t minor;		/* Minor firmware version number. */

  /* Indicates that the reader has been successfully loaded with the
     application firmware (major.minor) version number. */
#define S6350_READER_VERSION_TYPE_LOADED 0x07

  /* Indicates that the reader has not been loaded with application
     firmware, but does have the boot-loader firmware in place with
     which to download the appropriate application firmware. */
#define S6350_READER_VERSION_TYPE_NOT_LOADED 0x00

  u_int8_t reader_type;		/* Status of the RFID reader. */
} s6350_reader_version_t;

int s6350_reader_alloc(rfid_reader_t** reader);

/*
 * Opens s6350->device that must be a tty and initializes it in raw
 * mode. Probe the presence of a RFID reader and sets the carrier on.
 */
int s6350_init(s6350_t* s6350);

/*
 * Closes the s6350->fd file descriptor.
 */
void s6350_end(s6350_t* s6350);

/*
 * Returns the RFID transponder description in <transponder>, if one
 * is present.
 */
int s6350_transponder_details(s6350_t* s6350, s6350_transponder_t* transponder);

/*
 * Read RFID transponder blocks according to the <block_numbers> bitmap. Each is
 * stored in the correponding blocks[block_number]. The <transponder> structure is 
 * used to make sure the blocks read match the <transponder> identity. If it is
 * not the case, the read fails.
 */
int s6350_special_read_block(s6350_t* s6350, const s6350_transponder_t* transponder, u_int8_t block_numbers, rfid_block_t* blocks);

/*
 * Read a single <block> from the <transponder>. The block->block_number specifies
 * the block number to read. The <transponder> is used to figure out the size
 * of the data (from transponder->bytes_per_block).
 */
int s6350_read_block(s6350_t* s6350, const s6350_transponder_t* transponder, rfid_block_t* block);

/*
 * Write a single <block> to the <transponder>. The block->block_number specifies
 * the block number to read. The <transponder> is used to figure out the size
 * of the data (from transponder->bytes_per_block).
 */
int s6350_write_block(s6350_t* s6350, const s6350_transponder_t* transponder, const rfid_block_t* block);

/*
 * Read the <reader_version> of the RFID reader.
 */
int s6350_reader_version(s6350_t* s6350, s6350_reader_version_t* reader_version);

/*
 * Turn the carrier of the RFID reader on (if on_off is 1) or off (if on_off is 0).
 */
int s6350_rf_carrier(s6350_t* s6350, u_int8_t on_off);

/*
 * Send a ISO-15693 command.
 */
int s6350_iso15693_command(s6350_t* s6350, u_int8_t* data, int data_length);
/*
 * Read the response to a ISO-15693 command.
 */
int s6350_iso15693_response(s6350_t* s6350, u_int8_t** bufferp, int* buffer_sizep);

/*
 * Error codes from the RFID reader.
 */
#define S6350_ERROR_TRANSPONDER_NOT_FOUND			ENOENT
#define S6350_ERROR_COMMAND_NOT_SUPPORTED			EPERM
#define S6350_ERROR_PACKET_BCC_INVALID				EPIPE
#define S6350_ERROR_PACKET_FLAGS_INVALID_FOR_COMMAND		ERANGE
#define S6350_ERROR_GENERAL_WRITE_FAILURE			ENXIO
#define S6350_ERROR_WRITE_FAILURE_DUE_TO_LOCK_BLOCK		ENOLCK
#define S6350_ERROR_TRANSPONDER_DOES_NOT_SUPPORT_FUNCTION	ENOTSUP
#define S6350_ERROR_UNDEFINED_ERROR				ESRCH
#define S6350_ERROR_INVALID					ENOSYS

int s6350_transponder_cmp(s6350_transponder_t* a, s6350_transponder_t* b);
int s6350_transponder_null(s6350_transponder_t* transponder);

/*
 * Convert RFID reader error codes into strings.
 */
char* s6350_strerror(s6350_t* s6350);
/*
 * Return the latest error number. Positive values are errno(3)
 * values, negative values are S6350 reader specific error numbers.
 */
int s6350_error(s6350_t* s6350);

/*
 * Return the version of the library as a string (1.0.0 for instance).
 */
const char* s6350_version(void);

#endif /* _S6350_H_ */
