/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <ltdl.h>

#include <rfid_error.h>
#include <rfid_reader.h>

typedef int (*rfid_reader_allocator)(rfid_reader_t** reader);

#ifndef DRIVERDIR
#define DRIVERDIR "/usr/lib/rfid"
#endif /* DRIVERDIR */

int rfid_alloc(const char* driver_stack, rfid_reader_t** reader, int verbose)
{
  rfid_reader_allocator alloc_function = 0;
  char* filename = 0;
  char* function = 0;
  char* driver = 0;
  int error = -1;

  *reader = 0;

  if(!driver_stack) {
    errno = RFID_ERROR_DYNLOAD_NO_DRIVER_STACK;
    goto err;
  }

  {
    char* slash;
    driver = strdup(driver_stack);
    if((slash = strchr(driver, '/')))
      *slash = '\0';
  }

  if(lt_dlinit() > 0) {
    errno = RFID_ERROR_DYNLOAD_DLINIT;
    if(verbose) { const char* error = lt_dlerror(); fprintf(stderr, "lt_dlinit: %s\n", (error ? error : "")); }
    goto err;
  }

  if(!lt_dlgetsearchpath() || !strstr(lt_dlgetsearchpath(), DRIVERDIR)) {
    if(lt_dladdsearchdir(DRIVERDIR)) {
      errno = RFID_ERROR_DYNLOAD_DLADDSEARCHDIR;
      if(verbose) { const char* error = lt_dlerror(); fprintf(stderr, "lt_dladdsearchdir(%s): %s\n", DRIVERDIR, (error ? error : "")); }
      goto err;
    }
  }
    
  {
    lt_dlhandle handle;
    lt_ptr symbol;
    filename = (char*)malloc(strlen(driver) + 1 + 128);
    sprintf(filename, "lib%s.la", driver);
    handle = lt_dlopen(filename);
    if(!handle) {
      errno = RFID_ERROR_DYNLOAD_DLOPEN;
      if(verbose) { const char* error = lt_dlerror(); fprintf(stderr, "lt_dlopen(%s): %s\n", filename, (error ? error : "")); }
      goto err;
    }

    function = (char*)malloc(strlen(driver) + 1 + 128);
    sprintf(function, "%s_reader_alloc", driver);
    symbol = lt_dlsym(handle, function);
    if(!symbol) {
      errno = RFID_ERROR_DYNLOAD_ENTRY_POINT;
      if(verbose) { const char* error = lt_dlerror(); fprintf(stderr, "lt_dlsym(%s): %s\n", function, (error ? error : "")); }
      goto err;
    }

    alloc_function = (rfid_reader_allocator)symbol;

    alloc_function(reader);

    if(*reader == 0) {
      errno = RFID_ERROR_DYNLOAD_ALLOC;
      goto err;
    }
  }

  error = 0;

 err:
  if(function) free(function);
  if(filename) free(filename);
  if(driver) free(driver);
  return error;
}

int rfid_free(rfid_reader_t* reader)
{
  /* Could unload the plugins. */
  return reader->free_f(reader);
}

void rfid_dump_frame(u_int8_t* frame, int frame_length)
{
  int i;
  for(i = 0; i < frame_length; i++)
    fprintf(stderr, "0x%02x ", frame[i]);
}
