/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <memory.h>

#include "rfid.h"

static int verbose = 0;

static void report_error(rfid_t* rfid, const char* message)
{
  char* str = rfid_strerror(rfid);
  fprintf(stderr, "%s: %s\n", message, str);
  free(str);
}

static rfid_t rfid;

static RETSIGTYPE timeout_init(int sig)
{
  if(verbose) {
    fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to establish communication with RFID reader.\n\
FAILURE: Make sure a RFID reader is plugged in and turned on.\n\
\n\
");
  }
  exit(-1);
}

static RETSIGTYPE timeout_read(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to read information from a transponder.\n\
FAILURE: Make sure a transponder is placed on the RFID reader.\n\
\n\
");
  exit(-1);
}

static RETSIGTYPE timeout_write(int sig)
{
  if(verbose) {
    fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to write information to a transponder.\n\
FAILURE: Make sure a transponder is placed on the RFID reader.\n\
\n\
");
  } else {
    fprintf(stderr, "SKIP\tfunctions tests (no RFID transponder on the RFID reader).\n");
  }
  exit(0);
}

static char* test_string = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

static int transponder_callback(rfid_t* rfid, int event, const rfid_transponder_t* transponder, void* data)
{
  int* count = (int*)data;
  char* transponder_description = rfid->reader->transponder_describe_f(rfid->reader, transponder);

  (*count)--;

  if(rfid->verbose) {
    fprintf(stderr, "\
========================================\n\
%s",
	    transponder_description);
  }

  free(transponder_description);
  
  if(rfid->verbose) {
    int i;
    for(i = 0; i < transponder->data_length; i++)
      fprintf(stderr, "%c(0x%02x)/", transponder->data[i], transponder->data[i]);
    fprintf(stderr, "\n");
  }

  if(transponder->data_length < strlen(test_string) ||
     memcmp(transponder->data, test_string, strlen(test_string))) {
    fprintf(stderr, "FAILURE: expected %s\n", test_string);
    return RFID_CALLBACK_ERROR;
  }

  if(rfid->verbose)
    fprintf(stderr, "=> %d more read attempt to go\n", *count);

  return *count > 0 ? RFID_CALLBACK_OK : RFID_CALLBACK_END;
}

int main(int argc, char** argv)
{
  int timeout = 5;
  char* driver = RFID_DRIVER_GUESS;
  char* device = RFID_DEVICE_GUESS;

  if(argc > 1)
    driver = argv[1];
  if(argc > 2)
    device = argv[2];

  rfid.verbose = verbose;

  signal(SIGALRM, timeout_init);

  alarm(timeout);
  if(rfid_init(&rfid, driver, device) < 0) {
    report_error(&rfid, "rfid_init");
    return -1;
  }
  alarm(0);

  rfid.reader->verbose_f(rfid.reader, rfid.verbose);

  fprintf(stderr, "Testing rfid(3) with %s using driver %s\n", rfid_version(), rfid.reader->version_f(rfid.reader));

  signal(SIGALRM, timeout_write);

  alarm(timeout);
  {
    rfid_transponder_t* transponder = rfid_transponder_alloc(&rfid);
    char* transponder_description = 0;
    
    memcpy(transponder->data, test_string, strlen(test_string));
    transponder->data_length = strlen(test_string);
    if(rfid_write(&rfid, transponder) < 0) {
      report_error(&rfid, "rfid_write");
      rfid_transponder_free(&rfid, transponder);
      return -1;
    }

    transponder_description = rfid_transponder_describe(&rfid, transponder);

    if(rfid.verbose) {
      fprintf(stderr, "\
========================================\n\
Wrote string %s to transponder:\n\
%s",
	    test_string,
	    transponder_description);
    }
    free(transponder_description);
    rfid_transponder_free(&rfid, transponder);
  }
  fprintf(stderr, "OK\trfid_write (non-addressed)\n");
  alarm(0);

  alarm(timeout);
  {
    rfid_transponder_t* transponder = rfid_transponder_alloc(&rfid);
    /* Fake manufacturer, unlikely to match an existing one to trigger failure. */
    rfid_transponder_id_set(&rfid, transponder, "1.2.3.4.5.6.7.8");
    if(rfid_write(&rfid, transponder) >= 0 ||
       rfid.error != RFID_ERROR_TRANSPONDER_NOT_FOUND) {
      report_error(&rfid, "rfid_write (RFID_ERROR_TRANSPONDER_NOT_FOUND was expected)");
      rfid_transponder_free(&rfid, transponder);
      return -1;
    }
    rfid_transponder_free(&rfid, transponder);
  }
  fprintf(stderr, "OK\trfid_write (addressed)\n");
  alarm(0);
  
  signal(SIGALRM, timeout_read);

  alarm(timeout);
  {
    int count = 3;
    if(rfid_event_loop(&rfid, RFID_EVENT_NEW|RFID_EVENT_AGAIN, transponder_callback, 0, &count) < 0) {
      report_error(&rfid, "rfid_event_loop");
      return -1;
    }
  }
  fprintf(stderr, "OK\trfid_event_loop\n");
  alarm(0);

  alarm(timeout);
  {
    int count = 1;
    if(rfid_events_context_init(&rfid, RFID_EVENT_NEW|RFID_EVENT_AGAIN, transponder_callback, 0, &count) < 0) {
      report_error(&rfid, "rfid_events_context_init");
      return -1;
    }

    if(rfid_event_poll(&rfid) < 0) {
      report_error(&rfid, "rfid_event_poll");
      return -1;
    }

    if(rfid_events_context_end(&rfid) < 0) {
      report_error(&rfid, "rfid_events_context_end");
      return -1;
    }
  }
  fprintf(stderr, "OK\trfid_event_poll\n");
  alarm(0);
  
  if(rfid_fd(&rfid) < 0) {
    report_error(&rfid, "rfid_fd");
    return -1;
  }
    
  rfid_end(&rfid);
  
  return 0;
}
