; Copyright (c) 2004, Jonah Nathaniel Beckford
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without
; modification, are permitted provided that the following conditions
; are met:
;
;   Redistributions of source code must retain the above copyright
;   notice, this list of conditions and the following disclaimer.
;
;   Redistributions in binary form must reproduce the above copyright
;   notice, this list of conditions and the following disclaimer in
;   the documentation and/or other materials provided with the
;   distribution.
;
;   Neither the name of the author nor the names of its contributors
;   may be used to endorse or promote products derived from this
;   software without specific prior written permission.
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
; "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
; LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
; FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
; COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
; INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
; (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
; STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
; ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
; OF THE POSSIBILITY OF SUCH DAMAGE.
;
; jonah@usermail.com

(require-extension srfi-40)
(cond-expand
  [srfi-40
    (print "Using rgraph stream functions; found 'srfi-40 extension")]
  [else
    (print "Not using rgraph stream functions because no 'srfi-40 extension")])
(newline)

(require-for-syntax 'rgraph)
(require 'rgraph-base)
;(load "rgraph")
;(load "rgraph-base")


;; Adapted from
;; http://www.boost.org/libs/graph/doc/file_dependency_example.html
;; using Scheme style.
(define used-by
  (list 
    (cons 'dax_h 'foo_cpp) (cons 'dax_h 'bar_cpp) (cons 'dax_h 'yow_h)
    (cons 'yow_h 'bar_cpp) (cons 'yow_h 'zag_cpp)
    (cons 'boz_h 'bar_cpp) (cons 'boz_h 'zig_cpp) (cons 'boz_h 'zag_cpp)
    (cons 'zow_h 'foo_cpp) 
    (cons 'foo_cpp 'foo_o) 
    (cons 'foo_o 'libfoobar_a) 
    (cons 'bar_cpp 'bar_o) 
    (cons 'bar_o 'libfoobar_a) 
    (cons 'libfoobar_a 'libzigzag_a) 
    (cons 'zig_cpp 'zig_o) 
    (cons 'zig_o 'libzigzag_a) 
    (cons 'zag_cpp 'zag_o) 
    (cons 'zag_o 'libzigzag_a) 
    (cons 'libzigzag_a 'killerapp)))

(define-adjacency-list myg1
  (fill-graph! 
   depth-first-search
   topological-sort 
   partition-fidmat
   )
  (vl-vector) (vertex-name vertex-color)
  (el-hash) (edge-weight)
  #t #t)

(define g1 (make-myg1))
(myg1-fill-graph! g1 used-by set-myg1-vertex-name!)

(print "Vertex order [list]:")
(for-each
 (lambda (v) (print (myg1-vertex-name g1 v)))
 (myg1-vertices g1))

(print "Vertex order [stream]:")
(stream-for-each
 (lambda (v) (print (myg1-vertex-name g1 v)))
 (myg1-vertices* g1))

;; topo sort
(print "Topological sort [list]:")
(for-each
 (lambda (v) 
   (print (myg1-vertex-name g1 v)))
  (myg1-topological-sort g1))

(print "Topological sort [stream]:")
(stream-for-each
 (lambda (v) 
   (print (myg1-vertex-name g1 v)))
  (myg1-topological-sort* g1))

(print "Partition Fiduccia-Mattheyses:")
(let dummy ()
  (define g g1)
  (define partition-map 
    (prop-external-hash (lambda (v1 v2) (myg1-vertex-eq? g v1 v2))))
  (define partition (car partition-map))
  (define set-partition! (cdr partition-map))
  (define gain
    (lambda (g u)
      (let* ([u-nbs (myg1-neighbours g1 u)]
	     [total (length u-nbs)]
	     [p1 (partition g u)]
	     ;; what is edge cost now?
	     [now (rgraph-count
		   (lambda (u-nb) (not (eq? p1 (partition g (car u-nb)))))
		   u-nbs)]
	     ;; what is edge cost later, when moved?
	     [later (- total now)])
	;; if cell is moved and causes increase in solution cost
	;; (that is, later > now), then the gain is negative
	(- now later))))
  (define (cost)
    (let ([t 0])
      ;; count all edges from one partition to another (the edge
      ;; cut). note that this will double count if undirected edges,
      ;; since both edges A->B and B->A will be counted.
      (for-each
       (lambda (u)
	 (let ([p1 (partition g u)])
	   (set! t
		 (+ t (rgraph-count
		       (lambda (u-nb) (not (eq? p1 (partition g (car u-nb)))))
		       (myg1-neighbours g u))))))
       (myg1-vertices g))
      (quotient t 2)))
  (define (balance weight n#f n#t)
    (let ([b 0])
      (cond
       [(and n#f n#t)
	(set! b (- (* (car weight) n#t) (* (cdr weight) n#f)))]
       [else
	(set! n#t 0)
	(set! n#f 0)
	(for-each
	 (lambda (c)
	   (set! n#t (add1 n#t))
	   (set! b
		 (+ b
		    (if (partition g c)
			(car weight)
			(- (cdr weight))))))
	 (myg1-vertices g))])
      (quotient (* (abs b) 100) 
		(* (max (car weight) (cdr weight))
		   (+ n#t n#f)))))
  (define working-graph (make-myg1))
  ;; we could use prop-external-vector, but would only work if graph's
  ;; vertex list was vl-vector
  (define working-prop-map (prop-external-hash eq? (myg1-num-vertices g)))
  (define (output part)
    (map
     (lambda (u)
       (if (eq? (partition g u) part)
	   (myg1-vertex-name g u)
	   '-))
     (myg1-vertices g)))
  
  (set! myg1-partition-fidmat-check #t)
  (set! myg1-partition-fidmat-debug #t)
  (myg1-partition-fidmat
   g partition-map gain (prop-external-hash eq? (myg1-num-vertices g))
   working-graph (prop-external-hash eq? (myg1-num-vertices g))
   cost balance '(1 . 1) 25 3)
  (set! working-graph #f)
  
  (print "Partition #f")
  (print "  " (output #f))
  (newline)
  (print "Partition #t")
  (print "  " (output #t))
  (newline)
  (print "Balance " (balance '(1 . 1) #f #f)))

(void)
