;;;; A graph library for Scheme modelled after the successful Boost
;;;; Graph Library for C++

;;; Copyrights.

;;                       Rooster Graph.
;; <pre>
;;
;; Copyright (c) 2004, Jonah Nathaniel Beckford
;; All rights reserved.
;;
;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;;
;;   Redistributions of source code must retain the above copyright
;;   notice, this list of conditions and the following disclaimer.
;;
;;   Redistributions in binary form must reproduce the above
;;   copyright notice, this list of conditions and the following
;;   disclaimer in the documentation and/or other materials provided
;;   with the distribution.
;;
;;   Neither the name of the author nor the names of its contributors
;;   may be used to endorse or promote products derived from this
;;   software without specific prior written permission.
;;
;; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
;; CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
;; INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
;; MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
;; DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
;; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;; SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
;; LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
;; USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
;; AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
;; LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
;; ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
;; POSSIBILITY OF SUCH DAMAGE.
;;
;; jonah@usermail.com
;;
;; </pre>
(define rgraph-doc-copyright-rgraph #t)

;; Boost Software License - Version 1.0 - August 17th, 2003.
;;
;; <pre>
;;
;; Permission is hereby granted, free of charge, to any person or
;; organization obtaining a copy of the software and accompanying
;; documentation covered by this license (the "Software") to use,
;; reproduce, display, distribute, execute, and transmit the
;; Software, and to prepare derivative works of the Software, and to
;; permit third-parties to whom the Software is furnished to do so,
;; all subject to the following:
;;
;; The copyright notices in the Software and this entire statement,
;; including the above license grant, this restriction and the
;; following disclaimer, must be included in all copies of the
;; Software, in whole or in part, and all derivative works of the
;; Software, unless such copies or derivative works are solely in the
;; form of machine-executable object code generated by a source
;; language processor.
;;
;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
;; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
;; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND
;; NON-INFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
;; ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE FOR ANY DAMAGES OR
;; OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, ARISING
;; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
;; OTHER DEALINGS IN THE SOFTWARE.
;;
;; Copyright  2000-2001
;; 
;; Jeremy Siek, Indiana University (jsiek@osl.iu.edu)
;; Lie-Quan Lee, Indiana University (llee@cs.indiana.edu)
;; Andrew Lumsdaine, Indiana University (lums@osl.iu.edu)
;;
;; </pre>
(define rgraph-doc-copyright-boost #t)


;;;                            Usage Notes

;; Imports.
;;
;; <pre>
;;
;; The following imports must be imported by the user *BEFORE*
;; importing rgraph.
;;
;; NECESSARY
;; =========
;;
;; ---------- All Scheme Implementations ----------
;;
;; Srfi-0 : cond-expand
;;
;; ---------- CHICKEN ----------
;;
;; Specified in the rgraph.setup
;;
;; Extras : hash-table
;;
;; OPTIONAL
;; ========
;;
;; Srfi-40 : streams.  needed when you use the star-suffix versions of
;; the methods; for example, when using the stream-valued
;; (Graph-Name-out-edges* ...) instead of the list-valued
;; (Graph-Name-out-edges ...)
;;
;; </pre>
(define rgraph-doc-usage-imports #t)

;; Debugging.
;;
;; <pre> Debugging includes, at minimum, some type-checking of
;; function arguments.
;;
;; The feature (Srfi-0) 'rgraph-nodebug takes precedent over 'rgraph-debug.
;;
;; Chicken - If you are running in CSI, then 'rgraph-debug is implicitly
;; turned on.  You may override by explicitly setting the feature
;; 'rgraph-nodebug.
;;
;; </pre>
(define rgraph-doc-usage-debugging #t)

(cond-expand
  [rgraph-nodebug
    (define-macro rgraph-debug (lambda (body) '()))]
  [(or rgraph-debug csi)
   (define-macro rgraph-debug (lambda (body) body))]
  (else
    (define-macro rgraph-debug (lambda (body) '()))))

; Using (define-macro xxx (include "yyy")) only works for the CHICKEN
; syntax-case form of define-macro (that is, use -hygienic with CSI),
; not the built-in define-macro.  We have a script called expand.scm
; that expands these include statements if your system does not
; support the proper version of define-macro.

;;;			    Adjacency List

;; The adjacency_list class implements a generalized adjacency list
;; graph structure. The template parameters provide many configuration
;; options so that you can pick a version of the class that best meets
;; your needs. An adjacency-list is basically a two-dimensional
;; structure, where each element of the first dimension represents a
;; vertex, and each of the vertices contains a one-dimensional
;; structure that is its edge list. [description copied from boost]
;;
;; .title Adjacency List
;; .form (define-adjacency-list ...)
;; .reference "al" "Rooster Graph Adjacency List" "rgraph-al.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/graph/doc/adjacency_list.html"
(define rgraph-doc-adjacency-list #t)
(define-macro define-adjacency-list
  (include "rgraph-al.scm"))

;; Scheme macro specialization to Adjacency List that stores the
;; vertices in a logarithmetically growing (as you added vertices)
;; vector.
;;
;; .title Vector Vertex List
;; .form vl-vector
;; .reference "vl-vector" "Rooster Graph VL-VECTOR" "rgraph-vl-vector.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/graph/doc/using_adjacency_list.html#sec:choosing-graph-type"
(define rgraph-doc-vl-vector #t)
(define-macro define-vl-vector
  (include "rgraph-vl-vector.scm"))

;; Scheme macro specialization to Adjacency List that stores the
;; vertices in a hashtable.
;;
;; .title Hash Vertex List
;; .form vl-hash
;; .reference "vl-hash" "Rooster Graph VL-HASH" "rgraph-vl-hash.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/graph/doc/using_adjacency_list.html#sec:choosing-graph-type"
(define rgraph-doc-vl-hash #t)
(define-macro define-vl-hash
  (include "rgraph-vl-hash.scm"))

;; Scheme macro specialization to Adjacency List that stores the edges
;; in a singly linked list.
;;
;; .title Singly-Linked Edge List
;; .form el-slist
;; .reference "el-slist" "Rooster Graph EL-SLIST" "rgraph-el-slist.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/graph/doc/using_adjacency_list.html#sec:choosing-graph-type"
(define rgraph-doc-el-slist #t)
(define-macro define-el-slist
  (include "rgraph-el-slist.scm"))

;; Scheme macro specialization to Adjacency List that stores the edges
;; in a hashtable.
;;
;; .title Hash Edge List
;; .form el-hash
;; .reference "el-hash" "Rooster Graph EL-HASH" "rgraph-el-hash.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/graph/doc/using_adjacency_list.html#sec:choosing-graph-type"
(define rgraph-doc-el-hash #t)
(define-macro define-el-hash
  (include "rgraph-el-hash.scm"))

;;;                               Visitors

;; Rooster Graph Visitors
;;
;; .title Visitors
;; .reference "visitors" "Rooster Graph Visitors" "rgraph-vis.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/graph/doc/visitor_concepts.html"
(define rgraph-doc-visitors #t)

;;;                               Properties

;; Rooster Graph Properties
;;
;; .title Properties
;; .reference "properties" "Rooster Graph Properties" "rgraph-prop.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/property_map/property_map.html"
(define rgraph-doc-properties #t)

;;;			    Utility

;; Create a lexical scope with bindings to the graph methods.  The
;; scope will have the following bindings:
;;
;; <pre>
;; make-graph
;; add-edge!
;; remove-edge!
;; remove-edge2!
;; out-edges
;; out-edges*
;; out-edges+
;; out-degree
;; in-edges
;; in-edges*
;; in-edge+
;; in-degree
;; 	
;; add-vertex!
;; remove-vertex!
;; vertex
;; vertex-eq?
;; num-vertices
;; vertices
;; vertices*
;; vertices+
;; clear!
;; 	
;; source
;; target
;; edge-at
;; </pre>
;;
;; The + version of the binding (like vertices+) will be the stream
;; version if it exists, else it will be the list version.  These are
;; useful with the remaining bindings which process the results of the
;; + bindings:
;;
;; <pre>
;; for-each+
;; map+
;; </pre>
;;
;; .title let-rgraph
;; .parameter GTYPE The type name of the graph.
;; .form (let-rgraph GTYPE (body) ...)
(define rgraph-doc-let-rgraph #t)
(define-macro let-rgraph
  (lambda (GTYPE . rest)
    (define plus (cond-expand (srfi-40 "*") (else "")))
    (define prefix-plus (cond-expand (srfi-40 "stream-") (else "")))
    (define (pad . args)
      (string->symbol
       (apply string-append
	      (map (lambda (a)
		     (cond
		      [(string? a) a]
		      [(symbol? a) (symbol->string a)]
		      [else "UNKNOWN_PAD_SYMBOL"]))
		   args))))
    (let (
	  [for-each+ (pad prefix-plus "for-each")]
	  [map+ (pad prefix-plus "map")]

	  [make-graph (pad "make-" GTYPE)]
	  [add-edge! (pad GTYPE "-add-edge!")]
	  [remove-edge! (pad GTYPE "-remove-edge!")]
	  [remove-edge2! (pad GTYPE "-remove-edge2!")]
	  [out-edges (pad GTYPE "-out-edges")]
	  [out-edges* (pad GTYPE "-out-edges*")]
	  [out-edges+ (pad GTYPE "-out-edges" plus)]
	  [out-degree (pad GTYPE "-out-degree")]
	  [in-edges (pad GTYPE "-in-edges")]
	  [in-edges* (pad GTYPE "-in-edges*")]
	  [in-edges+ (pad GTYPE "-in-edges" plus)]
	  [in-degree (pad GTYPE "-in-degree")]

	  [vertex-index (pad GTYPE "-vertex-index")]
	  [add-vertex! (pad GTYPE "-add-vertex!")]
	  [remove-vertex! (pad GTYPE "-remove-vertex!")]
	  [vertex (pad GTYPE "-vertex")]
	  [vertex-eq? (pad GTYPE "-vertex-eq?")]
	  [num-vertices (pad GTYPE "-num-vertices")]
	  [vertices (pad GTYPE "-vertices")]
	  [vertices* (pad GTYPE "-vertices*")]
	  [vertices+ (pad GTYPE "-vertices" plus)]
	  [clear! (pad GTYPE "-clear!")]

	  [edge (pad GTYPE "-edge")]
	  [source (pad GTYPE "-source")]
	  [target (pad GTYPE "-target")]
	  [edge-at (pad GTYPE "-edge-at")]
	  )
    `(let (
	   [for-each+ ,for-each+]
	   [map+ ,map+]

	   [make-graph ,make-graph]
	   [add-edge! ,add-edge!]
	   [remove-edge! ,remove-edge!]
	   [remove-edge2! ,remove-edge2!]
	   [out-edges ,out-edges]
	   [out-edges* ,out-edges*]
	   [out-edges+ ,out-edges+]
	   [out-degree ,out-degree]
	   [in-edges ,in-edges]
	   [in-edges* ,in-edges*]
	   [in-edges+ ,in-edges+]
	   [in-degree ,in-degree]

	   [vertex-index ,vertex-index]
	   [add-vertex! ,add-vertex!]
	   [remove-vertex! ,remove-vertex!]
	   [vertex ,vertex]
	   [vertex-eq? ,vertex-eq?]
	   [num-vertices ,num-vertices]
	   [vertices ,vertices]
	   [vertices* ,vertices*]
	   [vertices+ ,vertices+]
	   [clear! ,clear!]

	   [edge ,edge]
	   [source ,source]
	   [target ,target]

	   )
       ,@rest))))

;; Fill graph from a list of edges, where each edge is a pair of the
;; form '(vertex1 . vertex2).  vertex1, vertex2, etc. must be
;; comparable using eq?.  Gets mutated graph.  Will fill internal
;; property 'vertex-name if defined.  Will set vertex_descriptor if a
;; hash vertex list.
;;
;; .title GTYPE-fill-graph!
;; .form (GTYPE-fill-graph! graph edges set-vertex-name!)
;; .parameter graph The graph object.
;; .parameter edges A list of edges, each a pair of the form '(vertex1 . vertex2)
;; .parameter set-vertex-name! Property to set the name of a vertex.  May be #f.
;; .returns Mutated, filled graph
(define rgraph-doc-fill-graph! #t)
(define-macro (import-fill-graph! GTYPE streamed? vertex-set? edge-set? directed? bidirectional?)
  (define (pad . args)
    (string->symbol
     (apply string-append
	    (map (lambda (a)
		   (cond
		    [(string? a) a]
		    [(symbol? a) (symbol->string a)]
		    [else "UNKNOWN_PAD_SYMBOL"]))
		 args))))
  (define (when-bi . in-args)
    (if bidirectional? in-args '()))
  (let (
	[algorithm (pad GTYPE "-fill-graph!")]
	[vertex-eq? (pad GTYPE "-vertex-eq?")]
	[add-vertex! (pad GTYPE "-add-vertex!")]
	[add-edge! (pad GTYPE "-add-edge!")]
	)
    `(define ,algorithm
       (lambda (g edges set-vertex-name!)
	 (define h (make-hash-table))
	 (for-each
	  (lambda (edge)
	    (let* ([v1 (car edge)]
		   [v2 (cdr edge)]
		   [vertex1 (hash-table-ref h v1)]
		   [vertex2 (hash-table-ref h v2)])
	      (cond [(not vertex1)
		     (set! vertex1 (,add-vertex! g v1))
		     (when set-vertex-name! (set-vertex-name! g vertex1 v1))
		     (hash-table-set! h v1 vertex1)])
	      (cond [(not vertex2)
		     (set! vertex2 (,add-vertex! g v2))
		     (when set-vertex-name! (set-vertex-name! g vertex2 v2))
		     (hash-table-set! h v2 vertex2)])
	      (,add-edge! g vertex1 vertex2)))
	  edges)
	 g))))

;;;			      Algorithms

;; Depth First Search and Depth First Visit
;;
;; .title Depth First Search
;; .reference "dfs" "Depth First Search Algorithms" "rgraph-alg-dfs.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/graph/doc/depth_first_search.html"
(define rgraph-doc-dfs #t)
(include "rgraph-alg-dfs.scm")

;; Topological sort
;;
;; .title Topological Sort
;; .reference "topsort" "Topological Sort" "rgraph-alg-topsort.html"
;; .reference "boost" "Boost Graph Library" "http://www.boost.org/libs/graph/doc/topological_sort.html"
(define rgraph-doc-topsort #t)
(include "rgraph-alg-topsort.scm")

;; Fiduccia-Mattheyses bi-partitioning
;;
;; .title Fiduccia-Mattheyses bi-partitioning
;; .reference "topsort" "Fiduccia-Mattheyses bi-partitioning" "rgraph-alg-part-fidmat.html"
;; .reference "caltech" "Caltech Tutorial" "http://www.cs.caltech.edu/courses/cs137/2004/winter/slides/day9_2up.pdf"
;; .reference "berkeley" "Berkeley Tutorial" "http://www-cad.eecs.berkeley.edu/~kuehl/courses/290a/lectures/007-partition-placement.pdf"
;; .reference "mes" "MES Tutorial" "http://www.microelectronic.e-technik.tu-darmstadt.de/lectures/summer/rse/english/download/Uebung/2.2/fiduccia_alg.pdf"
(define rgraph-doc-part-fidmat #t)
(include "rgraph-alg-part-fidmat.scm")
