/*-GNU-GPL-BEGIN-*
RULI - Resolver User Layer Interface - Querying DNS SRV records
Copyright (C) 2003 Everton da Silva Marques

RULI is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

RULI is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with RULI; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*-GNU-GPL-END-*/

/*
  $Id: ruli_addr.c,v 1.7 2004/06/06 04:27:48 evertonm Exp $
  */


#ifndef _GNU_SOURCE
#define _GNU_SOURCE /* for GNU memrchr() */
#endif


#include <stdio.h>  /* FIXME: debug */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <ruli_addr.h>
#include <ruli_mem.h>


ruli_addr_t *ruli_addr_inet_new(struct in_addr ia)
{
  ruli_addr_t *addr = (ruli_addr_t *) ruli_malloc(sizeof(*addr));
  if (!addr)
    return 0;

  addr->addr_family = PF_INET;
  addr->addr_size = sizeof(ia);
  addr->addr.ipv4 = ia;

  return addr;
}

ruli_addr_t *ruli_addr_inet6_new(struct in6_addr ia)
{
  ruli_addr_t *addr = (ruli_addr_t *) ruli_malloc(sizeof(*addr));
  if (!addr)
    return 0;

  addr->addr_family = PF_INET6;
  addr->addr_size = sizeof(ia);
  addr->addr.ipv6 = ia;

  return addr;
}

int ruli_addr_family(const ruli_addr_t *addr)
{
  return addr->addr_family;
}

int ruli_addr_size(const ruli_addr_t *addr){
  return addr->addr_size;
}

struct in_addr ruli_addr_inet(const ruli_addr_t *addr)
{
  assert(ruli_addr_family(addr) == PF_INET);
  assert(ruli_addr_size(addr) == sizeof(struct in_addr));

  return addr->addr.ipv4;
}

struct in6_addr ruli_addr_inet6(const ruli_addr_t *addr)
{
  assert(ruli_addr_family(addr) == PF_INET6);
  assert(ruli_addr_size(addr) == sizeof(struct in6_addr));

  return addr->addr.ipv6;
}

#define RULI_WORD16_LEN (sizeof(struct in6_addr) / 2)

static int search_forward(const char *cp, const char *past_end,
			  unsigned int *word16, int *wi,
			  const char **wildcard)
{
  const char *i;
  long w;

  *wi = 0;

  for (i = cp; i < past_end; ) {

    if (*i == ':') {
      ++i;
      if (*i == ':') {
        *wildcard = i;
	return 0;
      }
      /* can't start addr with : */
      if (*wi == 0)
        return -1;
    }

    if (*wi >= RULI_WORD16_LEN)
      return -1;

    if (!isxdigit(*i))
      return -1;

    w = strtol(i, 0, 16);

    if (w == LONG_MIN)
      return -1;
    if (w == LONG_MAX)
      return -1;
    if (w < 0)
      return -1;
    if (w > 0xFFFF)
      return -1;

    word16[*wi] = w;
    ++*wi;

    i = memchr(i, ':', past_end - i);
    if (!i)
      break;
  }

  return 0;
}

static int search_backward(const char *cp, const char *past_end,
			   unsigned int *word16, int *wj,
			   const char **wildcard)
{
  const char *i;
  long w;

  *wj = RULI_WORD16_LEN;

  for (i = past_end; i > cp; --i) {
    const char *j;

    i = memrchr(cp, ':', i - cp);
    if (!i)
      return -1;

    if (*wj <= 0)
      return -1;

    j = i + 1;

    w = strtol(j, 0, 16);

    if (w == LONG_MIN)
      return -1;
    if (w == LONG_MAX)
      return -1;
    if (w < 0)
      return -1;
    if (w > 0xFFFF)
      return -1;

    word16[--*wj] = w;

    if (*(i - 1) == ':') {
      *wildcard = i;
      return 0;
    }

    if (!isxdigit(*j))
      return -1;
  }

  return -1;
}

int ruli_inet6_aton(const char *cp, struct in6_addr *inp)
{
  unsigned int word16[RULI_WORD16_LEN];
  const char *past_end;
  const char *wildcard_fwd = 0;
  const char *wildcard_bwd;
  int wi, wj;

  /* locate end of string */
  past_end = memchr(cp, '\0', 40);
  if (!past_end)
    return 0;

  /* search from start up to :: or end */
  if (search_forward(cp, past_end, word16, &wi, &wildcard_fwd))
    return 0;

  if (wi < RULI_WORD16_LEN) {

    /* search from end up to :: */
    if (search_backward(cp, past_end, word16, &wj, &wildcard_bwd))
      return 0;

    /* wildcard :: was hit? */
    if (wildcard_fwd) {
      /* forward found the same wildcard as backward? */
      if (wildcard_fwd != wildcard_bwd)
        return 0;
    }
    
    /* clean unused slots */
    {
      int i;
      for (i = wi; i < wj; ++i)
	word16[i] = 0;
    }
  }
  
  /* convert 2-byte host-order array to 1-byte net-order array */
  {
    char *p = (char *) inp;

    for (wi = 0; wi < RULI_WORD16_LEN; ++wi) {
      unsigned int w = word16[wi];
      *p = w >> 8;
      ++p;
      *p = w & 0xFF;
      ++p;
    }
  }

  return -1;
}

int ruli_addr_parse(const char *p, ruli_addr_t *addr)
{
  if (ruli_inet6_aton(p, &addr->addr.ipv6)) {
    addr->addr_family = PF_INET6;
    addr->addr_size = sizeof(addr->addr.ipv6);
    return 0; 
  }

  if (inet_aton(p, &addr->addr.ipv4)) {
    addr->addr_family = PF_INET;
    addr->addr_size = sizeof(addr->addr.ipv4);
    return 0; 
  }

  addr->addr_family = PF_UNSPEC;
  addr->addr_size = 0;

  return -1;
}

ruli_addr_t *ruli_addr_parse_new(const char *p)
{
  struct in_addr addr;
  struct in6_addr addr6;

  if (ruli_inet6_aton(p, &addr6))
    return ruli_addr_inet6_new(addr6);

  if (inet_aton(p, &addr))
    return ruli_addr_inet_new(addr);

  return 0;
}

int ruli_addr_print(FILE *out, const ruli_addr_t *addr)
{
  switch (addr->addr_family) {
  case PF_INET:
    return fprintf(out, inet_ntoa(addr->addr.ipv4));

  case PF_INET6:
    {
      const char *begin = (const char *) &addr->addr.ipv6;
      const char *past_end = begin + addr->addr_size;
      int wr = 0;
      const char *i;

      assert(sizeof(addr->addr.ipv6) == addr->addr_size);

      for (i = begin; i < past_end; ++i) {
	int j = i - begin;
	if (j)
	  if (!(j % 2))
	    wr += fprintf(out, ":");

	wr += fprintf(out, "%02X", (unsigned char) *i);
      }

      return wr;
    }

  default:
    assert(0);
  }

  return -1;
}

int ruli_addr_snprint(char *buf, size_t size, const ruli_addr_t *addr)
{
  switch (addr->addr_family) {
  case PF_INET:
    return snprintf(buf, size, inet_ntoa(addr->addr.ipv4));

  case PF_INET6:
    {
      const char *begin = (const char *) &addr->addr.ipv6;
      const char *past_end = begin + addr->addr_size;
      int wr = 0;
      const char *i;

      assert(sizeof(addr->addr.ipv6) == addr->addr_size);

      for (i = begin; i < past_end; ++i) {
	int j = i - begin;
	int w;

	assert(wr < size);

	if (j)
	  if (!(j % 2)) {
	    w = snprintf(buf + wr, size - wr, ":");
	    if (w < 0)
	      return -1;
	    wr += w;
	    if (wr >= size)
	      return wr;

	    assert(wr < size);
	  }

	w = snprintf(buf + wr, size - wr, "%02X", (unsigned char) *i);
	if (w < 0)
	  return -1;
	wr += w;
	if (wr >= size)
	  return wr;

	assert(wr < size);
      }

      return wr;
    }

  default:
    assert(0);
  }

  return -1;
}
