#!/usr/bin/env ruby
#
# Fake SPARQL endpoint for BSBM test driver
#

class BsbmQuery
  SPARQL_QUERIES = '/mnt/d/trash/bsbm/bsbmtools/queries'
  SQUISH_QUERIES = '/mnt/d/trash/bsbm/bsbmtools/samizdat-bsbm'

  def initialize(id)
    sparql = File.open(File.join(SPARQL_QUERIES, "query#{id}.txt")).read.strip
    sparql.gsub!(/\s+/, ' ')

    @query_type =
      case sparql
      when /SELECT/    then :select
      when /CONSTRUCT/ then :construct
      when /DESCRIBE/  then :describe
      else raise RuntimeError, 'Unrecognized query type'
      end

    sparql = Regexp.escape(sparql)
    @params = []
    sparql.gsub!(/%([a-zA-Z0-9]+)%/) do
      @params.push($1.untaint.to_sym)
      '(.*?)'
    end
    sparql.gsub!('\ ', '\s+')
    @sparql_pattern = Regexp.new('\s*' + sparql + '\s*').freeze

    @squish = File.open(File.join(SQUISH_QUERIES, "query#{id}.txt")).read.untaint

    @vars = /SELECT.*?WHERE/m.match(@squish).to_s.scan(/\?[^,\s]+/).collect {|v| v[1..-1] }

    case @query_type
    when :select
      @limit = /LIMIT\s+(\d+)/.match(sparql).to_a[1]
      @offset = /OFFSET\s+(\d+)/.match(sparql).to_a[1]
      @distinct = /DISTINCT/.match(sparql) ? true : false
      @ordered = /ORDER BY/.match(sparql) ? true : false
    end
  end

  attr_reader :query_type, :vars, :distinct, :ordered

  # returns query parameters hash if query matches
  #
  def match(sparql)
    if m = @sparql_pattern.match(sparql)
      params = {}
      [ @params, m[1..-1] ].transpose.each do |param, value|
        if value =~ /(\d+)>$/
          value = $1.to_i
        end
        params[param] = value
      end
      params
    end
  end

  def run(params)
    results =
      case @query_type
      when :select
        rdf.select_all(@squish, @limit, @offset, params)
      when :construct, :describe
        [ rdf.select_one(@squish, params) ]
      end

    results.collect do |values|
      [ @vars, values ].transpose.collect {|var, value|
        %{   <binding name="#{var}"><literal>#{value}</literal></binding>\n}
      }.join
    end
  end
end

class BsbmController < Controller
  def initialize(request, id = nil)
    super
    @@query_map ||= load_query_map
  end

  def index
    query = @request['query'] or
      raise RuntimeError, 'No query'

    params = nil
    query = @@query_map.detect {|q| p = q.match(query) and params = p } or
      raise RuntimeError, "Unrecognized query"

    @request.headers['type'] =
      case query.query_type
      when :describe, :construct
        'application/rdf+xml'
      else
        'application/sparql-results+xml'
      end
    @layout = nil

    @content_for_layout = %{<?xml version="1.0"?>
<sparql xmlns="http://www.w3.org/2005/sparql-results#">

 <head>
#{ query.vars.collect {|v| %{  <variable name="#{v}"/>\n} }.join
} </head>
 <results distinct="#{query.distinct}" ordered="#{query.ordered}">
#{
      query.run(params).collect {|result|
        "  <result>\n" + result + "  </result>\n"
      }.join
} </results>
</sparql>}
  end

  private

  def load_query_map
    (1..12).collect {|i| BsbmQuery.new(i) }
  end
end
