#!/usr/bin/env ruby
#
# Samizdat publish message form
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'uri'
require 'ftools'
require 'samizdat'

session = Session.new

session.response do |t|
    session.id or raise AuthError, _('You should be logged in to post messages')

    title, content, format, parent =
        session.params %w[title content format parent]

    file = session['file']   # just a file object, not its contents

    # detect and check format
    if format.nil? and file.methods.include? 'content_type' and file.size > 0
        format = file.content_type.strip   # won't work with FastCGI
        format = 'image/jpeg' if format == 'image/pjpeg'   # MSIE...
    end
    raise UserError, sprintf(_("Format '%s' is not supported"), format) if
        format and not config['format'].values.flatten.include? format
    inline = (format.nil? or config['format']['inline'].include? format)

    if file and file.size > 0 then   # content uploaded from file
        content = nil
        format.nil? and raise UserError,
            _('It is not possible to upload a file without specifying format')
        # todo: fine-grained size limits
        file.size > config['limit']['content'] and raise UserError,
            sprintf(_('Uploaded file is larger than %s bytes limit'),
                config['limit']['content'])

        location = Resource.new( session
            ).content_location('upload', format, session.login)
        upload = session.filename(location)
        upload.untaint   # security: keep format and login controlled

        if (file.kind_of? StringIO or file.kind_of? Tempfile) and
        not session.has_key? 'confirm' then   # new upload
            if inline then
                content = file.read   # transform to inline message
                file = nil
            else
                config['site']['content'].nil? and raise UserError,
                    _('Multimedia upload is disabled on this site')
                File.makedirs(File.dirname(upload)) unless
                    File.exists?(File.dirname(upload))
                if file.kind_of? Tempfile then   # copy large files directly
                    file = File.syscopy(file.path, upload)
                else   # StringIO
                    File.open(upload, 'w') {|f| f.write(file.read) }
                    file = true
                end
            end

        elsif file.kind_of? String and session.has_key? 'confirm' then
            inline and raise UserError, 'Unexpected inline upload confirm'
            file = nil unless File.exists?(upload)
        else
            raise UserError, 'Unexpected upload state'
        end
    end   # at this point, file is true and content is nil if upload is ready

    if content then   # inline message
        file = nil
        inline or raise UserError,
            sprintf(_('You should upload %s content from file'), format)
        raise UserError,
            _('text/uri-list should contain at least one absolute URI') if
                format == 'text/uri-list' and not content =~ URI::ABS_URI
    end

    if title and (content or file) then
        if session.has_key? 'confirm' then   # write message into the database
            # todo: detect duplicates
            id = nil   # scope fix
            db.transaction do |db|
                # todo: translate into RDF assert
                db.do 'INSERT INTO Message (creator, title, parent, thread,
                    format, content) VALUES (?, ?, ?, ?, ?, ?)', session.id,
                    title, parent, 0, format, content
                id, = db.select_one "SELECT currval('Resource_id_seq')"
                if parent then
                    db.do "UPDATE Message SET thread =
                        (SELECT thread FROM Message WHERE id = ?)
                        WHERE id = ?", parent, id
                else   # new thread
                    db.do "UPDATE Message SET thread = ? WHERE id = ?", id, id
                end
                if file then
                    # todo: inject file into p2p net
                    File.rename(upload, session.filename(
                        Resource.new( session
                        ).content_location(id, format, session.login)
                    ).untaint)
                end
            end   # transaction

            # Pingback client
            if format == 'text/uri-list' then
                # attempt Pingback on the first URI
                catch :fail do
                    begin
                        require 'net/http'
                        require 'xmlrpc/client'
                    rescue LoadError
                        throw :fail
                    end
                    throw :fail unless content =~ URI::ABS_URI
                    uri, scheme = $&, $1
                    throw :fail unless scheme =~ /^http/
                    response = Net::HTTP.get_response(URI.parse(uri.untaint))
                    throw :fail unless response.kind_of? Net::HTTPSuccess
                    pingback = response['x-pingback']
                    if pingback.nil? then
                        throw :fail unless response.body =~
                            %r{<link rel="pingback" href="([^"]+)" ?/?>}
                        pingback = $1
                        pingback.gsub!(/&amp;/, '&')
                        pingback.gsub!(/&lt;/, '<')
                        pingback.gsub!(/&gt;/, '>')
                        pingback.gsub!(/&quot;/, '"')
                    end
                    throw :fail unless pingback =~ URI::ABS_URI
                    u = URI.parse(pingback.untaint)
                    server = XMLRPC::Client.new(u.host, u.path, u.port)
                    server.call2("pingback.ping", session.base + id.to_s, uri)
                    # discard the result
                    # todo: notify user that ping was registered
                end
            end
            # redirect to published message
            session.redirect(id.to_s)

        else   # preview message
            content = location if file   # preview upload.ext
            t.page( _('Message Preview'),
                t.message(:date => Time::now, :creator => session.id,
                    :full_name => session.full_name, :title => title,
                    :format => format, :content => content, :parent => parent) +
                t.form( 'message.rb',
                    [:submit, 'confirm', _('Confirm')],
                    [:hidden, 'parent', parent],
                    [:hidden, 'title', title],
                    [:hidden, 'content', file ? nil : content],
                    [:hidden, 'file', file],
                    [:hidden, 'format', format]
                ) + '<p>'+_("Press 'Back' button to change the message.")+'</p>'
            )
        end

    else   # edit message
        t.page(_('New Message'), t.form('message.rb',
            [:label, 'title', _('Title')], [:text, 'title', title],
            [:label, 'content', _('Content')], [:textarea, 'content', content],
            [:label, 'file', _('Upload content from file')],
                [:file, 'file', file],
            [:label, 'format', _('Format')],
                [:select, 'format', [[nil, _('Default')]] +
                    config['format']['inline'].to_a, format],
            [:label], [:submit, 'preview', _('Preview')],
            [:hidden, 'parent', parent]))
    end
end
