#!/usr/bin/env ruby
#
# Samizdat search query construction
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'samizdat'

request do |session|
    namespaces = ns   # standard namespaces

    t = session.template
    query, substring, skip = session.params %w[query substring skip]
    skip = skip.to_i

    # phase 1: define query

    if substring and session.has_key? 'search' then
        # search messages by a substring
        query = %{SELECT ?msg
WHERE (dc::date ?msg ?date)
      (dc::title ?msg ?title)
LITERAL ?title ILIKE '%#{substring.gsub(/%/, '\\\\\\%')}%'
ORDER BY ?date DESC
USING dc FOR http://purl.org/dc/elements/1.1/}
        session['run'] = 'Run'
    end

    if session.has_key? 'update' then   # regenerate query from form data
        nodes, literal, order, order_dir, using =
            session.params %w[nodes literal order order_dir using]
        # generate namespaces
        namespaces.update(Hash[*using.split])
        using = {}
        # generate query pattern
        pattern = []
        session.keys.grep(/\A(predicate|subject|object)_(\d{1,2})\z/) do |key|
            value, = session.params([key])
            next unless value
            i = $2.to_i - 1
            pattern[i] = [] unless pattern[i]
            pattern[i][ %w[predicate subject object].index($1) ] = value
            namespaces.each do |p, uri|
                if value =~ /\A#{p}::/ then
                    using[p] = uri   # only leave used namespace prefixes
                    break
                end
            end
        end

        query = "SELECT #{nodes}\nWHERE " +
            pattern.collect {|predicate, subject, object|
                "(#{predicate} #{subject} #{object})" if
                    predicate and subject and object and predicate !~ /\s/
                    # whitespace is only present in 'BLANK CLAUSE'
            }.compact.join("\n      ")
        query += "\nLITERAL #{literal}" if literal
        query += "\nORDER BY #{order} #{order_dir}" if order
        query += "\nUSING " + using.to_a.collect {|n|
            n.join(' FOR ')
        }.join("\n      ")
        session['run'] = 'Run'
    end

    if session.keys.size == 0 then   # default query
        query = %{SELECT ?resource
WHERE (dc::date ?resource ?date)
ORDER BY ?date DESC
USING dc FOR #{ns['dc']}}
    end

    # phase 2: validate query

    begin
        q = SquishQuery.new(query)
    rescue ProgrammingError
        raise UserError, _('Error in your query: ') + $!
    end

    (q.nodes.size != 1) and raise UserError,
_('Must-bind list should contain only one blank node, filters based on queries with a complex answer pattern are not implemented')

    (q.pattern.size > config['limit']['pattern']) and raise UserError,
        sprintf(_('User-defined query pattern should not be longer than %s clauses'), config['limit']['pattern'])

    # phase 3: act on query

    if session.has_key? 'run' then
        title = _('Search Result') +
            (skip > 0? sprintf(_(', page %s'), skip + 1) : '')
        begin
            sql = rdf.select(q)
        rescue ProgrammingError
            raise UserError, _('Error in your query: ') + $!
        end
        page = rdf.select_all(
            q, limit_page, limit_page * skip
        ).collect {|id,| Resource.new(session, id).render }
        page =
            if page.size > 0 then
                t.list( page,
                    t.nav(skip + 1,
                        %{query.rb?run&amp;query=#{CGI.escape(query)}&amp;}),
                    t.form('message.rb',
                        [:hidden, 'format', 'application/x-squish'],
                        [:hidden, 'content', query],
                        [:submit, 'preview', _('Publish This Query')])
                )
            else
                '<p>'+_('No matching resources found.')+'</p>'
            end
    end

    page = page ? [[title, page]] : []   # single- or multi-section page

    # phase 4: query construction interface

    edit = t.box(_('Select Target'),
        t.form_field(:select, 'nodes', q.pattern.collect {|p, s, o| s }.uniq))

    i = 0
    edit += t.box(_('Query Pattern (predicate subject object)'),
    ((q.pattern.size >= config['limit']['pattern'])?
    q.pattern : q.pattern + ['']).collect {|clause|
        predicate, subject, object = clause.collect {|uri| q.ns_shrink(uri) }
        i += 1
        t.form_field(:label) + %{#{i}. } +
        # todo: add external properties to the options list
        t.form_field(:select, "predicate_#{i}",
            [_('BLANK CLAUSE ')] + config['map'].keys.sort, predicate) +
            # make sure 'BLANK CLAUSE' includes whitespace in all translations
        t.form_field(:text, "subject_#{i}", subject) +
        t.form_field(:text, "object_#{i}", object)
        # todo: select subject and object from variables and known urirefs
    }.join)

    edit += t.box(_('Literal Condition'),
        t.form_field(:text, 'literal', q.literal))

    edit += t.box(_('Order By'),
        t.form_field(:select, 'order',
            q.pm.keys.grep(SquishQuery::BN), q.order) +
        t.form_field(:select, 'order_dir',
            [['ASC', _('Ascending')], ['DESC', _('Descending')]], q.order_dir))

    edit += t.box(nil, '<pre>USING ' +
        q.ns.to_a.collect {|n| n.join(' FOR ') }.join("\n      ") + '</pre>' +
        t.form_field(:hidden, 'using', q.ns.to_a.flatten.join(' ')))

    # wrap edit in other query variants
    edit = t.box(_('Search By Substring'),
            t.form_field(:text, 'substring', substring) +
            t.form_field(:submit, 'search', _('Search'))) +
        t.box(_('Construct Query'), edit +   # here
            t.form_field(:submit, 'update', _('Update'))) +
        t.box(_('Edit Raw Query'),
            t.form_field(:textarea, 'query', query) +
            t.form_field(:label) +
            t.form_field(:submit, 'run', _('Run')))

    # wrap edit in form tag
    edit = %{<form action="query.rb" method="post">#{edit}</form>}

    page.push [_('Edit Query'), edit]
    page.collect! {|h, b| [h, q.unstring(b)] }

    t.page(nil, page)
end
