# Samizdat session management
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'delegate'
require 'digest/md5'

begin
    raise LoadError if defined?(MOD_RUBY)
    require 'fcgi'
    def request
        FCGI.each_cgi do |cgi|
            catch :finish do
                Session.new(cgi).response {|s| yield s }
            end
        end
    end
rescue LoadError
    def request
        Session.new(CGI.new).response {|s| yield s }
    end
end

# session management and CGI parameter handling
#
class Session < SimpleDelegator
    @@cookie_prefix = config['session']['cookie_prefix'] + '_'
    @@login_timeout = config['session']['login_timeout']
    @@last_timeout = config['session']['last_timeout']

    # wrapper for CGI#cookies: add cookie name prefix, return first value
    #
    def cookie(name)
        @cgi.cookies[@@cookie_prefix + name][0]
    end

    # create cookie and add it to the HTTP response
    #
    # cookie name prefix is configured via config.yaml; default expiry timeout
    # is #forever as defined in samizdat.rb
    #
    def set_cookie(name, value=nil, expires=forever)
        options['cookie'] = [] if nil == options['cookie']
        options['cookie'] << CGI::Cookie.new({
            'name' => @@cookie_prefix + name,
            'value' => value,
            'expires' => Time.now + expires
        })
    end

    # translate location to a real file name
    #
    def filename(location)
        if defined?(MOD_RUBY) then
            Apache.request.lookup_uri(location).filename
        else
            @cgi.env_table['DOCUMENT_ROOT'] + location
        end
    end

    # set language
    #
    def language=(lang)
        lang = nil unless /\A([A-Za-z]+)(_[A-Za-z]+)?\z/ =~ lang
        lang.untaint   # todo: proper locale name validation (rfc?)
        if defined?(GetText) then
            bindtextdomain('samizdat', config['locale']['path'], lang, 'utf-8')
        end
        @language = lang
    end

    # current language
    attr_reader :language

    # set default CGI options (set charset to UTF-8)
    #
    # set id and refresh session if session cookie is valid
    #
    def initialize(cgi)
        @cgi = cgi
        class << @cgi
            public :env_table
        end
        @options = {'charset' => 'utf-8', 'cookie' => []}

        self.language = cookie('lang')

        proto = @cgi.env_table['HTTPS'] ? 'https' : 'http'
        port = @cgi.env_table['SERVER_PORT'].to_i
        port = (port == {'http' => 80, 'https' => 443}[proto]) ?
            '' : ':' + port.to_s
        @base = proto + '://' + @cgi.host + port + config['site']['base'] + '/'

        @template = Template.new(self)

        # check session
        @session = cookie('session')
        if @session and @session != '' then
db.transaction do |db|
            @id, @login, @full_name, @email, login_time, last_time = db.select_one 'SELECT id, login, full_name, email FROM Member WHERE session = ?', @session
            if @id then
                if (login_time and login_time < Time::now - @@login_timeout) or
                (last_time and last_time < Time::now - @@last_timeout) then
                    close   # stale session
                else
                    # uncomment to regenerate session on each access:
                    #@session = generate_session
                    #db.do "UPDATE Member SET last_time = 'now', session = ?
                    #WHERE id = ?", @session, @id
                    set_cookie('session', @session, @@last_timeout)
                end
            end
end
        end
        super @cgi
    end

    # HTTP response options
    attr_reader :options

    # base URI of the site
    attr_reader :base

    # Template object aware of this session's options
    attr_reader :template

    attr_reader :id, :login, :full_name, :email

    # open new session on login
    #
    # redirect to referer on success
    #
    def open(login, passwd)
        db.transaction do |db|
            @id, = db.select_one 'SELECT id FROM Member m WHERE login = ?
            AND passwd = ?', login, Digest::MD5.new(passwd).hexdigest
            if @id then
                @session = generate_session
                db.do "UPDATE Member SET login_time = 'now', last_time = 'now',
                session = ? WHERE id = ?", @session, @id
                db.commit
                set_cookie('session', @session, @@last_timeout)
                redirect(base)
            else
                response() { template.page(_('Login Failed'),
                    '<p>'+_('Wrong login name or password. Try again.')+'</p>')
                }
            end
        end
    end

    # erase session from database
    #
    def close
        db.do "UPDATE Member SET session=NULL WHERE id = ?", @id
        db.commit
        @id = nil
        set_cookie('session', nil, 0)
    end

    # return list of values of CGI parameters, tranform empty values to nils
    #
    # unlike CGI#params, Session#params takes array of parameter names
    #
    def params(keys)
        keys.collect do |key|
            value = self[key]
            raise UserError, _('Input size exceeds content size limit') if
                value.methods.include? :size and
                value.size > config['limit']['content']
            case value
            when StringIO, Tempfile then
                value = value.read
            end
            (value =~ /[^\s]/)? value : nil
        end
    end

    # always imitate CGI#[] from Ruby 1.8
    #
    def [](key)
        @cgi.params[key][0]
    end

    # plant a fake CGI parameter
    #
    def []=(key, value)
        @cgi.params[key] = [value]
    end

    # print header and optionally content, then clean-up and exit
    #
    # generate error page on RuntumeError exceptions
    #
    def response(options={})
        @options.update(options)
        if block_given? then
            page =
                begin
                    yield self
                rescue AuthError
                    @options['status'] = 'AUTH_REQUIRED'
                    template.page(_('Please Login'), %{<p>#{$!}.</p>})
                rescue UserError
                    template.page(_('User Error'),
%{<p>#{$!}.</p><p>}+_("Press 'Back' button of your browser to return.")+'</p>')
                rescue ResourceNotFoundError
                    @options['status'] = 'NOT_FOUND'
                    referer = ' ('+_('looks like it was')+%{ <a href="#{@cgi.referer}">#{@cgi.referer}</a>)} if @cgi.referer
                    template.page(_('Resource Not Found'),
'<p>'+_('The resource you requested was not found on this site. Please report this error back to the site you came from')+referer.to_s+'.</p>')
                rescue RuntimeError
                    template.page(_('Runtime Error'),
'<p>'+_('Runtime error has occured:')+%{ #{$!}.</p>
<pre>#{caller.join("\n")}</pre>
<p>}+_('Please report this error to the site administrator.')+'</p>')
                end
            print @cgi.out(@options) { page }
        else
            print @cgi.header(@options)
        end
        if defined?(FCGI) then
            throw :finish
        else
            exit
        end
    end

    def redirect(location=referer)
        location = base if location.nil?
        response({'status' => 'REDIRECT', 'location' => location})
    end

private

    def generate_session
        Digest::MD5.new(@id.to_s + Time::now.to_s).hexdigest
    end
end
