#!/usr/bin/env ruby
#
# Samizdat template tests
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'test/unit'
require 'rexml/document'
require 'test/util'
require 'samizdat'

class TestSession
    def initialize
        @id = 1
        @login = 'test'
        @full_name = 'Test'
        @options = {}
        @base = 'http://localhost' + config['site']['base'] + '/'
        bindtextdomain('samizdat', nil, 'C', 'utf-8')
    end

    attr_accessor :id, :login, :full_name, :options, :base
end

class TC_Template < Test::Unit::TestCase
    def setup
        # create a dummy session
        @s = TestSession.new
        @t = Template.new(@s)
    end

    def teardown
        @t = nil
        @s = nil
    end

    def test_box
        root = parse(@t.box('Title', 'Content.'))
        assert_equal 'box', root.attributes['class']
        assert_equal ['Title'], elements(root, 'div[@class="box-title"]')
        assert_equal ['Content.'], elements(root, 'div[@class="box-content"]')

        root = parse(@t.box(nil, 'Content.'))
        assert_equal [], elements(root, 'div[@class="box-title"]')
        assert_equal ['Content.'], elements(root, 'div[@class="box-content"]')
    end

    def test_form_field_label
        root = parse(@t.form_field(:label, 'some_field', 'Label'))
        assert_equal 'label', root.attributes['class']
        assert_equal nil, root.elements['label'].attributes['for']
        assert_equal ['Label'], elements(root, 'label')
    end

    def test_form_field_textarea
        root = parse(@t.form_field(:textarea, 't_a', "Some\nText."))
        assert_equal 't_a', root.attributes['name']
        assert_equal "Some\nText.", root.text.strip
    end

    def test_form_field_select
        root = parse(@t.form_field(:select, 't_select',
            ['a1', ['a2', 'Label 2']]))
        assert_equal 't_select', root.attributes['name']
        assert_equal ['a1'], elements(root, 'option[@value="a1"]')
        assert_equal ['Label 2'], elements(root, 'option[@value="a2"]')
    end

    def test_form_field_submit
        root = parse(@t.form_field(:submit, 't_submit', 'Click'))
        assert_equal 't_submit', root.attributes['name']
        assert_equal 'Click', root.attributes['value']
        assert_equal 'submit', root.attributes['type']
        assert_equal 'submit', root.attributes['class']
    end

    def test_form_field_input
        root = parse(@t.form_field(:password, 'passwd', 'secret'))
        assert_equal 'passwd', root.attributes['name']
        assert_equal 'secret', root.attributes['value']
        assert_equal 'password', root.attributes['type']
    end

    def test_form
        root = parse(@t.form('login.rb', [:password, 'passwd'], [:submit]))
        assert_equal 'login.rb', root.attributes['action']
        assert_equal 'post', root.attributes['method']
        assert_equal ['password', 'submit'], elements(root, 'div/input', 'type')

        root = parse(@t.form('message.rb', [:file, 't_file']))
        assert_equal 'message.rb', root.attributes['action']
        assert_equal 'post', root.attributes['method']
        assert_equal 'multipart/form-data', root.attributes['enctype']
        assert_equal ['file'], elements(root, 'div/input', 'type')
    end

    def test_member_line
        @s.id = nil
        @t = Template.new(@s)
        root = parse('<p>'+@t.member_line+'<p>')
        assert_equal ['login.rb'], elements(root, '//a', 'href')
    end

    def test_member_line_registered
        root = parse('<p>'+@t.member_line+'<p>')
        assert_equal [@s.id.to_s, 'member.rb', 'logout.rb'],
            elements(root, '//a', 'href')
    end

    def test_message_buttons
        root = parse(@t.message(:mode => :full,
            :full_name => 'Test', :title => 'Title'))
        assert_equal ['title', 'parent', 'reply', 'edit'],
            elements(root, 'div[@class="foot"]/form/div/input', 'name')
    end

    def test_message_info_date
        @t.message_info(:date => Time.now, :full_name => 'Test')
        @t.message_info(:date => '2004-03-11', :full_name => 'Test')
        @t.message_info(:date => nil, :full_name => 'Test')
        date = 'current time'
        class << date   # check duck typing
            def to_time
                Time.now
            end
        end
        @t.message_info(:date => date, :full_name => 'Test') =~ /<\/a> on (.*)$/
        assert_not_equal date, $1
    end

    def test_message_info_parent
        assert_equal true,
            @t.message_info( :full_name => 'Test', :parent => 2
            ).split(",\n ")[1].include?('href="2"')
    end

    def test_message_content
        assert_equal nil, @t.message_content(:mode => :list)

        root = parse(@t.box(nil, @t.message_content(
            :content => "line 1\n\nline 2")))
        assert_equal ['line 1', 'line 2'], elements(root, 'div/p')

        assert_equal 'pre',
            parse(@t.message_content(:format => 'text/plain')).name

        root = parse(@t.message_content(:format => 'text/uri-list',
            :content => "#comment\nhttp://localhost/"))
        assert_equal 'comment', text(root, '')
        assert_equal ['http://localhost/'], elements(root, 'a', 'href')

        root = parse(@t.message_content(:format => 'application/x-squish',
            :content => 'SELECT'))
        assert_equal ['query', nil, 'run'], elements(root, 'div/*', 'name')

        root = parse(@t.message_content(:format => 'image/jpeg',
            :title => 'A', :content => 'anarchy.png'))
        assert_equal 'A', root.attributes['alt']
        assert_equal 'anarchy.png', root.attributes['src']

        root = parse(@t.message_content(:format => 'application/pdf'))
        assert_equal ['application/pdf content'], elements(root, 'a')
    end

    def test_simple_page
        root = parse(@t.page('Test Title', 'Some text.'))
        assert_equal ['Test Title'],
            elements(root, '//div[@class="box-title"]')
    end

    def test_multi_page
        root = parse( @t.page('Title',
            [ ['Title 1', 'Body 1.'], ['Title 2', 'Body 2.']]))
        assert_equal [config['site']['name'] + ': Title'],
            elements(root, '/html/head/title')
        main = root.elements['/html/body/div[@id="main"]']
        assert_equal ['Title 1', 'Title 2'],
            elements(main, 'div/div[@class="box-title"]')
        assert_equal ['Body 1.', 'Body 2.'],
            elements(main, 'div/div[@class="box-content"]')
    end
end
