# Samizdat resource representation
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

class Resource
    def initialize(session, id=nil)
        @session = session
        @template = session.template
        case id
        when Integer then @id = id
        else
            @id = id.to_i   # only numeric ids allowed
        end
        @uriref = session.base + id.to_s
    end

    attr_reader :session, :template, :id, :uriref

    # in :list mode (default) return rendered desc
    #
    # in :short mode return hash with :type, :head, and :desc keys
    #
    # in :full mode, add :related resources and :focuses to the hash
    # (only one full resource per page is possible)
    #
    def render(mode=:list)
        if self.class == Focus and id.to_i == 0 then
            label = id.to_s
            external = true
        else
            external, literal, label = db.select_one('SELECT uriref, literal, label FROM Resource WHERE literal = false AND id = ?', id)
        end
        r = {}
        if external then
            r[:type] = 'Uriref'
            head = SquishQuery.ns_shrink(label)
            head.gsub!(/\Afocus::/, '') and head = _(head)   # focus is special
            r[:head] = head
            r[:desc] = '<div>' + sprintf(_('refers to <a href="%s">external uriref</a>'), label) + '</div>'
            # todo: select all statements with this subject
        elsif literal then
            r[:type] = 'Literal'
            r[:head] = label
            r[:desc] =
%{<a href="#{uriref}">#{SquishQuery.ns_shrink(uriref)}</a> = #{label}}
        else   # internal resource
            r =
                case label
                when nil then raise ResourceNotFoundError, id.to_s
                when 'Member' then render_member mode
                when 'Message' then render_message mode
                when 'Statement' then render_statement mode
                when 'Vote' then render_vote mode
                when 'Item' then render_item mode
                else raise RuntimeError, _("Unknown resource type '%s'") % label
                end
            r[:type] = label
        end
        return template.resource(id, r[:head], r[:desc]) if :list == mode
        r[:focuses] = render_focuses(r[:head], mode) if :full == mode
        class << r
            def to_s
                [ self[:desc], self[:focuses], self[:related] ].join
            end
        end
        r
    end

    # multimedia message content location
    #
    # if _format_ or _login_ are not specified, this Resource object should be
    # instantiated from a valid _id_ of an existing resource for this method to
    # work
    #
    def content_location(id=@id, format=nil, login=nil)
        if login.nil? or format.nil? then
            if id.kind_of? Integer and id > 0 then
                format, login = rdf.select_one %{
SELECT ?format, ?login
WHERE (dc::format #{id} ?format)
      (dc::creator #{id} ?creator)
      (s::login ?creator ?login)
USING PRESET NS}
            else
                raise RuntimeError,
                    _("Can't derive message format and creator login name")
            end
        end
        ext = config['file_extension'][format]
        ext = format.sub(/\A.*\//, '') if ext.nil?
        config['site']['content'] + '/' + login + '/' + id.to_s + '.' + ext
    end

    # multimedia message content filename (wrapper around content_location)
    #
    def content_filename(id, format=nil, login=nil)
        session.filename(content_location(id, format, login))
    end

private

    def render_focuses(head, mode)
        current = Focus.new(self)
        focuses = (:full == mode)? {current.id => current} : {}
        rdf.select_all( %{
SELECT ?focus
WHERE (rdf::subject ?stmt #{id})
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt ?focus)
      (s::rating ?stmt ?rating)
LITERAL ?rating > 0
ORDER BY ?rating DESC
USING PRESET NS}, limit_page).collect {|f,| f }.each {|focus|
            f = Focus.new(self, focus)
            focuses[f.id] = f
        }
        (:full == mode)? template.focus_box(id, head, focuses.values) :
            template.focus_line(id, focuses.values)
    end

    def render_member(mode)
        r = {}
        login, r[:head] = rdf.select_one "
SELECT ?login, ?full_name
WHERE (s::login #{id} ?login)
      (s::fullName #{id} ?full_name)
USING s FOR #{ns['s']}"
        r[:desc] = _('Login') +": #{login}"
        if mode == :full then
            skip, = session.params %w[skip]
            skip = skip.to_i
            messages = rdf.select_all( %{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (dc::creator ?msg #{id})
      (dct::isVersionOf ?msg ?current)
LITERAL ?current IS NULL
ORDER BY ?date DESC
USING PRESET NS}, limit_page, limit_page * skip
            ).collect {|msg,| Resource.new(session, msg).render }
            r[:desc] = template.box(nil, r[:desc])
            r[:related] = template.box( _('Latest Messages') +
                (skip > 0? sprintf(_(', page %s'), skip + 1) : ''),
                template.list(messages,
                    template.nav(skip + 1, %{resource.rb?id=#{id}&amp;}))
            ) if messages.size > 0
        end
        r
    end

    def render_message(mode)
        date, head, creator, name, login = rdf.select_one %{
SELECT ?date, ?title, ?creator, ?name, ?login
WHERE (dc::date #{id} ?date)
      (dc::title #{id} ?title)
      (dc::creator #{id} ?creator)
      (s::fullName ?creator ?name)
      (s::login ?creator ?login)
USING PRESET NS}
        replies, = rdf.select_one "
SELECT count(?msg) WHERE (s::inReplyTo ?msg #{id}) USING PRESET NS"
        focus = render_focuses(head, mode) unless :full == mode or
            Focus == self.class   # avoid recursion
        return { :head => head, :desc => template.message_info(
            :id => id, :date => date, :creator => creator, :full_name => name,
            :focus => focus, :replies => replies) } if :list == mode
        # extra data for :short and :full modes
        desc_id, = rdf.select_one "
SELECT ?desc WHERE (dc::description #{id} ?desc) USING PRESET NS"
        desc_id and desc_id != id and desc =
            Resource.new(session, desc_id).render(:short)[:content]
        format, = rdf.select_one "
SELECT ?format WHERE (dc::format #{id} ?format) USING PRESET NS"
        content, = rdf.select_one "
SELECT ?content WHERE (s::content #{id} ?content) USING PRESET NS"
        content = content_location(id, format, login) if content.nil? and
            format and not config['format']['inline'].include? format
        if mode == :full then   # show parent and versions only in full mode
            current, = rdf.select_one "
SELECT ?current WHERE (dct::isVersionOf #{id} ?current) USING PRESET NS"
            if current.nil? then   # non-current versions don't have history
                parent, = rdf.select_one "
SELECT ?parent WHERE (s::inReplyTo #{id} ?parent) USING PRESET NS"
                history, = rdf.select_one "
SELECT count(?version) WHERE (dct::isVersionOf ?version #{id}) USING PRESET NS"
                if creator != session.id then
                    open, = rdf.select_one "
SELECT ?open WHERE (s::openForAll #{id} ?open) USING PRESET NS"
                else 
                    open = true
                end
            end
        end
        r = { :head => head }
        r[:content] = template.message_content(:title => head,
            :content => content, :format => format, :mode => mode)
        r[:desc] = template.message(:id => id, :date => date,
            :creator => creator, :full_name => name, :focus => focus,
            :title => head, :format => format, :content => content,
            :parent => parent, :replies => replies, :current => current,
            :history => history, :open => open, :desc_id => desc_id,
            :desc => desc, :mode => mode)
        if mode == :full   # add replies
            skip, = session.params %w[skip]
            skip = skip.to_i
            replies = rdf.select_all( %{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (s::inReplyTo ?msg #{id})
ORDER BY ?date
USING PRESET NS}, limit_page, limit_page * skip
            ).collect {|msg,| Resource.new(session, msg).render(:short)[:desc] }
            r[:related] = template.box(
                _('Replies') +
                    (skip > 0? sprintf(_(', page %s'), skip + 1) : ''),
                template.list(replies,
                    template.nav(skip + 1, %{resource.rb?id=#{id}&amp;})),
                'replies'
            ) if replies.size > 0
            # navigation links
            template.link['made'] = creator
            template.link['up'] = parent if parent
        end
        r
    end

    def render_statement(mode)
        r = { :head => _('Statement') + ' ' + id.to_s }
        stmt = rdf.select_one %{
SELECT ?p, ?s, ?o
WHERE (rdf::predicate #{id} ?p)
      (rdf::subject #{id} ?s)
      (rdf::object #{id} ?o)
USING rdf FOR #{ns['rdf']}}
        n = [_('Predicate'), _('Subject'), _('Object')]
        if mode == :list then
            r[:desc] = '(' + stmt.collect {|id|
                %{<a href="#{id}">#{n.shift} #{id}</a>}
            }.join(', ') + ')'
        else
            r[:desc] = stmt.collect {|id|
                template.box(n.shift, Resource.new(session, id).render)
            }.join
        end
        # todo: display votes
        r
    end

    def render_vote(mode)
        r = { :head => _('Vote') + ' ' + id.to_s }
        date, stmt, member, name, rating = rdf.select_one "
SELECT ?date, ?stmt, ?member, ?name, ?rating
WHERE (dc::date #{id} ?date)
      (s::voteProposition #{id} ?stmt)
      (s::voteMember #{id} ?member)
      (s::fullName ?member ?name)
      (s::voteRating #{id} ?rating)
USING PRESET NS"
        r[:desc] = sprintf(_('<a href="%s">%s</a> gave rating %4.2f to the <a href="%s">Statement %s</a> on %s.'), member, name, rating, stmt, stmt, template.format_date(date).to_s)
        if mode == :full then
            r[:desc] = template.box(nil, r[:desc]) +
                template.box(_('Vote Proposition'),
                    Resource.new(session, stmt).render(:short)[:desc])
            template.link['made'] = member
        end
        r
    end

    def render_item(mode)
        # todo: short mode
        msg, date, creator, name, head, format, content, parent,
            contributor, c_name, possessor, p_name = rdf.select_one "
SELECT ?msg, ?date, ?creator, ?name, ?title, ?content, ?parent,
       ?contributor, ?c_name, ?possessor, ?p_name
WHERE (s::description #{id} ?msg)
      (dc::date ?msg ?date)
      (dc::creator ?msg ?creator)
      (s::fullName ?creator ?name)
      (dc::title ?msg ?title)
      (dc::format ?msg ?format)
      (s::content ?msg ?content)
      (s::inReplyTo ?msg ?parent)
      (s::contributor #{id} ?contributor)
      (s::fullName ?contributor ?c_name)
      (s::possessor #{id} ?possessor)
      (s::fullName ?possessor ?p_name)
USING dc FOR http://purl.org/dc/elements/1.1/
      s FOR #{ns['s']}"
        desc =
%'<p><b>Contributor:</b> <a href="#{contributor}">#{c_name}</a></p>
<p><b>Possessor:</b> <a href="#{possessor}">#{p_name}</a></p>
<p><b>Description:</b></p>' +
            template.message(msg, date, creator, name, nil, \
                head, format, content, parent)
        head += ' / ' + possessor
        if mode == :full then
            query = "
SELECT ?item, ?name
WHERE (s::description #{id} ?msg)
      (s::description ?item ?msg)
      (s::possessor ?item ?possessor)
      (s::fullName ?possessor ?name)
USING s FOR #{ns['s']}"
            desc += template.box(_('Similar Items'),
                _('<p><b>Possessed by:</b></p>') +
                db.execute(rdf.select(query)) {|sth|
                    sth.fetch_many(limit_page).collect {|item,name|
                        %'<a href="#{item}">#{name}</a><br>'
                    }.join })
        end
        return head, desc
    end
end
