# Samizdat HTML templates
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'uri/common'

begin
    require 'redcloth'
rescue LoadError
    class RedCloth < String
        def to_html   # revert to text/plain
            "<pre>#{CGI.escapeHTML(self)}</pre>"
        end
    end
end

class Template
    def initialize(session)
        @session = session
        @link = {'start' => session.base}
    end

    # document navigation links (made, start, next, ...)
    attr_accessor :link

    # HTML header, title, and style settings
    #
    def head(title='')
        @link[:pingback] = @session.options['x-pingback'] if
            @session.options['x-pingback']
%{<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN"
    "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en">
<head>
    <title>#{title}</title>
    <meta name="generator" content="Samizdat #{SAMIZDAT_VERSION}"/>
    <link rel="stylesheet" type="text/css" href="css/default.css" media="screen"/>} +
        @link.to_a.collect {|rel, href|
            %{\n    <link rel="#{rel}" href="#{href}" />}
        }.join + %{
</head>
<body>\n}
    end

    # close HTML
    #
    def foot
        %{</body>\n</html>}
    end

    # site head and subhead
    #
    def site_head
        # todo: replace head-left with site logo
%{<a href="#main" style="display:none;">Skip navigation</a>
<div id="head">
    <div id="head-left"><a href="#{@session.base}" title="#{config['site']['name']}">#{config['site']['logo']}</a></div>
    <div id="head-right"><a href="message.rb">}+_('Publish')+%{</a>
        :: <a href="item.rb">}+_('Share')+%{</a>
        :: <a href="query.rb">}+_('Search')+%{</a></div>
</div>
<div id="subhead">
    <div id="subhead-right">#{member_line}</div>
</div>\n}
    end

    # site foot (attributions)
    #
    def main_foot
%{<div id="foot">
Powered by <a href="http://www.fsf.org/">Free Software</a>, including
<a href="http://www.apache.org/">Apache</a> web server,
<a href="http://www.ruby-lang.org/">Ruby</a> programming language,
and <a href="http://www.postgresql.org/">PostgreSQL</a> database.
<a href="http://www.nongnu.org/samizdat/">Samizdat</a> engine is free
software; you can distribute/modify it under the terms of the GNU
<a href="COPYING">General Public License</a> version 2 or later.
</div>}
    end

    # member options when logged on, link to login form otherwise
    #
    def member_line
        settings = %{<a href="member.rb">#{_('Settings')}</a>, }
        if @session.id then
            resource_href(@session.id, @session.full_name) + ': ' +
                settings + %{<a href="logout.rb">#{_('Log out')}</a>}
        else
            settings + %{<a href="login.rb">#{_('Log in')}</a>}
        end
    end

    # wrap content in <div id="sidebar"> if non-empty
    #
    def sidebar(content)
        if content and content != '' then
%{<div id="sidebar">
#{content}
</div>\n}
        else '' end
    end

    # wrap title and content into a CSS-rendered box
    #
    def box(title, content, id=nil)
        box_title = %{<div class="box-title">#{title}</div>} if title
        box_id = %{ id="#{id}"} if id
%{<div class="box"#{box_id}>
    #{box_title}<div class="box-content">
#{content}
    </div>
</div>\n}
    end

    # navigation link to a given page number
    #
    def nav(skip, script=File.basename(@session.script_name)+'?', name='skip')
%{<div class="nav"><a href="#{script}#{name}=#{skip}">}+_('next page')+%{</a></div>\n}
    end

    # resource list with navigation link
    #
    def list(list, nav, foot='')
        even = 1
        %{<ul>\n} +
        list.collect {|li|
            even = 1 - even
            %{<li#{' class="even"' if even == 1}>#{li}</li>\n}
        }.join + %{</ul>\n<div class="foot">\n} +
        foot + ((list.size < limit_page)? '' : nav) + %{</div>\n}
    end

    # resource table with navigation link
    #
    def table(table, nav, foot='')
        even = 1
        %{<table>\n<thead><tr>\n} +
        table.shift.collect {|th| "<th>#{th}</th>\n" }.join +
        %{</tr></thead>\n<tbody>\n} +
        table.collect {|row|
            even = 1 - even   # todo: a CSS-only way to do this
            %{<tr#{' class="even"' if even == 1}>\n} + row.collect {|td|
                "<td>#{td or '&nbsp;'}</td>\n"
            }.join + "</tr>\n"
        }.join +
        %{</tbody></table>\n<div class="foot">\n} +
        foot + ((table.size < limit_page)? '' : nav) + %{</div>\n}
    end

    # type can be any of the following:
    #
    # [:label]
    #   wrap label _value_ in a <div> tag and associate it with _name_ field
    # [:textarea] fixed text area 70x20 with _value_
    # [:select] _value_ is an array of options or pairs of [option, label]
    # [:submit] _value_ is a button label
    # [standard HTML input type] copy _type_ as is into <input> tag
    #
    def form_field(type, name=nil, value=nil, default=nil)
        value = CGI.escapeHTML(value) if value.class == String
        name = %{ name="#{name}"} if name
        name += ' disabled=""' if name and :disabled == default
        case type
        when :label   # todo: generate field idref for label for attribute
%{<div class="label"><label>#{value}</label></div>\n}
        when :textarea
%{<textarea#{name} cols="70" rows="20">#{value}</textarea>\n}
        when :select
%{<select#{name}>\n} + value.collect {|option|
    v, l = (option.class == Array)? option : [option, option]
    selected = (v == default)? ' selected="selected"' : ''
    %{    <option#{selected} value="#{v}">#{l}</option>\n}
}.join + "</select>\n"
        when :submit
            value = _('Submit') if value.nil?
%{<input#{name} type="submit" value="#{value}" class="submit"/>\n}
        else
            if :checkbox == type then
                name += ' checked=""' if value
                value = 'true'
            end
%{<input#{name} type="#{type}" value="#{value}"/>\n}
        end
    end

    # wrap a list of form fields into a form (see form_field)
    #
    # automatically detects if multipart/form-data is necessary
    #
    def form(action, *fields)
        if fields.assoc(:file) then
            enctype = ' enctype="multipart/form-data"'
        end
        %{<form action="#{action}" method="post"#{enctype}><div>\n} +
        fields.collect {|param| form_field(*param) }.join + "</div></form>\n"
    end

    # wrap focus name and rating in <span> tags
    #
    def focus_info(id, focus, current_id=Focus.new(focus.resource).id)
        return nil if focus.uriref == focus.resource.uriref
        name = focus.name
        name += ' ('+ _('Your Current Focus') +')' if focus.id == current_id
        name = resource_href(focus.uriref, name) if focus.id.to_i > 0
%{#{name}: #{focus.to_s} (<a title="#{_('Click to vote on how this resource is related to this focus')}" href="resource.rb?id=#{id}&amp;related=#{CGI.escape(focus.id.to_s)}">#{_('vote')}</a>)}
    end

    # list supplied focuses using focus_info
    #
    def focus_box(id, title, focuses)
        return '' unless focuses.class == Array
        current_id = Focus.new(focuses[0].resource).id
        box( _('Related Focuses'), focuses.sort {|a, b|
            b.sort_index <=> a.sort_index
        }.collect {|focus|
            %{<p>#{focus_info(id, focus, current_id)}</p>\n}
        }.join, 'focuses') +
        box(nil, %{<p><a title="#{_('To vote on how other resources are related to this one, focus on it')}" href="resource.rb?set_focus=#{id}">} +
            _('Focus on this resource') + %{ (#{title})</a></p>})
    end

    # list supplied focuses in a straight line
    def focus_line(id, focuses)
        %{<a href="#{id}#focuses">#{_('related to')}</a>: } +
        focuses.sort {|a, b|
            b.sort_index <=> a.sort_index
        }.collect {|focus|
            resource_href(focus.uriref, focus.name)
        }.join(', ') if focuses.size > 0
    end

    # focus rating vote form
    #
    def vote_form(focus)
        form( 'resource.rb',
            [:hidden, 'id', focus.resource.id],
            [:hidden, 'related', focus.id],
            [:label, 'rating', _('Give a rating to how this resource is related to focus: %s') % focus.name],
            [:select, 'rating', [
                [-2, _('-2 (No)')],
                [-1, _('-1 (Not Likely)')],
                [0, _('0 (Uncertain)')],
                [1, _('1 (Likely)')],
                [2, _('2 (Yes)')] ], 0],
            [:label], [:submit, nil, _('Submit')]
        )
    end

    # transform date to a standard string representation
    #
    def format_date(date)
        date = date.to_time if date.methods.include? 'to_time'   # duck
        date = date.strftime '%Y-%m-%d %H:%M' if date.kind_of? Time
        date
    end

    # render link to resource with a tooltip
    #
    def resource_href(id, title)
        %{<a title="#{_('Click to view the resource')}" href="#{id}">#{CGI.escapeHTML(title.to_s)}</a>}
    end

    # render resource description for resource listing
    #
    def resource(id, title, info)
%{<div class="resource">
<div class="title">#{resource_href(id, title)}</div>
<div class="info">#{info}</div>
</div>\n}
    end

    # render message info line, _params_ is inherited from #message
    #
    def message_info(params)
        date = format_date(params[:date])
        parent = %{<a href="#{params[:parent]}">} +
            _('parent message') + '</a>' if params[:parent]
        replies = %{<a href="#{params[:id]}#replies">} + _('replies') +
            '</a>:&nbsp;' + params[:replies].to_s if params[:replies].to_i > 0
        current = %{<a href="#{params[:current]}">} + _('current version') +
            '</a>' if params[:current]
        history = %{<a href="history.rb?id=#{params[:id]}">} + _('history') +
            '</a>' if params[:history].to_i > 0
        [ sprintf( _('by&nbsp;<a href="%s">%s</a> on&nbsp;%s'),
            params[:creator], CGI.escapeHTML(params[:full_name]), date.to_s
        ), parent, current, history, replies, params[:focus] ].compact.join(",\n ")
    end

    # render message content, _params_ is inherited from #message
    #
    def message_content(params)
        return nil if :list == params[:mode] or
            (params[:desc] and :short == params[:mode])
        content = params[:content].to_s
        case params[:format]
        when nil   # default text rendering
            CGI.escapeHTML(content).split(/^\s*$/).collect {|p|
                '<p>' + p + "</p>\n"
            }.join
        when 'text/plain'   # inline verbatim text
            "<pre>#{CGI.escapeHTML(content)}</pre>"
        when 'text/uri-list'   # RFC2483 commented URI list
            '<p>' + content.split("\n").collect {|line|
                case line
                when /\A#\s*(.*?)\z/
                    CGI.escapeHTML($1) + '<br/>'
                when URI::URI_REF
                    line = CGI.escapeHTML(line)
                    %{<a href="#{line}">#{line}</a><br/>}
                end
            }.join + '</p>'
        when 'application/x-textile'   # textile formatted text
            RedCloth.new(content).to_html   # todo: filter_html
        when 'application/x-squish'   # inline query form
            form('query.rb',
                [:textarea, 'query', content], [:label],
                [:submit, 'run', _('Run')])
        when /^image\//   # <img/> reference
            %{<img alt="#{CGI.escapeHTML(params[:title].to_s)}" src="#{content}"/>}
        else   # <a/> reference
            %{<p><a href="#{content}">} +
                sprintf(_('%s content'), params[:format]) + '</a></p>'
        end
    end

    # render full message
    #
    # _params_ is a Hash with keys :id, :date, :creator, :full_name, :focus,
    # :title, :format, :content, :parent, :desc_id, :desc, :mode
    #
    def message(params) 
        info = message_info(params)
        content = message_content(params)
        params[:desc] and content =
            case params[:mode]
            when :full
                box(resource_href(params[:desc_id], _('Description')),
                    params[:desc], 'desc') + content
            when :short
                params[:desc] + %{<p><a href="#{params[:id]}">} +
                    _('See the full message') + '</a></p>'
            end
        buttons = %{<div class="foot">\n} + form( 'message.rb',
            [:hidden, 'title', params[:title]],
            [:hidden, 'parent', params[:id]],
            [:submit, 'reply', _('Reply')],
            [:submit, 'edit', _('Edit'), (params[:open]? nil : :disabled )]
        ) + %{</div>\n} if :full == params[:mode] and not params[:current]
%{<div class="message" id="#{params[:id]}">
<div class="title">#{resource_href(params[:id], params[:title])}</div>
<div class="info">#{info}</div>
<div class="content">#{content}</div>
#{buttons}
</div>\n}
    end

    # language selection box
    #
    def language_box
        return '' unless defined?(GetText)
        old_language = @session.language
        list = ''
        config['locale']['languages'].each do |lang|
            lang.untaint
            @session.language = lang
            list += %{<a href="member.rb?set_lang=#{lang}">} +
                _('English') + "</a>\n"
        end
        @session.language = old_language
        box(_('Change Interface Language'), list)
    end

    # calculate the table of differences
    #
    # return Array of changesets [[left, right], [same], ... ]
    #
    def diff_calculate(old, new)
        begin
            require 'algorithm/diff'
        rescue LoadError
            return [[old.join, new.join]]
        end
        table = []
        offset = 0
        i = 0
        diff = old.diff(new)   # [[op, pos, [lines]], ... ]
        while i < old.size or diff.size > 0 do
            if 0 == diff.size or
            i < diff[0][1] - ((diff[0][0] == :+)? offset : 0) then   # copy
                if table[-1] and 1 == table[-1].size then
                    table[-1][0] << old[i]
                else
                    table.push [old[i]]
                end
                i += 1
                next
            end
            if diff[0][0] == :- then
                if diff[1] and diff[1][0] == :+ and
                diff[1][1] - offset == diff[0][1] then # replace
                    table.push [diff[0][2], diff[1][2]]
                    offset += diff[1][2].size - diff[0][2].size
                    i += diff[0][2].size
                    diff.slice!(0, 2)
                else   # delete
                    table.push [diff[0][2], []]
                    offset -= diff[0][2].size
                    i += diff[0][2].size
                    diff.shift
                end
            else   # add
                table.push [[], diff[0][2]]
                offset += diff[0][2].size
                diff.shift
            end
        end
        table
    end

    # render difference table
    #
    # _old_ and _new_ are Hashes with keys :info, :format, :content
    #
    def diff(old, new)
        [old, new].each do |msg|
            msg[:format] = 'text/plain' if
                'application/x-squish' == msg[:format]
            msg[:content] =
                case msg[:format]
                when nil
                    msg[:content].split("\n\n").collect {|s| s << "\n\n" }
                when 'text/plain', 'text/uri-list',
                'application/x-textile', 'application/x-squish'
                    msg[:content].split("\n").collect {|s| s << "\n" }
                else
                    [msg[:content]]
                end
        end
        %{<table class="diff">\n<thead><tr>\n} +
        [old, new].collect {|msg| "<th>#{msg[:info]}</th>\n" }.join +
        %{</tr></thead>\n<tbody>\n} +
        diff_calculate(old[:content], new[:content]).collect {|left, right|
            width = right ? ' width="50%"' : ' colspan="2"'
            left = {:format => old[:format], :content => left,
                :class => (right ? ' class="delete"' : '')} if left
            right = {:format => new[:format], :content => right,
                :class => ' class="add"'} if right
            %{<tr>\n} + [left, right].compact.collect {|line|
%{<td#{width}#{line[:class] if line[:content].size > 0}><div class="content">#{message_content(line)}#{'&nbsp;' unless line[:content].size > 0}</div></td>\n}
            }.join + "</tr>\n"
        }.join + %{</tbody></table>\n}
    end

    # wrap page title and body with heads and foots
    #
    # body can be String or Array of pairs [title, body]; in latter case, title
    # can be defaulted to the title of the first pair
    #
    def page(title, body)
        body = [[title, body]] unless body.class == Array
        title = body[0][0] if title.nil?
        main = body.collect {|t, b| box(CGI.escapeHTML(t.to_s), b) }.join
        head(config['site']['name'] + ': ' + title.to_s) + site_head() +
            %{\n<div id="main">\n} + main + main_foot + "\n</div>\n" + foot
    end
end
