#!/usr/bin/env ruby
#
# Samizdat member registration and preferences
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'samizdat'

LOGIN_PATTERN = Regexp.new(/\A[a-zA-Z0-9]+\z/).freeze
EMAIL_PATTERN = Regexp.new(/\A[[:alnum:]._-]+@[[:alnum:].-]+\z/).freeze

def send_mail(to, subject, body)
    if to =~ EMAIL_PATTERN then
        to.untaint
    else
        raise UserError, sprintf(_("Malformed email address: '%s'"), email)
    end
    message_id = config['email']['address'].sub(/^[^@]*/,
        Time.now.strftime("%Y%m%d%H%M%S." + Process.pid.to_s))
    IO.popen(config['email']['sendmail'].untaint+' '+to, 'w') do |io|
        io.write(
%{From: Samizdat <#{config['email']['address']}>
To: #{to}
Subject: #{subject}
Message-Id: <#{message_id}>

} + body)
    end
    0 == $? or raise RuntimeError, _('Failed to send email')
end

def confirm_hash(login)
    digest(login + Time.now.to_s)
end

def request_confirmation(session, email, hash, action)
    send_mail(email, 'CONFIRM ' + hash,
%{Site: #{session.base}

#{action}

To confirm this action, visit the following web page:

#{session.base + 'member.rb?confirm=' + hash}

To cancel this action, ignore this message.
})
end

# check in confirmed changes
#
def accept_confirmation(session, confirm)
db.transaction do |db|
    login, = db.select_one 'SELECT login FROM Member
        WHERE confirm = ?', confirm
    login.nil? and raise UserError, _('Confirmation hash not found')
    session.login and session.login != login and raise UserError,
        _('This confirmation hash is intended for another user')
    passwd = session.prefs(login)['passwd'] and db.do(
        'UPDATE Member SET passwd = ? WHERE login = ?', passwd, login)
    email = session.prefs(login)['email'] and db.do(
        'UPDATE Member SET email = ? WHERE login = ?', email, login)
    session.prefs(login).delete('passwd')
    session.prefs(login).delete('email')
    session.save_prefs(login)
end
    session.template.page(_('Confirmation Accepted'),
        '<p>'+_('Changes confirmed.')+'</p>')
end

# recover lost password
#
def lost_password(session, login)
    t = session.template
    if login and login =~ LOGIN_PATTERN then
        p = ''; 1.upto(10) { p << (?a + rand(26)).chr }   # random password
db.transaction do |db|
        email, = db.select_one 'SELECT email FROM Member WHERE login = ?', login
        email or raise UserError, _('Wrong login')
        confirm = confirm_hash(login)
        session.prefs(login)['passwd'] = digest(p)
        session.prefs(login).delete('email')   # don't change email
        session.save_prefs(login, confirm)
        request_confirmation(session, email, confirm,
            %{New password was generated for your account: } + p)
end # transaction
        return t.page(_('Recover Lost Password'),
            '<p>'+_('New password has been sent to you.')+'</p>')
    end
    t.page(_('Recover Lost Password'), t.form('member.rb',
        [:label, 'login', _('Login')], [:text, 'login'],
        [:submit, 'recover']))
end

# check full_name and email for duplicates
#
def check_duplicates(session, full_name, email)
    suffix = session.id ? (' AND id != ' + session.id.to_s) : ''
    if full_name and full_name != session.full_name then
        id, = db.select_one 'SELECT id FROM Member WHERE full_name = ?' +
            suffix, full_name
        id.nil? or raise UserError,
            _('Full name you have specified is used by someone else')
    end
    if email and email != session.email then
        id, = db.select_one 'SELECT id FROM Member WHERE email = ?' +
            suffix, email
        id.nil? or raise UserError,
            _('Email address you have specified is used by someone else')
    end
end

# change existing account
#
def change_account(session, full_name, email, passwd)
    t = session.template
db.transaction do |db|   # AutoCommit assumed off
    check_duplicates(session, full_name, email)
    if full_name and full_name != session.full_name then
        db.do 'UPDATE Member SET full_name = ? WHERE id = ?',
            full_name, session.id
    end
    if passwd then
        db.do 'UPDATE Member SET passwd = ? WHERE id = ?',
            digest(passwd), session.id
    end
    if email and email != session.email then
        if config['email'] then   # with confirmation
            confirm = confirm_hash(session.login)
            session.prefs['email'] = email
            session.prefs.delete('passwd')   # don't change password
            session.save_prefs(session.login, confirm)
            request_confirmation(session, email, confirm,
                'Your email address was specified for an account.')
            db.commit   # mind the transaction!
            return t.page(_('Confirmation Requested'),
'<p>'+_('Confirmation request is sent to your new email address.')+'</p>')
        else   # without confirmation
            db.do 'UPDATE Member SET email = ? WHERE id = ?',
                email, session.id
        end
    end
end # transaction
    t.page _('Changes Accepted'),
        '<p>'+_("Press 'Back' button of your browser to return.")+'</p>'
end

# create new account
#
def create_account(session, login, full_name, email, passwd)
db.transaction do |db|   # AutoCommit assumed off
    check_duplicates(session, full_name, email)

    # validate form
    (login and full_name and email and passwd) or raise UserError,
        _("You didn't fill all mandatory fields")
    login == 'pingback' and raise UserError,
        _('Login name you specified is reserved for Pingback server')
    login =~ LOGIN_PATTERN or raise UserError,
        _('Use only latin letters and numbers in login name')
    id, = db.select_one 'SELECT id FROM Member WHERE login = ?', login
    id.nil? or raise UserError,
        _('Login name you specified is already used by someone else')

    # create account
    p = digest(passwd)
    db.do 'INSERT INTO Member (login, full_name, email, passwd)
        VALUES (?, ?, ?, ?)', login, full_name, email,
        (config['email'] ? nil : p)
    if config['email'] then   # request confirmation
        confirm = confirm_hash(login)
        session.prefs(login)['email'] = email
        session.prefs(login)['passwd'] = p
        session.save_prefs(login, confirm)
        request_confirmation(session, email, confirm,
            'Your email address was used to create an account.')
    end
end # transaction
    session.open(login, passwd) or raise RuntimeError,
        _('Login error: failed to open session for new account')
end

request do |session|
    set_lang, set_style, set_moderate =
        session.params %w[set_lang set_style set_moderate]
    if set_lang then
        session.set_cookie('lang', set_lang)
        session.redirect
    end
    if set_style then
        session.set_cookie('style', set_style)
        session.redirect
    end
    if set_moderate then
        session.set_cookie('moderate', set_moderate,
            config['session']['moderate_timeout'])
        session.redirect
    end

    if session.moderator then
        id, block, unblock = session.params %w[id block unblock]
        if id.to_i > 0 then
db.transaction do |db|
            login, passwd, prefs = db.select_one(
                'SELECT login, passwd, prefs FROM Member WHERE id = ?', id)
            raise ResourceNotFoundError, 'Account does not exist' if login.nil?
            raise UserError, _('Moderator accounts can not be blocked') if
                config['site']['moderators'].include? login
            prefs = yaml_hash(prefs)
            db.do 'INSERT INTO Moderation (moderator, resource) VALUES (?, ?)',
                session.id, id
            if block and not passwd.nil? then
                prefs['blocked_by'] = session.id
                prefs['passwd'] = passwd   # remember password
                db.do 'UPDATE Member SET passwd = NULL, session = NULL,
                    prefs = ? WHERE id = ?', YAML.dump(prefs), id
            elsif unblock and passwd.nil? and prefs['passwd'] then
                prefs.delete('blocked_by')
                passwd = prefs.delete('passwd')   # restore password
                db.do 'UPDATE Member SET passwd = ?, prefs = ? WHERE id = ?',
                    passwd, YAML.dump(prefs), id
            end   # do nothing in unexpected situations
end
            session.redirect(id)
        end
    end

    confirm, = session.params %w[confirm]
    confirm and next accept_confirmation(session, confirm)

    # account parameters
    login, full_name, email, passwd1, passwd2 = \
        session.params %w[login full_name email passwd1 passwd2]

    # recover lost password
    config['email'] and session.has_key?('recover') and session.id.nil? and
        next lost_password(session, login)

    # validate password
    raise UserError, _('Passwords do not match') if passwd1 != passwd2

    # validate email
    raise UserError, sprintf(_("Malformed email address: '%s'"), email) if
        email and email !~ EMAIL_PATTERN

    if session.has_key? 'submit' then
        page = session.id ? change_account(session, full_name, email, passwd1) :
            create_account(session, login, full_name, email, passwd1)
        next page if page
    end

    # account form
    #
    if session.id then   # change existing account
        form = []
        full_name = session.full_name unless full_name
        email = session.email unless email
    else   # new account
        form = [[:label, 'login', _('Login')], [:text, 'login', login]]
    end

    form.push(
        [:label, 'full_name', _('Full name')], [:text, 'full_name', full_name],
        [:label, 'email', _('Email')], [:text, 'email', email],
        [:label, 'passwd1', _('Password')], [:password, 'passwd1'],
        [:label, 'passwd2', _('Reenter password to confirm')],
            [:password, 'passwd2'],
        [:label], [:submit, 'submit']
    )

    t = session.template

    style = ''
    config['style'].each {|s, name|
        style << %{<a href="member.rb?set_style=#{s}">#{name}</a>\n}
    }
    style = t.box(_('Change Theme'), style)

    if config['site']['moderators'].include? session.login then
        moderation = t.box(_('Moderation Facility'), (session.moderator ?
            %{<a href="member.rb?set_moderate=no">} + _('Disable') + '</a>' :
            %{<a href="member.rb?set_moderate=yes">} + _('Enable') + '</a>') )
    end

    title = session.id ? _('Change Account Settings') : _('Create New Account')
    t.page(_('Member Settings'),
        t.box(title, t.form('member.rb', *form)) +
        t.language_box + style + moderation.to_s)
end
