#!/usr/bin/env ruby
#
# Samizdat publish message form
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'uri'
require 'ftools'
require 'samizdat'

# write message into the database, return its id
#
def write_message(resource, creator, title, format, content, parent, desc, open, action)
    content.gsub!(/\r\n/, "\n") if content
    # todo: detect duplicates
    # todo: translate into RDF assert
    if :takeover == action or :displace == action then   # moderation log
        db.do 'INSERT INTO Moderation (moderator, resource) VALUES (?, ?)',
            creator, parent
    end
    if :edit == action or :takeover == action then
        parent or raise UserError, _('Reference to previous version lost')
        # save current version at new id
        db.do 'INSERT INTO Message (version_of, creator, title, format,
            description, content, open) SELECT id, creator, title, format,
            description, content, open FROM Message WHERE id = ?', parent
        db.do "UPDATE Resource SET published_date =
            (SELECT published_date FROM Resource WHERE id = ?)
            WHERE id = currval('Resource_id_seq')", parent
        # check if old version had content in a file
        p_format, = db.select_one "SELECT format FROM Message
            WHERE id = ?", parent
        if p_format and not config['format']['inline'].include? p_format then
            p_filename = resource.content_filename(parent.to_i).untaint
        end
        # write new version at old id
        db.do 'UPDATE Message SET creator = ?, title = ?,
            format = ?, description = ?, content = ? WHERE id = ?',
            creator, title, format, desc, content, parent
        db.do "UPDATE Resource SET published_date = 'now' WHERE id = ?",
            parent
        if p_filename then   # rename old file
            id, = db.select_one "SELECT currval('Resource_id_seq')"
            File.rename(p_filename, resource.content_filename(id.to_i).untaint)
        end
        if :takeover == action then   # disable s:openForAll
            db.do 'UPDATE Message SET open = false WHERE id = ?', parent
        end
        id = parent
    elsif :displace == action then
        # displace content without saving previous version
        db.do 'UPDATE Message SET creator = ?, title = ?, format = ?,
            description = ?, content = ?, open = false WHERE id = ?',
            creator, title, format, desc, content, parent
        id = parent
    else   # new message
        db.do 'INSERT INTO Message (creator, title, format, description,
            content, open, parent) VALUES (?, ?, ?, ?, ?, ?, ?)',
            creator, title, format, desc, content, (open or false), parent
        id = db.select_one("SELECT currval('Resource_id_seq')")[0]
    end
end

# attempt Pingback on the first URI
#
def register_pingback(source, content)
    catch :fail do
        begin
            require 'net/http'
            require 'xmlrpc/client'
        rescue LoadError
            throw :fail
        end
        throw :fail unless content =~ URI::ABS_URI
        uri, scheme = $&, $1
        throw :fail unless scheme =~ /^http/
        response = Net::HTTP.get_response(URI.parse(uri.untaint))
        throw :fail unless response.kind_of? Net::HTTPSuccess
        pingback = response['x-pingback']
        if pingback.nil? then
            throw :fail unless response.body =~
                %r{<link rel="pingback" href="([^"]+)" ?/?>}
            pingback = $1
            pingback.gsub!(/&amp;/, '&')
            pingback.gsub!(/&lt;/, '<')
            pingback.gsub!(/&gt;/, '>')
            pingback.gsub!(/&quot;/, '"')
        end
        throw :fail unless pingback =~ URI::ABS_URI
        u = URI.parse(pingback.untaint)
        server = XMLRPC::Client.new(u.host, u.path, u.port)
        server.call2("pingback.ping", source, uri)
        # discard the result
        # todo: notify user that ping was registered
    end
end

request do |session|
    session.id or raise AuthError,
        _('You should be <a href="login.rb">logged in</a> to post messages')

    t = session.template
    resource = Resource.new(session)   # utility object
    title, content, format, desc, open, parent, action =
        session.params %w[title content format desc open parent action]
    file = session['file']   # just a file object, not its contents
    file = nil if file and 0 == file.size

    # input validation
    raise UserError, _('Bad input') if parent and 0 == parent.to_i
    if parent then   # don't allow to edit or reply to old versions
        session[action] = true if
            action and %w[reply edit takeover displace].include? action
        action =
            if session.has_key?('edit') then
                :edit
            elsif session.moderator and session.has_key?('takeover') then
                :takeover
            elsif session.moderator and session.has_key?('displace') then
                :displace
            else
                :reply
            end
        current, = rdf.select_one "
SELECT ?current WHERE (dct::isVersionOf #{parent} ?current) USING PRESET NS"
        raise UserError, sprintf(_('Use only <a href="%s">current version</a> for replies and edits'), current) if current and :displace != action
        if :edit == action then   # check if message is open for editing
            owner, open = rdf.select_one "
SELECT ?creator, ?open
WHERE (dc::creator #{parent} ?creator)
      (s::openForAll #{parent} ?open)
USING PRESET NS"
            raise UserError, _('You are not allowed to edit this message') if
                owner != session.id and not open
        end
    end
    edit = [:edit, :takeover, :displace].include? action

    # check description
    if desc then   # desc should refer to an existing message
        desc.gsub!(Regexp.new('\A'+Regexp.quote(session.base)+'(\d+)\z'), '\1')
        raise UserError, _('Invalid reference to description') if
            desc !~ /\A\d+\z/ or (edit and desc.to_i == parent.to_i)
        label, = db.select_one('SELECT label FROM Resource WHERE literal = false AND uriref = false AND id = ?', desc.to_i)
        raise UserError, _('Description should refer to an existing message') unless 'Message' == label
    end

    # detect and check format
    if format.nil? and file.methods.include? 'content_type' and file.size > 0
        format = file.content_type.strip   # won't work with FastCGI
        format = 'image/jpeg' if format == 'image/pjpeg'   # MSIE...
    end
    raise UserError, sprintf(_("Format '%s' is not supported"), format) if
        format and not config['format'].values.flatten.include? format
    inline = (format.nil? or config['format']['inline'].include? format)

    if file and file.size > 0 then   # content uploaded from file
        content = nil
        format.nil? and raise UserError,
            _('It is not possible to upload a file without specifying format')
        # todo: fine-grained size limits
        file.size > config['limit']['content'] and raise UserError,
            sprintf(_('Uploaded file is larger than %s bytes limit'),
                config['limit']['content'])

        location = resource.content_location('upload', format, session.login)
        upload = session.filename(location)
        upload.untaint   # security: keep format and login controlled

        if (file.kind_of? StringIO or file.kind_of? Tempfile) and
        not session.has_key? 'confirm' then   # new upload
            if inline then
                content = file.read   # transform to inline message
                file = nil
            else
                config['site']['content'].nil? and raise UserError,
                    _('Multimedia upload is disabled on this site')
                File.makedirs(File.dirname(upload)) unless
                    File.exists?(File.dirname(upload))
                if file.kind_of? Tempfile then   # copy large files directly
                    file = File.syscopy(file.path, upload)
                else   # StringIO
                    File.open(upload, 'w') {|f| f.write(file.read) }
                    file = true
                end
            end

        elsif file.kind_of? String and session.has_key? 'confirm' then
            inline and raise UserError, 'Unexpected inline upload confirm'
            file = nil unless File.exists?(upload)
        else
            raise UserError, 'Unexpected upload state'
        end
    end   # at this point, file is true and content is nil if upload is ready

    if content then   # inline message
        file = nil
        inline or raise UserError,
            sprintf(_('You should upload %s content from file'), format)
        raise UserError,
            _('text/uri-list should contain at least one absolute URI') if
                format == 'text/uri-list' and not content =~ URI::ABS_URI
    end

    if title and (content or file) then   # write or preview message
        if session.has_key? 'confirm' then
            id = nil   # scope fix
            db.transaction do |db|
                id = write_message(resource, session.id, title, format,
                    content, parent, desc, open, action)
                if file then
                    # todo: inject file into p2p net
                    File.rename(upload, resource.content_filename(
                        id, format, session.login).untaint)
                end
            end   # transaction
            if format == 'text/uri-list' then
                register_pingback(session.base + id.to_s, content)
            end
            session.redirect(id.to_s)   # redirect to published message

        else   # preview message
            content = location if file   # preview upload.ext
            warning = '<p>'+sprintf(_('Warning: content is longer than %s characters. In some situations, it will be truncated.'), config['limit']['short'])+'</p>' if content != t.limit_string(content, config['limit']['short'])
            desc_content = Resource.new(
                session, desc).render(:short)[:content] if desc
            t.page( _('Message Preview'),
                t.message(:id => 0, :date => Time.now, :creator => session.id,
                    :full_name => session.full_name, :title => title,
                    :format => format, :content => content, :parent => parent,
                    :desc_id => desc, :desc => desc_content,
                    :mode => :preview) +
                warning.to_s +
                '<p>'+_("Press 'Back' button to change the message.")+'</p>' +
                t.form( 'message.rb',
                    [:submit, 'confirm', _('Confirm')],
                    [:hidden, 'title', title],
                    [:hidden, 'content', file ? nil : content],
                    [:hidden, 'file', file],
                    [:hidden, 'format', format],
                    [:hidden, 'desc', desc],
                    [:hidden, 'open', open],
                    [:hidden, 'parent', parent],
                    [:hidden, 'action', action.to_s]
                )
            )
        end

    else   # edit message
        if edit then   # show current content if inline
            format, desc = rdf.select_one "
SELECT ?format, ?desc
WHERE (dc::format #{parent} ?format)
      (dc::description #{parent} ?desc)
USING PRESET NS"
            if format.nil? or config['format']['inline'].include? format then
                content, = rdf.select_one "
SELECT ?content WHERE (s::content #{parent} ?content) USING PRESET NS"
            end
            # todo: message locking
        end
        head =
            case action
            when :reply then 'Reply'
            when :edit then 'Edit Message'
            when :takeover then 'Take Over Message'
            when :displace then 'Displace Message'
            else 'New Message'
            end
        comment =
            case action
            when :reply
                t.box(_('Parent Message'),
                    Resource.new(session, parent).render(:short))
            when :takeover
                '<p><strong>' + _('Only you will be able to edit this message after takeover.') + '</strong></p>'
            when :displace
                '<p><strong>' + _('MESSAGE WILL BE COMPLETELY DISPLACED, NO RECOVERY WILL BE POSSIBLE. PLEASE PROVIDE DETAILED JUSTIFICATION FOR THIS ACTION.') + '</strong></p>'
            else ''
            end
        t.page(_(head), t.form('message.rb',
            [:label, 'title', _('Title')], [:text, 'title', title],
            [:label, 'content', _('Content')], [:textarea, 'content', content],
            [:label, 'file', _('Upload content from file')],
                [:file, 'file', file],
            [:label, 'format', _('Format')],
                [:select, 'format', [[nil, _('Default')]] +
                    config['format']['inline'].to_a, format],
            [:label, 'desc',
                _('Reference to description (ID or URL of another message on this site)')],
                [:text, 'desc', desc],
            [:label, 'open', _('Editing is open for all members')],
                [:checkbox, 'open', open, (edit ? :disabled : nil)],
            [:label], [:submit, 'preview', _('Preview')],
            [:hidden, 'parent', parent],
            [:hidden, 'action', action.to_s]) + comment)
    end
end
