# Samizdat focus handling
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

class Focus < Resource
    # get focus id from current focus if not supplied; get current focus from
    # config if focus cookie is not set; derive focus resource id from uriref
    # when necessary; keep external uriref if applies
    #
    # extract readable focus name from resource title or focus uriref
    #
    def initialize(resource, id=nil)   # todo: should unit-test this beast
        if id.nil? then   # current focus
            id = resource.session.cookie('focus')
            id = config['focus'][0] if id.nil?   # use default
        end
        if id.to_i == 0 then   # derive id from uriref
            uriref = id
            id, = rdf.select_one(
                %{SELECT ?focus WHERE (s::id #{uriref} ?focus) USING PRESET NS})
        end
        super resource.session, id   # todo: unchild from Resource
        @id = @uriref = uriref if id.nil? and uriref   # keep external uriref

        if @id.kind_of? Integer then
            begin
                @name = render(:title)   # watch out for recursion!
            rescue ResourceNotFoundError   # bad id in focus cookie
                session.set_cookie('focus', nil, 0)   # delete bad cookie
                session.redirect(session.base)
            end
        else
            @name = uriref.sub(
                /\A(focus::|#{ns['focus']}|.*?[\/#](?=[^\/#]+\z))/, '')
        end
        @name = _(@name)   # translate focus names
        @resource = resource
    end

    attr_reader :resource, :name

    # rating is an integer from -2 to 2
    #
    def Focus.validate_rating(rating)
        rating = rating.to_i if rating
        (rating and rating >= -2 and rating <= 2)? rating: nil
    end

    # read and cache rating from SamizdatRDF
    #
    def rating
        if @rating.nil? then
            @rating, = rdf.select_one %{
SELECT ?rating
WHERE (rdf::subject ?stmt #{resource.id})
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{@id})
      (s::rating ?stmt ?rating)
USING PRESET NS}
            @rating = @rating ? @rating.to_f : _('none')
        end
        @rating
    end

    # update rating in SamizdatRDF and in memory
    #
    def rating=(value)
        value = Focus.validate_rating(value)
        if value then
            rdf.assert %{
UPDATE ?rating = '#{value}'
WHERE (rdf::subject ?stmt #{resource.id})
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{@id})
      (s::voteProposition ?vote ?stmt)
      (s::voteMember ?vote #{session.id})
      (s::voteRating ?vote ?rating)
USING PRESET NS}
            @rating = nil   # invalidate rating cache
        end
    end

    # order by rating, unrated after rated
    #
    def sort_index
        rating.kind_of?(Numeric) ? rating : -100
    end

    # print focus rating
    #
    def to_s
        case rating
        when Numeric then "%4.2f" % rating
        when String then rating
        else rating.to_s
        end
    end
end
