#!/usr/bin/env ruby
#
# Samizdat functional regression test
#
#   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'test/unit'
require 'net/http'
require 'uri'
require 'rexml/document'
require 'test/util'
require 'samizdat'

# WARNING: this test will DESTROY DATA in your samizdat database!
#
# it assumes that samizdat database is empty: if you have any data in your
# site, back it up and recreate a clean database using database/*.sql
#
class TC_Robot < Test::Unit::TestCase
    STAMP = Time.new.to_i.to_s

    def setup
        @base = URI.parse('http://localhost' + config['site']['base'] + '/')
        @login = 'test' + STAMP
        @full_name = 'Test' + STAMP
        @email = 'test' + STAMP + '@localhost'
        @passwd = 'test'
    end

    def teardown
        @base, @login, @full_name, @email, @passwd = nil
    end

    # utility methods

    def post(action, data, header=nil)
        Net::HTTP.start(@base.host) do |http|
            http.post(@base.path + action, data, header)
        end
    end

    def get(action, header=nil)
        Net::HTTP.start(@base.host) do |http|
            http.get(@base.path + action, header)
        end
    end

    # order-sensitive tests

    def test_00_anonymous
        # load front page
        assert_equal Net::HTTPOK, (response = get('')).class
        root = parse(response.body)
        # test version
        version = root.elements[
            '/html/head/meta[@name="generator"]'
        ].attributes['content']
        assert version.gsub!(/^Samizdat\s+/, '')
        assert_equal SAMIZDAT_VERSION, version
        # look for login form
        assert_equal 'login.rb',
            root.elements['/html/body/div[@id="subhead"]/div/a[2]'
            ].attributes['href']

        # test anonymous check on publish
        response = get('message.rb')
        assert_equal Net::HTTPUnauthorized, response.class
        root = parse(response.body)
        assert_equal ['Please Login'], elements(root,
            '/html/body/div[@id="main"]/div/div[@class="box-title"]')
    end

    def test_01_member
        response = post('member.rb',
            "login=#{@login}&full_name=#{@full_name}&email=#{@email}&passwd1=#{@passwd}&passwd2=#{@passwd}&submit", {'Referer' => @base.path})
        case response
        when Net::HTTPOK
            root = parse response.body
            assert_equal ['User Error'], elements(root,
                '/html/body/div[@id="main"]/div/div[@class="box-title"]')
            print 'test login already exists'
        when Net::HTTPFound
            assert @@session = response['set-cookie']
            assert_equal @base.to_s, response['location']

            # check if valid session is reflected in subhead
            response = get('', {'Cookie' => @@session})
            assert_equal Net::HTTPOK, response.class
            root = parse(response.body)
            assert_equal @full_name, elements(root,
                '/html/body/div[@id="subhead"]/div/a[1]')[0]
        else assert false, 'Unexpected HTTP code'
        end

        # todo: test checks for duplicates
    end

    def test_02_login
        # test if login fails with wrong passwd
        response = post 'login.rb', "login=#{@login}&passwd=#{@passwd}wrong"
        assert_equal nil, response['set-cookie']

        # try to login with correct passwd
        response = post 'login.rb', "login=#{@login}&passwd=#{@passwd}"
        assert @@session = response['set-cookie']

        # extract member id
        response = get('', {"Cookie" => @@session})
        assert @@session = response['set-cookie']
        assert_equal Net::HTTPOK, response.class
        root = parse(response.body)
        @@id = root.elements[
            '/html/body/div[@id="subhead"]/div/a[1]'].attributes['href']
        assert @@id.to_i > 0
    end

    def test_03_message
        # post message
        title, content = 'Test Message 1', 'blah blah'
        response = post('message.rb',
            "title=#{CGI.escape(title)}&content=#{CGI.escape(content)}&confirm",
            {"Cookie" => @@session})
        assert @@session = response['set-cookie']
        assert_equal Net::HTTPFound, response.class
        assert id = response['location']
        assert_equal Net::HTTPOK, (response = get(id)).class
        msg = parse(response.body).elements['/html/body/div[@id="main"]/div']
        assert_equal [title], elements(msg, %{//div[@class="title"]/a[@href="#{id}"]})
        assert_equal content, elements(msg,
            '//div[@class="content"]/p').join.strip

        # post plain text reply
        content = "plain\ntext"
        response = post('message.rb', "title=#{CGI.escape(title)}&content=#{CGI.escape(content)}&format=text/plain&parent=#{id}&confirm",
            {"Cookie" => @@session})
        assert @@session = response['set-cookie']
        assert_equal Net::HTTPFound, response.class
        assert id2 = response['location']
        assert_equal Net::HTTPOK, (response = get(id2)).class
        msg = parse(response.body).elements['/html/body/div[@id="main"]/div']
        assert_equal [title], elements(msg, %{//div[@class="title"]/a[@href="#{id2}"]})
        assert_equal content, elements(msg,
            '//div[@class="content"]/pre').join.strip

        # todo: publish query (test_query)
        # todo: test file upload
    end

    def test_04_resource
        # test 404 on nonexistent resoource
        assert_equal Net::HTTPNotFound, (response = get('resource.rb')).class

        # get a test resource
        assert (response = get('')).kind_of?(Net::HTTPSuccess)
        main = parse(response.body).elements['/html/body/div[@id="main"]']
        msg = main.elements[%{//div[@class="info"]/a[@href="#{@@id}"]/../..}]
        assert id = msg.elements['div[@class="title"]/a'].attributes['href']

        # test anonymous check for vote
        response = post('resource.rb',
            "id=#{id}&related=#{config['focus'][0]}&rating=2")
        assert_equal Net::HTTPUnauthorized, response.class
        root = parse(response.body)
        assert_equal ['Please Login'], elements(root,
            '/html/body/div[@id="main"]/div/div[@class="box-title"]')

        # vote on rating
        response = post('resource.rb',
            "id=#{id}&related=#{config['focus'][0]}&rating=1",
            {"Cookie" => @@session, "Referer" => @base.path + id})
        assert @@session = response['set-cookie']
        assert_equal Net::HTTPFound, response.class
        assert_equal id, response['location']
        assert_equal Net::HTTPOK, (response = get(id)).class
        root = parse(response.body)
        assert_equal [': 1.00 ('], elements(root,
            %{/html/body//div[@id="focuses"]/div[@class="box-content"]/p[1]})
    end

    def test_05_stress
        title, content = 'Test Thread', '.'
        response = post('message.rb', "title=#{CGI.escape(title)}&content=#{CGI.escape(content)}&confirm", {"Cookie" => @@session})
        @@session = response['set-cookie']
        parent = thread = response['location']
        count = 10   # increase if you have time to wait
        while count > 0 do
            title, content = 'Test Message ' + count.to_s, 'blah blah.'
            response = post('message.rb', "title=#{CGI.escape(title)}&content=#{CGI.escape(content)}&parent=#{CGI.escape(thread)}&confirm", {"Cookie" => @@session})
            @@session = response['set-cookie']
            parent = response['location']
            response = post('resource.rb',
                "id=#{parent}&related=#{config['focus'][0]}&rating=1",
                {"Cookie" => @@session, "Referer" => @base.path + parent})
            @@session = response['set-cookie']
            count = count - 1
        end
        # todo: test skip (index.rb, resource.rb)
    end

    def test_06_pingback
        # don't test pingback if it's not properly enabled
        return if config['pingback'].nil? or config['pingback']['login'].nil?
        id, = db.select_one('SELECT id FROM Member WHERE login = ?',
            config['pingback']['login'])
        return if id.nil?

        title, content = 'Test Pingback - Target', '.'
        response = post('message.rb', "title=#{CGI.escape(title)}&content=#{CGI.escape(content)}&confirm", {"Cookie" => @@session})
        @@session = response['set-cookie']
        target = response['location']

        title = 'Test Pingback - Sources'
        response = post('message.rb', "title=#{CGI.escape(title)}&content=#{CGI.escape(content)}&confirm", {"Cookie" => @@session})
        @@session = response['set-cookie']
        parent = response['location']

        content = "#Test Pingback - Source 1\n" + @base.to_s + target
        response = post('message.rb', "title=#{CGI.escape(title)}&content=#{CGI.escape(content)}&format=#{CGI.escape('text/uri-list')}&parent=#{CGI.escape(parent)}&confirm", {"Cookie" => @@session})
        @@session = response['set-cookie']
        pingback1 = response['location']

        content = "#Test Pingback - Source 2\n" + @base.to_s + target
        response = post('message.rb', "title=#{CGI.escape(title)}&content=#{CGI.escape(content)}&format=#{CGI.escape('text/uri-list')}&parent=#{CGI.escape(parent)}&confirm", {"Cookie" => @@session})
        @@session = response['set-cookie']
        pingback2 = response['location']

        # now check if both pingbacks are there
        root = parse(get(target.to_s).body)
        assert_equal [@base.to_s + pingback1, @base.to_s + pingback2],
            elements(root, %{/html/body/div[@id="main"]//li/div/div[@class="content"]/p/a}, 'href')
    end
end
