/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* background_cb.c -- GUI functions for the Canvas Properties window */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#endif

#include <math.h>

#include "support.h"
#include "interface.h"

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gdk-pixdata.h>

#include <message_windows.h>
#include <cache_pixbuf.h>
#include <cursor.h>
#include <sql_job_hierarchy.h>
#include <sql_background.h>
#include <background_cb.h>

#define ICON_BG_WIDTH_MAX  100
#define ICON_BG_HEIGHT_MAX 75

/*
 * Free a GString
 */
static void
string_free (gpointer string)
{
	g_string_free ((GString *)string, TRUE);
}


/*
 * Free a unsigned int array
 */
static void
unsigned_long_int_array_free (gpointer array)
{
	if (array != NULL) {
		free (array);
	}
}


/*
 * Ratio button (width and height) callback
 */
void
background_cb_size_ratio_clicked (GtkButton *button)
{
	GtkWidget *w;
	GdkPixbuf *p;
	gint active;

	active = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (button),
							"active"));

	if (active == 0) {
		p = create_pixbuf ("gschedwi/vchain-broken.png");
		g_object_set_data (	G_OBJECT (button), "active", 
					GINT_TO_POINTER (1));
	}
	else {
		p = create_pixbuf ("gschedwi/vchain.png");
		g_object_set_data (	G_OBJECT (button), "active",
					GINT_TO_POINTER (0));

		w = lookup_widget (	GTK_WIDGET (button),
					"spinbutton_bg_width");
		g_object_set_data (G_OBJECT (button), "width",
				GINT_TO_POINTER (
					gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w))));
		w = lookup_widget (	GTK_WIDGET (button),
					"spinbutton_bg_height");
		g_object_set_data (G_OBJECT (button), "height",
				GINT_TO_POINTER (
					gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w))));
	}
	w = lookup_widget (GTK_WIDGET (button), "image_bg_size_ratio");
	gtk_image_set_from_pixbuf (GTK_IMAGE (w), p);
	g_object_unref (p);
}


/*
 * Canvas width changed callback
 */
void
background_cb_width_changed_cb (GtkSpinButton *spin)
{
	GtkWidget *w;
	gint value, active, width, height;
	gdouble ratio, min, max;

	value = gtk_spin_button_get_value_as_int (spin);

	w = lookup_widget (GTK_WIDGET (spin), "button_bg_ratio");
	active = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (w), "active"));
	if (active == 0) {
		width = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (w),
								"width"));
		height = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (w),
								"height"));
		if (width != 0 && height != 0) {
			gtk_spin_button_get_range (spin, &min, &max);

			ratio = (gdouble)width / (gdouble)height;
			height = round (value / ratio);

			if (height > max) {
				gtk_spin_button_set_value (spin, value - 1);
				return;
			}
			if (height < min) {
				gtk_spin_button_set_value (spin, value + 1);
				return;
			}

			w = lookup_widget (	GTK_WIDGET (spin),
						"spinbutton_bg_height");
			if (height != gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w)))
			{
				gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
								height);
			}
		}
	}
}


/*
 * Canvas height changed callback
 */
void
background_cb_height_changed_cb (GtkSpinButton *spin)
{
	GtkWidget *w;
	gint value, active, width, height;
	gdouble ratio, min, max;

	value = gtk_spin_button_get_value_as_int (spin);

	w = lookup_widget (GTK_WIDGET (spin), "button_bg_ratio");
	active = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (w), "active"));
	if (active == 0) {
		width = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (w),
								"width"));
		height = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (w),
								"height"));
		if (width != 0 && height != 0) {
			gtk_spin_button_get_range (spin, &min, &max);

			ratio = (gdouble)width / (gdouble)height;
			width = round (value * ratio); 

			if (width > max) {
				gtk_spin_button_set_value (spin, value - 1);
				return;
			}
			if (width < min) {
				gtk_spin_button_set_value (spin, value + 1);
				return;
			}

			w = lookup_widget (	GTK_WIDGET (spin),
						"spinbutton_bg_width");
			if (width != gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w)))
			{
				gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
								width);
			}
		}
	}
}


/*
 * Draw an icon of the provided background image in the wallpaper area
 */
static void
background_cb_draw_icon (GtkWidget *dialog_bg, GdkPixbuf *p)
{
	gint width, height;
	GtkWidget *w;
	GdkPixbuf *bg, *scaled;
	gchar *s;
	gdouble ratio;

	/* Store the real-size image in the "image" property */
	g_object_set_data_full (G_OBJECT (dialog_bg), "image", p,
				g_object_unref);
	if (p == NULL) {
		/* Label */
		w = lookup_widget (dialog_bg, "label_bg_wallpaper"),
		gtk_label_set_markup (GTK_LABEL (w), _("<b>No Wallpaper</b>"));

		/* Image */
		w = lookup_widget (dialog_bg, "image_bg_wallpaper");
		bg = create_pixbuf ("gschedwi/no_bg.png");
		gtk_image_set_from_pixbuf (GTK_IMAGE (w), bg);
		g_object_unref (bg);
	}
	else {
		g_object_ref (p);

		width = gdk_pixbuf_get_width (p);
		height = gdk_pixbuf_get_height (p);

		/* Label */
		s = g_strdup_printf (_("%d x %d"), width, height);
		w = lookup_widget (dialog_bg, "label_bg_wallpaper"),
		gtk_label_set_markup (GTK_LABEL (w), s);
		g_free (s);

		/* Reduce the background image */
		ratio = (gdouble)width / (gdouble)height;
		if (ICON_BG_WIDTH_MAX / ratio > ICON_BG_HEIGHT_MAX) {
			height = ICON_BG_HEIGHT_MAX;
			width = ratio * ICON_BG_HEIGHT_MAX;
		}
		else {
			height = ICON_BG_WIDTH_MAX / ratio;
			width = ICON_BG_WIDTH_MAX;
		}

		scaled = gdk_pixbuf_scale_simple (	p, width, height,
                                             		GDK_INTERP_BILINEAR);

		w = lookup_widget (dialog_bg, "image_bg_wallpaper");
		gtk_image_set_from_pixbuf (GTK_IMAGE (w), scaled);
		g_object_unref (scaled);
	}
}


/*
 * Fill the dialog
 *     values[0] --> Jobset ID
 *     values[1] --> Image
 *     values[2] --> Background mode 
 *     values[3] --> First gradient color
 *     values[4] --> Last gradient color
 *     values[5] --> Canvas width
 *     values[6] --> Canvas height
 */
static void
background_cb_fill (	GtkWidget *dialog_bg,
			char **values, unsigned long int *values_len)
{
	GtkWidget *w;
	gint width, height;
	GdkColor color;
	GdkPixbuf *p;
	gchar *err_msg = NULL;

	/* Background image */
	if (values[1] == NULL || values_len[1] == 0) {
	
		/* No background */
		background_cb_draw_icon (dialog_bg, NULL);

		gtk_widget_set_sensitive (
				lookup_widget (dialog_bg, "button_bg_delete"),
				FALSE);
		gtk_widget_set_sensitive (
				lookup_widget (dialog_bg, "button_bg_save"),
				FALSE);
		
	}
	else {
		if (add_pixbuf_to_cache (	values[1], values_len[1],
						&p, NULL, NULL, NULL, NULL,
						&err_msg) != TRUE)
		{
			warning_window (
				_("Could not load the background image"),
				err_msg);
			g_free (err_msg);
			p = NULL;
			gtk_widget_set_sensitive (
				lookup_widget (	dialog_bg, "button_bg_save"),
				FALSE);
		}
		else {
			gtk_widget_set_sensitive (
				lookup_widget (	dialog_bg, "button_bg_save"),
				TRUE);
		}
		background_cb_draw_icon (dialog_bg, p);
		gtk_widget_set_sensitive (
				lookup_widget (	dialog_bg, "button_bg_delete"),
				TRUE);
	}

	/* Background style */
	w = lookup_widget (dialog_bg, "combobox_bg_style");
	gtk_combo_box_set_active (GTK_COMBO_BOX (w), atoi (values[2]));

	w = lookup_widget (dialog_bg, "colorbutton_bg_color_first");
	if (gdk_color_parse (values[3], &color) == TRUE) {
		gtk_color_button_set_color (GTK_COLOR_BUTTON (w), &color);
	}

	w = lookup_widget (dialog_bg, "colorbutton_bg_color_last");
	if (gdk_color_parse (values[4], &color) == TRUE) {
		gtk_color_button_set_color (GTK_COLOR_BUTTON (w), &color);
	}

	/* Canvas size */
	w = lookup_widget (dialog_bg, "spinbutton_bg_width");
	width = atoi (values[5]);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w), (gdouble) width);

	w = lookup_widget (dialog_bg, "spinbutton_bg_height");
	height = atoi (values[6]);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w), (gdouble) height);

	w = lookup_widget (dialog_bg, "button_bg_ratio");
	g_object_set_data (G_OBJECT (w), "width", GINT_TO_POINTER (width));
	g_object_set_data (G_OBJECT (w), "height", GINT_TO_POINTER (height));
}


/*
 * `Use the parent jobset parameter' toggle button
 */
void
background_cb_toggled (GtkToggleButton *togglebutton)
{
	GtkWidget *dialog_bg;
	char **values;
	unsigned long int *values_len;
	gboolean active;

	dialog_bg = lookup_widget (	GTK_WIDGET (togglebutton),
					"dialog_background");
	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = (char **) g_object_get_data (	G_OBJECT (dialog_bg),
							"values_h");
		values_len = (unsigned long int *) g_object_get_data (
							G_OBJECT (dialog_bg),
							"values_len_h");
	}
	else {
		values = (char **) g_object_get_data (	G_OBJECT (dialog_bg),
							"values");
		values_len = (unsigned long int *) g_object_get_data (
							G_OBJECT (dialog_bg),
							"values_len");
	}

	if (values != NULL && values_len != NULL) {
		background_cb_fill (dialog_bg, values, values_len);
	}

	gtk_widget_set_sensitive (
				lookup_widget (	GTK_WIDGET (togglebutton),
						"frame_bg_style"),
				(active == TRUE)? FALSE: TRUE);
	gtk_widget_set_sensitive (
				lookup_widget (	GTK_WIDGET (togglebutton),
						"frame_bg_wallpaper"),
				(active == TRUE)? FALSE: TRUE);
	gtk_widget_set_sensitive (
				lookup_widget (	GTK_WIDGET (togglebutton),
						"frame_bg_size"),
				(active == TRUE)? FALSE: TRUE);
}


/*
 * Initialize the window
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
dialog_background_init (GtkWidget *dialog_bg, const gchar *jobset_id,
			void (*refresh_func)(void *),
			void *refresh_func_user_data)
{
	sql_job_hierarchy_ptr ptr;
	char **values, **values_h;
	unsigned long int *values_len, *values_len_h;
	GtkWidget *c;
	GString *jobset_id_str;

	/* Retrieve the jobset parameters */
	ptr = sql_job_hierarchy_new (jobset_id, 0,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (ptr == NULL) {
		return -1;
	}

	if (sql_job_hierarchy_get (ptr,
				"jobset_background",
				"job_id,image,background_mode,gradient_color_primary,gradient_color_secondary,canvas_width,canvas_height",
				&values, &values_len,
				&values_h, &values_len_h) != 0)
	{
		sql_job_hierarchy_destroy (ptr);
		return -1;
	}
	sql_job_hierarchy_destroy (ptr);

	/* Store the jobset ID in the widget */
	jobset_id_str = g_string_new (jobset_id);
	g_object_set_data_full (G_OBJECT (dialog_bg), "jobset_id",
				(gpointer)jobset_id_str, string_free);

	/* Store the values in the widget */
	g_object_set_data_full (G_OBJECT (dialog_bg), "values",
				(gpointer)values,
				(GDestroyNotify) sql_free_row);
	g_object_set_data_full (G_OBJECT (dialog_bg), "values_len",
				(gpointer)values_len,
				unsigned_long_int_array_free);
	g_object_set_data_full (G_OBJECT (dialog_bg), "values_h",
				(gpointer)values_h,
				(GDestroyNotify) sql_free_row);
	g_object_set_data_full (G_OBJECT (dialog_bg), "values_len_h",
				(gpointer)values_len_h,
				unsigned_long_int_array_free);

	/*
	 * Check the `Use the parent jobset parameter' toggle button
	 * if the there is no background parameters defined for this
	 * jobset
	 */
	c = lookup_widget (dialog_bg, "checkbutton_bg_hierarchy");

	/*
	 * If it's the root jobset, there is no parent then remove the
	 * check button
	 */
	if (jobset_id[0] == '1' && jobset_id[1] == '\0') {
		gtk_widget_destroy (c);
		if (values == NULL) {
			background_cb_fill (dialog_bg, values_h, values_len_h);
		}
		else {
			background_cb_fill (dialog_bg, values, values_len);
		}
	}
	else {
		if (values == NULL) {
			gtk_toggle_button_set_active (	GTK_TOGGLE_BUTTON (c),
							TRUE);
		}
		else {
			gtk_toggle_button_set_active (	GTK_TOGGLE_BUTTON (c),
							FALSE);
		}
		background_cb_toggled (GTK_TOGGLE_BUTTON (c));
	}

	/*
	 * Store the function to call to refresh the canvas if the
	 * canvas parameters are changed
	 */
	g_object_set_data (	G_OBJECT (dialog_bg), "refresh_func",
				(gpointer) refresh_func);
	g_object_set_data (	G_OBJECT (dialog_bg), "refresh_func_user_data",
				(gpointer) refresh_func_user_data);
	return 0;
}


/*
 * Create a new canvas properties dialog for the provided jobset id
 *
 * Return:
 *   The GtkWidget of the new window (which has been `show'ed by this function)
 *   or NULL in case of error
 */
GtkWidget *
new_dialog_background (	const gchar *jobset_id,
			void (*refresh_func)(void *),
			void *refresh_func_user_data)
{
	GtkWidget *dialog_bg;

	dialog_bg = create_dialog_background ();
	if (dialog_background_init (dialog_bg, jobset_id,
				refresh_func, refresh_func_user_data) != 0)
	{
		gtk_widget_destroy (dialog_bg);
		return NULL;
	}
	else {
		gtk_widget_show (dialog_bg);
		return dialog_bg;
	}
}


/*
 * Update the image preview in file chooser dialog
 */
static void
update_preview_cb (GtkFileChooser *file_chooser, gpointer data)
{
	GtkWidget *preview;
	char *filename;
	GdkPixbuf *pixbuf;
	gboolean have_preview;

	preview = GTK_WIDGET (data);
	filename = gtk_file_chooser_get_preview_filename (file_chooser);

	if (filename != NULL) {
		pixbuf = gdk_pixbuf_new_from_file_at_size (filename, 128, 128,
								NULL);
		have_preview = (pixbuf != NULL);
		g_free (filename);
	}
	else {
		have_preview = FALSE;
		pixbuf = NULL;
	}

	gtk_image_set_from_pixbuf (GTK_IMAGE (preview), pixbuf);
	if (pixbuf != NULL) {
		g_object_unref (pixbuf);
	}

	gtk_file_chooser_set_preview_widget_active (	file_chooser,
							have_preview);
}


/*
 * Upload new background image button callback
 */
void
background_cb_upload_clicked (GtkButton *button)
{
	GtkWidget *dialog_bg, *dialog, *preview;
	gchar *filename;
	GError *err = NULL;
	GdkPixbuf *p;

	dialog_bg = lookup_widget (GTK_WIDGET (button), "dialog_background");

	/* Load file chooser dialog */
	dialog = gtk_file_chooser_dialog_new (
					"Load Background Image",
					GTK_WINDOW (dialog_bg),
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);
	preview = gtk_image_new ();
	gtk_file_chooser_set_preview_widget (	GTK_FILE_CHOOSER (dialog),
						preview);
	g_signal_connect (	dialog, "update-preview",
				G_CALLBACK (update_preview_cb), preview);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {

		/* Get the input file name */
		filename = gtk_file_chooser_get_filename (
						GTK_FILE_CHOOSER (dialog));

		/* Load the image pixbuf */
		p =  gdk_pixbuf_new_from_file (filename, &err);
		if (p == NULL) {
			gtk_widget_destroy (dialog);
			error_window (	_("Failed to load the image"),
					(err != NULL) ? err->message : NULL);
			if (err != NULL) {
				g_error_free (err);
			}
			g_free (filename);
			return;
		}

		if (err != NULL) {
			g_error_free (err);
		}
		g_free (filename);

		/* Draw the image */
		background_cb_draw_icon (dialog_bg, p);
		g_object_unref (p);
	}
	gtk_widget_destroy (dialog);
}


/*
 * Delete background image button callback
 */
void
background_cb_delete_clicked (GtkButton *button)
{
	GtkWidget *dialog_bg;

	dialog_bg = lookup_widget (GTK_WIDGET (button), "dialog_background");

	if (question_delete_window (dialog_bg,
		_("Are you sure you want to remove the image?")) == FALSE)
	{
		return;
	}
	background_cb_draw_icon (dialog_bg, NULL);
}


/*
 * Save button callback
 */
void
background_cb_save_clicked (GtkButton *button)
{
	GtkWidget *dialog_bg, *dialog;
	gchar *filename, *s, *format;
	GError *err = NULL;
	GdkPixbuf *p;

	dialog_bg = lookup_widget (GTK_WIDGET (button), "dialog_background");

	/* Save file chooser dialog */
	dialog = gtk_file_chooser_dialog_new (
					"Save Background Image",
					GTK_WINDOW (dialog_bg),
					GTK_FILE_CHOOSER_ACTION_SAVE,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					NULL);
/*TODO GTK 2.8
gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);
 */

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {

		/* Retrieve the image pixbuf */
		p = (GdkPixbuf *) g_object_get_data (	G_OBJECT (dialog_bg),
							"image");

		/* Get the output file name */
		filename = gtk_file_chooser_get_filename (
						GTK_FILE_CHOOSER (dialog));

		/* Does the file already exist */
		if (g_file_test (filename, G_FILE_TEST_EXISTS) == TRUE) {

			s = g_strdup_printf (
					_("A file named '%s' already exists."),
					filename);
			if (question_replace_window (dialog, s,
		_("Do you want to replace it with the image you are saving?"))
					== FALSE)
			{
				gtk_widget_destroy (dialog);
				g_free (s);
				g_free (filename);
				return;
			}
			g_free (s);
		}

		/* Get the output format from the file extension */
		if (	   g_strrstr (filename, ".jpeg") != NULL
			|| g_strrstr (filename, ".jpg") != NULL)
		{
			format = "jpeg";
		}
		else {
		if (g_strrstr (filename, ".ico") != NULL) {
			format = "ico";
		}
		else {
		if (g_strrstr (filename, ".bmp") != NULL) {
			format = "bmp";
		}
		else {
			format = "png";
		}}}

		/* Write the image */
		if (gdk_pixbuf_save (p, filename, format, &err, NULL) != TRUE)
		{
			gtk_widget_destroy (dialog);
			error_window (	_("Failed to save the image"),
					(err != NULL) ? err->message : NULL);
			if (err != NULL) {
				g_error_free (err);
			}
			g_free (filename);
			return;
		}
		if (err != NULL) {
			g_error_free (err);
		}
		g_free (filename);
	}
	gtk_widget_destroy (dialog);
}


/*
 * Callback for the `OK' button
 */
void
background_cb_ok (GtkButton *button)
{
	GtkWidget *dialog_bg, *w;
	GString *jobset_id_str;
	gint mode, width, height;
	gchar *first_color, *last_color;
	GdkColor color;
	GdkPixbuf *p;
	gpointer ptr;
	GdkPixdata pixdata;
	guint8 *stream;
	guint stream_len;
	void (*refresh_func)(void *);
	void *refresh_func_user_data;

	cursor_busy (GTK_WIDGET (button));
	dialog_bg = lookup_widget (GTK_WIDGET (button), "dialog_background");

	jobset_id_str = g_object_get_data (G_OBJECT (dialog_bg), "jobset_id");
	if (jobset_id_str == NULL || jobset_id_str->str == NULL) {
		cursor_normal (GTK_WIDGET (button));
		gtk_widget_destroy (dialog_bg);	
		return;
	}

	if (jobset_id_str->len == 1 && jobset_id_str->str[0] == '1') {
		w = NULL;
	}
	else {
		w = lookup_widget (	GTK_WIDGET (button),
					"checkbutton_bg_hierarchy");
	}
	if (	   w != NULL
		&& gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)) == TRUE)
	{
		/*
		 * Use the parent jobset parameter and then delete the
		 * canvas parameter for the current jobset from the
		 * database
		 */
		sql_job_hierarchy_del (jobset_id_str->str,
				"jobset_background",
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	}
	else {
		/* Retrieve the image pixbuf */
		p = (GdkPixbuf *) g_object_get_data (	G_OBJECT (dialog_bg),
							"image");
		/* Serialize the image */
		if (p != NULL) {
			ptr = gdk_pixdata_from_pixbuf (&pixdata, p, TRUE);
			stream = gdk_pixdata_serialize (&pixdata, &stream_len);
			g_free (ptr);
		}
		else {
			stream = NULL;
			stream_len = 0;
		}

		/* Background style */
		w = lookup_widget (dialog_bg, "combobox_bg_style");
		mode = gtk_combo_box_get_active (GTK_COMBO_BOX (w));

		/* First color */
		w = lookup_widget (dialog_bg, "colorbutton_bg_color_first");
		gtk_color_button_get_color (GTK_COLOR_BUTTON (w), &color);
		first_color = g_strdup_printf (	"#%2.2x%2.2x%2.2x",
						color.red / 257,
						color.green / 257,
						color.blue / 257);

		/* Last color*/
		w = lookup_widget (dialog_bg, "colorbutton_bg_color_last");
		gtk_color_button_get_color (GTK_COLOR_BUTTON (w), &color);
		last_color = g_strdup_printf (	"#%2.2x%2.2x%2.2x",
						color.red / 257,
						color.green / 257,
						color.blue / 257);

		/* Width */	
		w = lookup_widget (dialog_bg, "spinbutton_bg_width");
		width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));

		/* Height */	
		w = lookup_widget (dialog_bg, "spinbutton_bg_height");
		height = gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w));

		/* Update the database */
		sql_background_set (
				jobset_id_str->str,
				(char *)stream, (unsigned long int)stream_len,
				mode, first_color, last_color,
				width, height,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
		g_free (stream);
		g_free (first_color);
		g_free (last_color);
	}

	/* Refresh the canvas for the parameters to be taken into account */
	refresh_func = (void (*)(void *)) g_object_get_data (
						G_OBJECT (dialog_bg),
						"refresh_func"); 
	refresh_func_user_data = (void *) g_object_get_data (
						G_OBJECT (dialog_bg),
						"refresh_func_user_data");
	refresh_func (refresh_func_user_data);
	cursor_normal (GTK_WIDGET (button));

	/* Destroy the dialog */
	gtk_widget_destroy (dialog_bg);
}

/*------------------------======= End Of File =======------------------------*/
