/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* env_widget.c - Environment choice widget */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include "support.h"
#include "interface.h"

#include <message_windows.h>
#include <cursor.h>
#include <sql_env.h>
#include <sql_common.h>
#include <env_widget.h>


/*
 * Callback when the selection changes
 * The buttons are made sensitive depending of the selected item
 */
static void
env_widget_selection_changed_cb (	GtkTreeSelection *selection,
					env_widget_ptr ptr)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreePath *path;

	/* Nothing selected */
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == FALSE)
	{
		gtk_widget_set_sensitive (ptr->button_delete, FALSE);
		gtk_widget_set_sensitive (ptr->button_up, FALSE);
		gtk_widget_set_sensitive (ptr->button_down, FALSE);
		gtk_widget_set_sensitive (ptr->button_top, FALSE);
		gtk_widget_set_sensitive (ptr->button_bottom, FALSE);
		return;
	}

	/* Is there an item before the selected one? */
	gtk_widget_set_sensitive (ptr->button_delete, TRUE);
	path =  gtk_tree_model_get_path (model, &iter);
	if (gtk_tree_path_prev (path) == TRUE) {
		gtk_widget_set_sensitive (ptr->button_up, TRUE);
		gtk_widget_set_sensitive (ptr->button_top, TRUE);
	}
	else {
		gtk_widget_set_sensitive (ptr->button_up, FALSE);
		gtk_widget_set_sensitive (ptr->button_top, FALSE);
	}
	gtk_tree_path_free (path);

	/* Is there an item bellow the selected one? */
	if (gtk_tree_model_iter_next (model, &iter) == TRUE) {
		gtk_widget_set_sensitive (ptr->button_down, TRUE);
		gtk_widget_set_sensitive (ptr->button_bottom, TRUE);
	}
	else {
		gtk_widget_set_sensitive (ptr->button_down, FALSE);
		gtk_widget_set_sensitive (ptr->button_bottom, FALSE);
	}
}


/*
 * Callback for the `Move to top' button
 */
static void
env_widget_top_clicked (GtkButton *button, gpointer user_data)
{
	env_widget_ptr ptr = (env_widget_ptr) user_data;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptr->view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		gtk_list_store_move_after (	GTK_LIST_STORE (model),
						&iter, NULL);
		env_widget_selection_changed_cb (select, ptr);
	}

}


/*
 * Callback for the `Move to bottom' button
 */
static void
env_widget_bottom_clicked (GtkButton *button, gpointer user_data)
{
	env_widget_ptr ptr = (env_widget_ptr) user_data;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptr->view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		gtk_list_store_move_before (	GTK_LIST_STORE (model),
						&iter, NULL);
		env_widget_selection_changed_cb (select, ptr);
	}
}


/*
 * Callback for the `Move up' button
 */
static void
env_widget_up_clicked (GtkButton *button, gpointer user_data)
{
	env_widget_ptr ptr = (env_widget_ptr) user_data;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, prev;
	GtkTreePath *path;

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptr->view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		path = gtk_tree_model_get_path (model, &iter);
		gtk_tree_path_prev (path);
		gtk_tree_model_get_iter (model, &prev, path);
		gtk_list_store_swap (GTK_LIST_STORE (model), &iter, &prev);
		gtk_tree_path_free (path);
		env_widget_selection_changed_cb (select, ptr);
	}
}


/*
 * Callback for the `Move down' button
 */
static void
env_widget_down_clicked (GtkButton *button, gpointer user_data)
{
	env_widget_ptr ptr = (env_widget_ptr) user_data;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptr->view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		next = iter;
		gtk_tree_model_iter_next (model, &next);
		gtk_list_store_swap (GTK_LIST_STORE (model), &iter, &next);
		env_widget_selection_changed_cb (select, ptr);
	}
}


/*
 * Callback for the `Remove' button
 */
static void
env_widget_delete_clicked (GtkButton *button, gpointer user_data)
{
	env_widget_ptr ptr = (env_widget_ptr) user_data;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptr->view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		/* Move the selection to the next environment */
		next = iter;
		if (gtk_tree_model_iter_next (model, &next) == TRUE) {
			gtk_tree_selection_select_iter (select, &next);
		}
		/* Remove the environment */
		gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
		/* Refresh the selection */
		env_widget_selection_changed_cb (select, ptr);
	}
}


/*
 * Add an environment to a combo box
 *
 * The provided id and name are added to a list which is stored in the
 * combo box user data area.  When the user will select an item, this item
 * will be search in the list to retrieve the associated id
 *
 * Return:
 *   Always 0
 */
static int
add_environment_combo (void *ptr, const char *id, const char *name)
{
	GtkWidget *combo = ptr;
	GSList *list;
	GPtrArray *env_item;

	/* Add the name in the combo box list */
	gtk_combo_box_append_text (GTK_COMBO_BOX (combo), name);

	/* Retrieve the list of the combo box items */
	list = (GSList *) g_object_get_data (G_OBJECT (combo), "list");

	/* Build an array of two strings: the environment id and name */
	env_item = g_ptr_array_sized_new (2);
	g_ptr_array_add (env_item, g_strdup (id));
	g_ptr_array_add (env_item, g_strdup (name));

	/* Store this array in the list */
	list = g_slist_append (list, env_item);

	/*
	 * Store the list in the combo box object
	 * This list will be freed when the `Choose environment' popup will be
	 * destroyed (the callback of the destroy event is the
	 * env_widget_env_dialog_choose_destroy() function)
	 */
	g_object_set_data (G_OBJECT (combo), "list", list);

	return 0;
}


/*
 * Add the environment id of the provided item to the provided list
 *
 * Return:
 *   FALSE --> No error
 *    TRUE --> Memory allocation error (this return value stops the
 *             gtk_tree_model_foreach() calling function)
 */
static gboolean
add_id_to_list (	GtkTreeModel *model,
			GtkTreePath *path,
			GtkTreeIter *iter,
			gpointer data)
{
	lwc_LL *current_lst = data;
	gchar *id;

	gtk_tree_model_get (model, iter, 0, &id, -1);
	if (lwc_addEndLL (current_lst, id) != 0) {
		g_free (id);
		lwc_emptyLL (current_lst, (void (*)(const void *))g_free);
		error_window (_("Memory allocation error"), NULL);
		return TRUE;
	}
	return FALSE;
}


/*
 * Comparison function between an environment array and a string
 *
 * Return:
 *       0 --> Match
 *   other --> No match
 */
static gint
cmp_strings (gconstpointer a, gconstpointer b)
{
	const GPtrArray *env = a;

	return strcmp ((gchar *) g_ptr_array_index (env, 1), (gchar *)b);
}


/*
 * Callback of the `OK' button of the choose popup
 */
void
env_widget_env_choose_clicked (GtkButton *button)
{
	env_widget_ptr ptr;
	GtkWidget *dialog_choose, *combo;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, new_iter;
	GSList *envs, *env_item;
	gchar *selected_env;

	/* Retrieve the required objects */
	dialog_choose = lookup_widget (	GTK_WIDGET (button),
					"dialog_env_choose");
	combo = lookup_widget (GTK_WIDGET (button), "combobox_env_choose");
	ptr = (env_widget_ptr) g_object_get_data (	G_OBJECT (combo),
							"env_widget");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptr->view));
	envs = (GSList *) g_object_get_data (G_OBJECT (combo), "list");
	selected_env = gtk_combo_box_get_active_text (GTK_COMBO_BOX (combo));

	/*
	 * Find in the environment list the one which match the selected
	 * one in the combo box.
	 */
	env_item = g_slist_find_custom (envs, selected_env, cmp_strings);
	g_free (selected_env);

	/*
	 * If an item is selected in the tree view, add the environment
	 * just after the selected one.  If not add it at the end
	 */
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		gtk_list_store_insert_after (	GTK_LIST_STORE (model),
						&new_iter,
						&iter);
		gtk_list_store_set (
			GTK_LIST_STORE (model),
			&new_iter,
			/* Environment ID */
			0, g_ptr_array_index((GPtrArray *)(env_item->data), 0),
			/* Environment name */
			1, g_ptr_array_index((GPtrArray *)(env_item->data), 1),
			-1);
		/* Refresh the selection */
		env_widget_selection_changed_cb (select, ptr);
	}
	else {
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (ptr->view));	
		gtk_list_store_append (GTK_LIST_STORE (model), &new_iter);
		gtk_list_store_set (
			GTK_LIST_STORE (model),
			&new_iter,
			/* Environment ID */
			0, g_ptr_array_index((GPtrArray *)(env_item->data), 0),
			/* Environment name */
			1, g_ptr_array_index((GPtrArray *)(env_item->data), 1),
			-1);
	}
	gtk_widget_destroy (dialog_choose);
}


/*
 * Free the provided string
 */
static void
free_string (gpointer str, gpointer user_data)
{
	g_free (str);
}


/*
 * Free the provided array of strings
 */
static void
free_env_array (gpointer array, gpointer user_data)
{
	g_ptr_array_foreach ((GPtrArray *)array, free_string, NULL);
	g_ptr_array_free ((GPtrArray *)array, TRUE);
}


/*
 * Free the provided list of array
 */
static void
env_list_free (GSList *lst)
{
	g_slist_foreach (lst, free_env_array, NULL);
	g_slist_free (lst);
}


/*
 * Callback when the choose popup is destroyed
 */
void
env_widget_env_choose_destroy (GtkObject *dialog_choose)
{
	GtkWidget *combo;
	GSList *envs;

	combo = lookup_widget (	GTK_WIDGET (dialog_choose),
				"combobox_env_choose");
	/* Free the environment list */
	envs = (GSList *) g_object_get_data (G_OBJECT (combo), "list");
	env_list_free (envs);
}


/*
 * Callback of the `Add an environment' button
 *
 * This callback will display a popup which contains a combo box of the
 * available environments.  The environments already in the view tree won't
 * be available in the combo box
 */
static void
env_widget_add_clicked (GtkButton *button, gpointer user_data)
{
	env_widget_ptr ptr = (env_widget_ptr) user_data;
	GtkWidget *dialog_choose, *combo;
	GtkTreeModel *model;
	lwc_LL *current_lst;
	int nb;

	/* Allocate memory for the list */
	current_lst = lwc_newLL ();
	if (current_lst == NULL) {
		error_window (_("Memory allocation error"), NULL);
		return;
	}

	cursor_busy (GTK_WIDGET (button));

	/* Retrieve the model */
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (ptr->view));

	/*
	 * Built the list of the environments in the tree view
	 * These environments won't be displayed in the combo box as they are
	 * already in the list.
	 */
	gtk_tree_model_foreach (model, add_id_to_list, current_lst);

	/* Create the choose dialog (but don't show it yet) */
	dialog_choose = create_dialog_env_choose ();
	combo = lookup_widget (dialog_choose, "combobox_env_choose");
	g_object_set_data (G_OBJECT (combo), "env_widget", (gpointer)ptr);

	/* Retrieve the available environments */
	nb = sql_env_list (	current_lst,
				add_environment_combo, combo,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (nb < 0) {
		lwc_delLL (current_lst, (void (*)(const void *))g_free);
		gtk_widget_destroy (dialog_choose);
		cursor_normal (GTK_WIDGET (button));
		return;
	}

	lwc_delLL (current_lst, (void (*)(const void *))g_free);

	/* All the environments are already associated with this host/job */
	if (nb == 0) {
		gtk_widget_destroy (dialog_choose);
		cursor_normal (GTK_WIDGET (button));
		info_window (_("No more environments available"), NULL);
		return;
	}

	/* By default, select the first item of the combo box */
	gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);

	cursor_normal (GTK_WIDGET (button));

	/* Display the choose window */
	gtk_widget_show (dialog_choose);
}


/*
 * Destroy the provided env_widget_ptr object.  This function is called
 * automatically when the Tree View is destroyed
 */
static void
env_widget_destroy_real (gpointer data)
{
	env_widget_ptr ptr = (env_widget_ptr) data;

	g_free (ptr);
}


/*
 * Destroy the provided env_widget_ptr object
 */
void
env_widget_destroy (env_widget_ptr ptr)
{
	if (ptr != NULL) {
		/* Remove the callback handlers */
		if (ptr->button_add != NULL) {
			g_signal_handler_disconnect (
						(gpointer)ptr->button_add,
						ptr->handler_button_add);
		}
		if (ptr->button_delete != NULL) {
			g_signal_handler_disconnect (
						(gpointer)ptr->button_delete,
						ptr->handler_button_delete);
		}
		if (ptr->button_up != NULL) {
			g_signal_handler_disconnect (
						(gpointer)ptr->button_up,
						ptr->handler_button_up);
		}
		if (ptr->button_down != NULL) {
			g_signal_handler_disconnect (
						(gpointer)ptr->button_down,
						ptr->handler_button_down);
		}
		if (ptr->button_top != NULL) {
			g_signal_handler_disconnect (
						(gpointer)ptr->button_top,
						ptr->handler_button_top);
		}
		if (ptr->button_bottom != NULL) {
			g_signal_handler_disconnect (
						(gpointer)ptr->button_bottom,
						ptr->handler_button_bottom);
		}
		if (ptr->view != NULL) {
			/*
			 * Set the "env_widget" variable to NULL to
			 * force the destroy callback to be called
			 * (env_widget_destroy_real())
			 */
			g_object_set_data (	G_OBJECT (ptr->view),
						"env_widget", NULL);
		}
		else {
			env_widget_destroy_real ((gpointer)ptr);
		}
	}
}


/*
 * Double-click on an environment row
 */
/*
static void
view_onRowActivated (	GtkTreeView *treeview,
			GtkTreePath *path,
			GtkTreeViewColumn *col,
			gpointer userdata)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *env_id;

	model = gtk_tree_view_get_model (treeview);
	if (gtk_tree_model_get_iter (model, &iter, path) == TRUE) {
		cursor_busy (GTK_WIDGET (treeview));
		gtk_tree_model_get (model, &iter, 0, &env_id, -1);
		new_dialog_env (treeview, env_id);
		g_free (env_id);
		cursor_normal (GTK_WIDGET (treeview));
	}
}
*/


/*
 * Create a new env_widget_ptr object
 *
 * Return:
 *   The new env_widget_ptr.  This object will be automatically destroyed
 *   when the provided view id destroyed.  Alternatively, the object can
 *   be destroyed by env_widget_destroy()
 */
env_widget_ptr
env_widget_new (GtkWidget *view,
		const gchar *button_add,
		const gchar *button_delete,
		const gchar *button_up,
		const gchar *button_down,
		const gchar *button_top,
		const gchar *button_bottom)
{
	env_widget_ptr ptr;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;

	/* Create a new object */
	ptr = g_new0 (env_widget, 1);

	/* Initialize the object */
	ptr->view = view;

	if (button_add != NULL) {
		ptr->button_add = lookup_widget (view, button_add);
		ptr->handler_button_add = g_signal_connect (
					(gpointer)ptr->button_add, "clicked",
					G_CALLBACK (env_widget_add_clicked),
					ptr);
	}
	if (button_delete != NULL) {
		ptr->button_delete = lookup_widget (view, button_delete);
		ptr->handler_button_delete = g_signal_connect (
					(gpointer)ptr->button_delete,
					"clicked",
					G_CALLBACK (env_widget_delete_clicked),
					ptr);
	}
	if (button_up != NULL) {
		ptr->button_up = lookup_widget (view, button_up);
		ptr->handler_button_up = g_signal_connect (
					(gpointer)ptr->button_up, "clicked",
					G_CALLBACK (env_widget_up_clicked),
					ptr);
	}
	if (button_down != NULL) {
		ptr->button_down = lookup_widget (view, button_down);
		ptr->handler_button_down = g_signal_connect (
					(gpointer)ptr->button_down, "clicked",
					G_CALLBACK (env_widget_down_clicked),
					ptr);
	}
	if (button_top != NULL) {
		ptr->button_top = lookup_widget (view, button_top);
		ptr->handler_button_top = g_signal_connect (
					(gpointer)ptr->button_top, "clicked",
					G_CALLBACK (env_widget_top_clicked),
					ptr);
	}
	if (button_bottom != NULL) {
		ptr->button_bottom = lookup_widget (view, button_bottom);
		ptr->handler_button_bottom = g_signal_connect (
					(gpointer)ptr->button_bottom,
					"clicked",
					G_CALLBACK (env_widget_bottom_clicked),
					ptr);
	}

	/* Store the object in the view */
	g_object_set_data_full (G_OBJECT (view), "env_widget",
				(gpointer)ptr, env_widget_destroy_real);

	/* Initialize the view */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (	GTK_TREE_VIEW (view),
							-1,
							_("Environment"),
							renderer,
							"text", 1,
							NULL);

	/* Selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_BROWSE);
	g_signal_connect (	G_OBJECT (select), "changed",
				G_CALLBACK (env_widget_selection_changed_cb),
				ptr);
	return ptr;
}


/*
 * Retrieve the env_widget_ptr from the provided view
 */
env_widget_ptr
env_widget_get_from_view (GtkWidget *view)
{
	return (env_widget_ptr) g_object_get_data (	G_OBJECT (view),
							"env_widget");
}


/*
 * Fill the view with the provided values.  values is a list of rows.
 * Each row is an array of strings:
 *     row[0] --> Position
 *     row[1] --> Environment ID
 *     row[2] --> Environment name
 */
void
env_widget_build_rows (env_widget_ptr ptr, lwc_LL *values)
{
	GtkListStore *store;
	GtkTreeIter iter;
	GtkTreeSelection *select;
	char **row;


	/* Create and fill the GtkListStore object */
	store = gtk_list_store_new (	2,
					G_TYPE_STRING,	/* Env id */	
					G_TYPE_STRING);	/* Env Name */

	lwc_rewindLL (values);
	while ((row = (char **)lwc_nextLL (values)) != NULL) {
		/*
		 * row[0] --> Position
		 * row[1] --> Environment ID
		 * row[2] --> Environment name
		 */
		if (atoi (row[0]) < 0) {
			break;
		}
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (	store, &iter,
					0, row[1], /* Environment ID */
					1, row[2], /* Environment Name */
					-1);
	}

	/* Associate the model with the view */
	gtk_tree_view_set_model (	GTK_TREE_VIEW (ptr->view),
					GTK_TREE_MODEL (store));
	g_object_unref (store);
	/* Refresh the selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ptr->view));
	env_widget_selection_changed_cb (select, ptr);
}


/*
 * Fill the view by calling a SQL function.  The rows returned by the SQL
 * function must be as follow:
 *     row[0] --> Position
 *     row[1] --> Environment ID
 *     row[2] --> Environment name
 *
 * Return:
 *       0 --> No error
 *   other --> SQL error (an error message has been displayed)
 */
int
env_widget_build_sql (	env_widget_ptr ptr,
			const gchar *id,
			int (*sql_func)(
				const char *,
				lwc_LL **,
				void (*)(void *, const char *, unsigned int),
				void *))	
{
	lwc_LL *values;
	int ret;

	if (id != NULL && id[0] != '\0') {
		ret = sql_func (id, &values,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
		if (ret != 0) {
			return ret;
		}
	}
	else {
		values = NULL;
	}
	env_widget_build_rows (ptr, values);
	lwc_delLL (values, (void (*)(const void *))sql_free_row);
	return 0;
}


/*
 * Save the environments to the database
 *
 * The environment list is saved by calling the provided sql_func() function.
 * This sql_func() function must return 0 on success
 *
 * Return:
 *       0 --> No error
 *   other --> Error (a message has been displayed)
 */
int
env_widget_save_list (
		env_widget_ptr ptr,
		const gchar *id,
		int (*sql_func)(
			const char *,
			lwc_LL *,
			void (*)(void *, const char *, unsigned int),
			void *))
{
	lwc_LL *list;
	GtkTreeModel *model;
	int ret;

	/* Allocate the list of the environments to save */
	list = lwc_newLL ();
	if (list == NULL) {
		error_window (_("Memory allocation error"), NULL);
		return -1;
	}

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (ptr->view));

	/* Built the list of the environments in the list */
	gtk_tree_model_foreach (model, add_id_to_list, list);

	/* Update the database */
	ret = sql_func (id, list,
			(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
			_("Database error"));
	lwc_delLL (list, (void (*)(const void *)) g_free);
	return ret;
}

/*------------------------======= End Of File =======------------------------*/
