/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* find_cb.c -- GUI functions for the find job window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <cursor.h>
#include <message_windows.h>
#include <main_cb.h>
#include <sql_jobsets.h>
#include <sql_hierarchy.h>
#include <find_cb.h>


/*
 * Close (hide) the find window
 */
static void
find_close (GtkWidget *w)
{
	gtk_widget_hide (lookup_widget (w, "dialog_find"));
}


/*
 * Select in the main window the selected item
 */
static void
find_select (GtkWidget *w)
{
	GtkWidget *view, *dialog_find, *parent_widget;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *id, *parent_id, *job_type;

	dialog_find = lookup_widget (w, "dialog_find");
	view = lookup_widget (w, "treeview_find");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE) {
		parent_widget = g_object_get_data (	G_OBJECT (dialog_find),
							"parent_widget");
		gtk_tree_model_get (	model, &iter,
					0, &id,
					1, &parent_id,
					4, &job_type,
					-1);
		main_select_job (parent_widget, id, parent_id, job_type);
		g_free (id);
		g_free (parent_id);
		g_free (job_type);
	}
}


/*
 * Select button callback
 */
void
find_select_clicked (GtkButton *button)
{
	find_select (GTK_WIDGET (button));
}


/*
 * Close button callback
 */
void
find_close_clicked (GtkButton *button)
{
	find_close (GTK_WIDGET (button));
}


/*
 * delete-event callback (when the user closes the window)
 */
gboolean
find_delete_event (GtkWidget *widget)
{
	find_close (widget);
	/*
	 * Return TRUE to stop other handlers from being invoked for
	 * the event. Otherwise, the window will be destroyed
	 */
	return TRUE;
}


/*
 * Double-click on a row
 */
static void
view_onRowActivated (   GtkTreeView *treeview,
			GtkTreePath *path,
			GtkTreeViewColumn *col,
			gpointer userdata)
{
	find_close (GTK_WIDGET (treeview));
	find_select (GTK_WIDGET (treeview));
}


/*
 * Callback when the selection changes
 * The Select button is made sensitive if an item is selected
 */
static void
tree_selection_changed_cb (GtkTreeSelection *selection, gpointer data)
{
	GtkWidget *w = (GtkWidget *)data;
	GtkTreeModel *model;

	if (gtk_tree_selection_get_selected (selection, &model, NULL) == TRUE)
	{
		gtk_widget_set_sensitive (
				lookup_widget (w, "button_find_select"),
				TRUE);
	}
	else {
		gtk_widget_set_sensitive (
				lookup_widget (w, "button_find_select"),
				FALSE);
	}
}


/*
 * Initialize the window
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
dialog_find_init (GtkWidget *w, GtkWidget *dialog_find)
{
	GtkWidget *result_view;
	GtkListStore *store;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;

	/*
	 * Store the provided parent widget to be able to
	 * ask to display and select the found job
	 */
	g_object_set_data (     G_OBJECT (dialog_find),
				"parent_widget", (gpointer)w);

	/* Initialize the result list */
	result_view = lookup_widget (dialog_find, "treeview_find");

	store = gtk_list_store_new (5,
				G_TYPE_STRING,	/* ID (not shown) */
				G_TYPE_STRING,	/* Parent ID (not shown) */
				G_TYPE_STRING,	/* Name */
				G_TYPE_STRING,	/* Type (job or jobset) */
				G_TYPE_STRING);	/* Type ("1", "0") not shown */

	/* Associate the model with the view */
	gtk_tree_view_set_model (	GTK_TREE_VIEW (result_view),
					GTK_TREE_MODEL (store));
	g_object_unref (store);
	gtk_tree_view_set_search_column (GTK_TREE_VIEW (result_view), 2);

	/* Type column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (	_("Type"),
								renderer,
								"text", 3,
								NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (result_view), column);

	/* Name column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (	_("Name"),
								renderer,
								"text", 2,
								NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (result_view), column);
	
	/* Selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (result_view));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (      G_OBJECT (select), "changed",
				G_CALLBACK (tree_selection_changed_cb),
				result_view);
	g_signal_connect(	G_OBJECT (result_view), "row-activated",
				G_CALLBACK (view_onRowActivated), NULL);

	return 0;
}


/*
 * Create a new find dialog
 *
 * Return:
 *   The GtkWidget of the new window (which has been `show'ed by this function)
 *   or NULL in case of error
 */
GtkWidget *
new_dialog_find (GtkWidget *w)
{
	GtkWidget *dialog_find;

	dialog_find = create_dialog_find ();
	if (dialog_find_init (w, dialog_find) != 0) {
		gtk_widget_destroy (dialog_find);
		return NULL;
	}
	else {
		gtk_widget_show (dialog_find);
		return dialog_find;
	}
}


/*
 * Internal structure to exchange parameters between fin_go() and find_job_cb()
 */
struct _find_cb {
	GtkListStore *model;
	guint count;
};


/*
 * Add the provided job/jobset to the model
 *
 * Return:
 *   0 --> No error
 *  -1 --> Database error (a message has been displayed)
 */
static int
find_job_cb (	void *data, const char *id, const char *jobset_id,
		const char *name, const char *job_type)
{
	struct _find_cb *ptr = (struct _find_cb *)data;
	GtkTreeIter iter;
	char *path, *err_msg;

	err_msg = path = NULL;
	if (get_job_path (id, &path, &err_msg) != 0) {
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -1;
	}

	if (path == NULL) {
		return 0;
	}

	ptr->count++;
	gtk_list_store_append (ptr->model, &iter);
	gtk_list_store_set (ptr->model, &iter,
				0, id,
				1, jobset_id,
				2, path,
				3, (job_type == NULL || job_type[0] == '0')
					? _("Jobset"): _("Job"),
				4, job_type,
				-1);
	free (path);
	return 0;
}


/*
 * Seach the job in the database and fill the model with the result
 */
static void
find_go (GtkWidget *w, const gchar *str_to_find)
{
	GtkTreeView *view;
	struct _find_cb str_data; 
	gchar *s;

	view = (GtkTreeView *)lookup_widget (w, "treeview_find");
	str_data.model = (GtkListStore *)gtk_tree_view_get_model (view);
	gtk_list_store_clear (str_data.model);
	str_data.count = 0;
	gtk_widget_set_sensitive (lookup_widget (w, "button_find_select"),
					FALSE);
	
	if (str_to_find == NULL || str_to_find[0] == '\0') {
		gtk_label_set_markup (
				(GtkLabel *)lookup_widget (w, "label_find"),
				_("<b>Result List (0 objects):</b>"));
		return;
	}

	cursor_busy (w);
	sql_jobset_search (	str_to_find,
				find_job_cb,
				&str_data,
				(void (*)(void *, const char*, unsigned int))
					error_window_ignore_errno,
				_("Database error"));
	s = g_strdup_printf (
		(str_data.count == 1)
			? _("<b>Result List (%d object):</b>")
			: _("<b>Result List (%d objects):</b>"),
		str_data.count);
	gtk_label_set_markup ((GtkLabel *)lookup_widget (w, "label_find"), s);
	g_free (s);
	cursor_normal (w);
}


/*
 * Entry activated
 */
void
find_entry_activate (GtkEntry *entry)
{
	find_go (GTK_WIDGET (entry), gtk_entry_get_text (entry));
}


/*
 * Find button callback
 */
void
find_button_clicked (GtkButton *button)
{
	find_entry_activate ((GtkEntry *)lookup_widget (GTK_WIDGET (button),
							"entry_find"));
}

/*-----------------============== End Of File ==============-----------------*/
