/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* host_cb.c -- GUI functions for the host window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <gnutls/gnutls.h>
#include <gnutls/x509.h>

#include <sql_hosts.h>
#include <sql_host_env.h>
#include <message_windows.h>
#include <cursor.h>
#include <env_widget.h>
#include <hosts_cb.h>
#include <certificate_cb.h>
#include <host_cb.h>


/*
 * Free a GString
 */
static void
string_free (gpointer string)
{
	g_string_free ((GString *)string, TRUE);
}


/*
 * Functions used to set values in the `Main' tab
 */
static void
set_id (const char *value, void *data)
{
	GString *value_str;

	/* Store the host id in the widget */
	value_str = g_string_new (value);
	g_object_set_data_full (G_OBJECT (data), "host_id",
				(gpointer)value_str,
				string_free);
}

static void
set_hostname (const char *value, void *data)
{
	GtkWidget *entry;

	entry = lookup_widget (GTK_WIDGET (data), "entry_host_name");
	gtk_entry_set_text (GTK_ENTRY (entry), value);
	gtk_window_set_title (GTK_WINDOW (data), value);
}

static void
set_port (const char *value, void *data)
{
	GtkWidget *entry;

	entry = lookup_widget (GTK_WIDGET (data), "entry_host_port");
	gtk_entry_set_text (GTK_ENTRY (entry), value);
}

static void
set_useSSL (const char *value, void *data)
{
	GtkWidget *check;

	check = lookup_widget (GTK_WIDGET (data), "checkbutton_host_SSL");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
			(value == NULL || value[0] == '0') ? FALSE : TRUE);
}

static void
set_certificate (const char *value, void *data)
{
	GString *value_str;
	GtkWidget *button_view;

	if (value == NULL || value[0] == '\0') { 
		button_view = lookup_widget (GTK_WIDGET (data),
					"button_host_view_certificate");
		gtk_widget_set_sensitive (button_view, FALSE);
	}
	else {
		/* Store the certificate in the widget */
		value_str = g_string_new (value);
		g_object_set_data_full (G_OBJECT (data), "certificate",
				(gpointer)value_str,
				string_free);
	}
}

static void
set_description (const char *value, void *data)
{
	GtkWidget *text;
	GtkTextBuffer *buffer;

	text = lookup_widget (GTK_WIDGET (data), "textview_host_descr");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_set_text (buffer, value, -1);
}


/*
 * Initialize the window
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
dialog_host_init (	GtkTreeView *view, GtkWidget *dialog_edit,
			const gchar *host_id)
{
	GtkWidget *env_view, *view_cert;
	env_widget_ptr ptr;

	gtk_window_set_type_hint (	GTK_WINDOW (dialog_edit),
					GDK_WINDOW_TYPE_HINT_NORMAL);

	/*
	 * Store the tree view of the host list to be able to
	 * refresh it when a host is updated
	 */
	g_object_set_data (	G_OBJECT (dialog_edit),
				"host_view", (gpointer)view);

	/* Initialize the environment list */
	env_view = lookup_widget (dialog_edit, "treeview_host_env");
	ptr = env_widget_new (	env_view,
				"button_host_env_add",
				"button_host_env_remove",
				"button_host_env_up",
				"button_host_env_down",
				"button_host_env_top",
				"button_host_env_bottom");
	if (env_widget_build_sql (ptr, host_id, sql_hosts_env_list) != 0) {
		env_widget_destroy (ptr);
		return -1;
	}

	if (host_id == NULL || host_id[0] == '\0') {
		view_cert = lookup_widget (dialog_edit,
					"button_host_view_certificate");
		gtk_widget_set_sensitive (view_cert, FALSE);
		return 0;
	}

	/* Retrieve the host main parameters */
	if (sql_host_get_main (	host_id,
				set_id,
				set_hostname,
				set_port,
				set_useSSL,
				set_certificate,
				set_description,
				dialog_edit,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	return 0;
}


/*
 * Create a new host dialog for the provided host id
 *
 * Return:
 *   The GtkWidget of the new window (which has been `show'ed by this function)
 *   or NULL in case of error
 */
GtkWidget *
new_dialog_host (GtkTreeView *view, const gchar *host_id)
{
	GtkWidget *dialog_edit;

	dialog_edit = create_dialog_host ();
	if (dialog_host_init (view, dialog_edit, host_id) != 0) {
		gtk_widget_destroy (dialog_edit);
		return NULL;
	}
	else {
		gtk_widget_show (dialog_edit);
		return dialog_edit;
	}
}


/*
 * Check that the provided values for the main parameters are alright and
 * can be stored in the database
 *
 * Return:
 *   0 --> Parameters OK
 *  -1 --> Error (an error popup has been displayed)
 */
static int
check_host_values (	const gchar *hostname, const gchar *portnum,
			char sslenable, const gchar *sslcert)
{
	GString *err_msg;
	int err;

	err_msg = g_string_new ("");
	err = 0;

	if (hostname == NULL || hostname[0] == '\0') {
		err_msg = g_string_append (err_msg,
				_("The Host Name field is required.\n"));
		err++;
	}
	if (portnum == NULL || portnum[0] == '\0') {
		err_msg = g_string_append (err_msg,
				_("The Port field is required.\n"));
		err++;
	}
	if (sslenable == 1 && (sslcert == NULL || sslcert[0] == '\0')) {
		err_msg = g_string_append (err_msg,
	_("As SSL is enabled, the client SSL certificate is required.\n"));
		err++;
	}
	switch (err) {
		case 0:
			g_string_free (err_msg, TRUE);
			return 0;

		case 1:
			error_window (_("A required information is missing"),
					err_msg->str);
			g_string_free (err_msg, TRUE);
			break;
		default:
			error_window (_("Required informations are missing"),
					err_msg->str);
			g_string_free (err_msg, TRUE);
			break;
	}
	return -1;
}


/*
 * Retrieve the main host parameters from the widgets
 *
 * Return:
 *   0 --> No error (*hostname, *portnum, ssenable, *sslcert and *description
 *         are set - only *sslcert and *description must be freed by
 *         the caller)
 *  -1 --> Error (a required parameter is missing - an error popup has been
 *         displayed for the user)
 */
static int
get_host_values (	GtkWidget *dialog_edit,
			const gchar **hostname, const gchar **portnum,
			char *sslenable, gchar **sslcert,
			gchar **description)
{
	GtkWidget *entry, *check, *text;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	GString *cert_str;

	entry = lookup_widget (dialog_edit, "entry_host_name");
	*hostname = gtk_entry_get_text (GTK_ENTRY (entry));

	entry = lookup_widget (dialog_edit, "entry_host_port");
	*portnum = gtk_entry_get_text (GTK_ENTRY (entry));

	check = lookup_widget (dialog_edit, "checkbutton_host_SSL");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check)) == TRUE) {
		*sslenable = 1;
	}
	else {
		*sslenable = 0;
	}

	cert_str = g_object_get_data (G_OBJECT (dialog_edit), "certificate");
	if (cert_str == NULL || (cert_str->str)[0] == '\0') {
		*sslcert = NULL;
	}
	else {
		*sslcert = g_strdup (cert_str->str);
	}

	text = lookup_widget (dialog_edit, "textview_host_descr");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	*description = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	return check_host_values (*hostname, *portnum, *sslenable, *sslcert);
}


/*
 * Store a new host in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
host_create (GtkWidget *dialog_edit)
{
	const gchar *hostname, *portnum;
	gchar *sslcert = NULL;
	gchar *description = NULL;
	char sslenable;
	unsigned long int host_id;
	GString *host_id_str;
	GtkWidget *env_view;

	/*
	 * Add the new host main parameters to the database
	 */
	if (	   get_host_values (	dialog_edit,
					&hostname, &portnum,
					&sslenable, &sslcert,
					&description) != 0
		|| sql_host_new (	hostname, portnum, sslenable, sslcert,
					description, &host_id,
				(void (*)(void *, const char*, unsigned int))
						error_window_check_duplicate,
					_("Database error")) != 0)
	{
		g_free (sslcert);
		g_free (description);
		return -1;
	}
	g_free (sslcert);
	g_free (description);


	/*
	 * Add the host environments to the database
	 */
	host_id_str = g_string_new ("");
	g_string_printf (host_id_str, "%lu", host_id);
	env_view = lookup_widget (dialog_edit, "treeview_host_env");
	if (env_widget_save_list (	env_widget_get_from_view (env_view),
					host_id_str->str,
					sql_hosts_env_add) != 0)
	{
		g_string_free (host_id_str, TRUE);
		return -1;
	}
	g_string_free (host_id_str, TRUE);
	return 0;
}


/*
 * Update a host in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
host_update (GtkWidget *dialog_edit, GString *host_id)
{
	GtkWidget *env_view;
	const gchar *hostname, *portnum;
	gchar *sslcert = NULL;
	gchar *description = NULL;
	char sslenable;

	/*
	 * Update the host main parameters
	 */
	if (	   get_host_values (	dialog_edit, &hostname, &portnum,
					&sslenable, &sslcert,
					&description) != 0
		|| sql_host_update (	host_id->str,
					hostname, portnum,
					sslenable, sslcert, description,
				(void (*)(void *, const char*, unsigned int))
						error_window_check_duplicate,
					_("Database error")) != 0)
	{
		g_free (sslcert);
		g_free (description);
		return -1;
	}
	g_free (sslcert);
	g_free (description);


	/*
	 * Update the environments
	 */

	/* First, delete the old environment list from the database */
	if (sql_hosts_env_delete (host_id->str,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	/* Now, add the host environment */
	env_view = lookup_widget (dialog_edit, "treeview_host_env");
	if (env_widget_save_list (	env_widget_get_from_view (env_view),
					host_id->str,
					sql_hosts_env_add) != 0)
	{
		return -1;
	}
	return 0;
}


/*
 * Callback for the `OK' button
 */
void
host_ok_clicked (GtkButton *button)
{
	GtkWidget *dialog_edit;
	GtkTreeView *view;
	GString *host_id;
	int ret;

	cursor_busy (GTK_WIDGET (button));
	dialog_edit = lookup_widget (GTK_WIDGET (button), "dialog_host");
	host_id = g_object_get_data (G_OBJECT (dialog_edit), "host_id");
	view = g_object_get_data (G_OBJECT (dialog_edit), "host_view");
	if (host_id == NULL) {
		ret = host_create (dialog_edit);
	}
	else {
		ret = host_update (dialog_edit, host_id);
	}
	cursor_normal (GTK_WIDGET (button));
	if (ret == 0) {
		hosts_refresh (view);
		gtk_widget_destroy (dialog_edit);
	}
}


/*
 * Callback for the `View certificate' button
 */
void
host_view_certificate_view_clicked (GtkButton *button)
{
	GtkWidget *dialog_edit;
	GString *cert;

	dialog_edit = lookup_widget (GTK_WIDGET (button), "dialog_host");
	cert = g_object_get_data (G_OBJECT (dialog_edit), "certificate");
	new_dialog_certificate (cert);
}


/*
 * Callback for the `Upload certificate' button
 */
void
host_upload_certificate_view_clicked (GtkButton *button)
{
	GtkWidget *dialog_edit;
	GtkWidget *dialog;
	GtkWidget *button_view;
	gchar *filename, *cert;
	gsize len;
	GString *cert_str;

	gnutls_datum_t certificate_string;
	gnutls_x509_crt_t certificate_cert;
	int ret;
	GError *err = NULL;

	dialog_edit = lookup_widget (GTK_WIDGET (button), "dialog_host");

	/* File chooser */
	dialog = gtk_file_chooser_dialog_new (
					"Upload certificate",
					GTK_WINDOW (dialog_edit),
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {

		/* Get the input file name */
		filename = gtk_file_chooser_get_filename (
						GTK_FILE_CHOOSER (dialog));

		/* Load the content of the file */
		if (g_file_get_contents (filename, &cert, &len, &err) == FALSE)
		{
			gtk_widget_destroy (dialog);
			error_window (_("Failed to read the certificate"),
					(err != NULL) ? err->message : NULL);
			if (err != NULL) {
				g_error_free (err);
			}
			g_free (filename);
			return;
		}
		if (err != NULL) {
			g_error_free (err);
		}
		g_free (filename);

		/*
		 * Check if it's a certificate
		 */
		gnutls_x509_crt_init (&certificate_cert);
		certificate_string.data = (unsigned char *)cert;
		certificate_string.size = len;

		/* Check if it's a PEM encoded certificate */ 
		ret = gnutls_x509_crt_import (	certificate_cert,
						&certificate_string,
						GNUTLS_X509_FMT_PEM);
		if (ret != 0) {
			/*
			 * Not a PEM encoded certificate so check if it's a
			 * DER encoded one.
			 */
			if (gnutls_x509_crt_import (certificate_cert,
						&certificate_string,
						GNUTLS_X509_FMT_DER) != 0)
			{
				gnutls_x509_crt_deinit (certificate_cert);
				gtk_widget_destroy (dialog);
				error_window (	_("Certificate error"),
						gnutls_strerror (ret));
				g_free (cert);
				return;
			}
		}
		gnutls_x509_crt_deinit (certificate_cert);

		/* Store the certificate */
		cert_str = g_string_new (cert);
		g_object_set_data_full (G_OBJECT (dialog_edit), "certificate",
				(gpointer)cert_str,
				string_free);
		g_free (cert);
		button_view = lookup_widget (GTK_WIDGET (button),
					"button_host_view_certificate");
		gtk_widget_set_sensitive (button_view, TRUE);
	}
	gtk_widget_destroy (dialog);
}

/*------------------------======= End Of File =======------------------------*/
