/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
     
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
     
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * jobset_item.c -- Structure associated with a jobset to draw
 *                  on a GnomeCanvas
 */

#include <schedwi.h>

#include <sql_hierarchy.h>
#include <message_windows.h>
#include <cache_pixbuf.h>
#include <child_job.h>
#include <link.h>
#include <background.h>
#include <sql_children_job.h>
#include <sql_jobsets.h>
#include <schedwi_style.h>
#include <canvas_utils.h>
#include <main_cb.h>
#include <jobset_item.h>

#define CANVAS_DEFAULT_WIDTH 800
#define CANVAS_DEFAULT_HEIGHT 600

static gboolean get_jobset_details_from_database OF((const gchar *id,
						jobset_item_t *ptr,
						canvas_mode_t mode));

/*
 * RAZ the provided jobset_item_t object
 */
void
jobset_item_raz (jobset_item_t *ptr)
{
	if (ptr != NULL) {
		if (ptr->path != NULL) {
			free (ptr->path);
		}
		g_free (ptr->id);
		destroy_job_select (ptr->selection);
		destroy_jobset_background (ptr->background);
		destroy_grid (ptr->grid);
		destroy_child_job_list (ptr->jobs);
		destroy_link_list (ptr->links);
                if (ptr->group_grid != NULL) {
                        gtk_object_destroy (GTK_OBJECT (ptr->group_grid));
                }
                if (ptr->group_links != NULL) {
                        gtk_object_destroy (GTK_OBJECT (ptr->group_links));
                }
                if (ptr->group_jobs != NULL) {
                        gtk_object_destroy (GTK_OBJECT (ptr->group_jobs));
                }
		job_tooltips_destroy (ptr->tooltips);
		ptr->path        = NULL;
		ptr->id          = NULL;
		ptr->selection   = NULL;
		ptr->background  = NULL;
		ptr->grid        = NULL;
		ptr->jobs        = NULL;
		ptr->links       = NULL;
		ptr->group_grid  = NULL;
		ptr->group_links = NULL;
		ptr->group_jobs  = NULL;
		ptr->tooltips    = NULL;
	}
}


/*
 * Destroy a jobset_item_t object
 */
void
destroy_jobset_item (jobset_item_t *ptr)
{
	if (ptr != NULL) {
		jobset_item_raz (ptr);
		g_free (ptr);
	}
}


/*
 * Load an icon from the database, and store it in the pixbuf cache
 *
 * Return:
 *   TRUE --> No error.  pixbufs is filled with the pixbufs associated with
 *            the retrieved icon (the icon as stored in the database, the
 *            same icon highlighted, darkened and darkened-highlighted)
 *            These returned pixbufs belongs to the cache and must not be
 *            freed/unreferenced by the caller
 *  FALSE --> Error.  An error message popup has been displayed
 */
static gboolean
load_icon (lwc_LL *list, const gchar *tablename, GdkPixbuf **pixbufs)
{
	char *pixdata_stream, *err_msg;
	unsigned long int pixdata_stream_len;
	gchar *s;

	/* Retrieve the serialize pixdata from the database */
	if (get_job_parameter (	list, tablename, "icon",
				&pixdata_stream, &pixdata_stream_len,
				&err_msg) != 0)
	{
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
		return FALSE;
	}

	/* A serialize pixdata has been found in the database */
	if (pixdata_stream != NULL) {
		/*
		 * Deserialize the pixdata, built the associated pixbufs with
		 * different highlight levels, store them in the cache and
		 * return a pointer to these pixbufs
		 */
		if (add_pixbuf_to_cache (pixdata_stream, pixdata_stream_len,
				&(pixbufs[DEFAULT]),
				&(pixbufs[HIGHLIGHTED]),
				&(pixbufs[SELECTED]),
				&(pixbufs[SELECTED_HIGHLIGHTED]),
				&(pixbufs[CUT]),
				&err_msg) != TRUE)
		{
			s = g_strdup_printf (
			_("%s.\nA replacement icon will be used instead."),
				err_msg);
			g_free (err_msg);
			warning_window (_("Could not load an icon"), s);
			g_free (s);
			pixbufs[DEFAULT] = NULL;
			pixbufs[HIGHLIGHTED] = NULL;
			pixbufs[SELECTED] = NULL;
			pixbufs[SELECTED_HIGHLIGHTED] = NULL;
			pixbufs[CUT] = NULL;
			
		}
		free (pixdata_stream);
	}
	else {
		/* No icon found in the database */
		pixbufs[DEFAULT] = NULL;
		pixbufs[HIGHLIGHTED] = NULL;
		pixbufs[SELECTED] = NULL;
		pixbufs[SELECTED_HIGHLIGHTED] = NULL;
		pixbufs[CUT] = NULL;
	}

	return TRUE;
}


/*
 * Set the width and height of the canvas
 */
static void
set_canvas_size (const char *width, const char *height, void *data)
{
	jobset_item_t *ptr = data;

	if (width == NULL || (ptr->canvas_width = atoi (width)) <= 0) {
		ptr->canvas_width = CANVAS_DEFAULT_WIDTH;
	}
	if (height == NULL || (ptr->canvas_height = atoi (height)) <= 0) {
		ptr->canvas_height = CANVAS_DEFAULT_HEIGHT;
	}
}


/*
 * Add a link to the list of links
 */
void
jobset_item_add_link (jobset_item_t *ptr, link_t *link)
{
	if (ptr != NULL && link != NULL) {
		ptr->links = g_slist_prepend (ptr->links, link);
	}
}


/*
 * Initialize the provided jobset_item_t object
 *
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed)
 */
static gboolean
jobset_item_init (jobset_item_t *ptr, const gchar *id, canvas_mode_t mode)
{
	/* Retrieve the jobset details from the database */
	if (get_jobset_details_from_database (id, ptr, mode) == FALSE) {
		return FALSE;
	}

	/* Retrieve the jobs/jobsets below this jobset */
	ptr->jobs = get_child_job_list (id, ptr);

	/* Retrieve the links between the jobs/jobsets below this jobset */
	ptr->links = get_link_list (ptr->jobs);

	/* Build the grid */
	ptr->grid = new_grid ();

	/* Retrieve the colors from gtk/theme */
	schedwi_style_get_text_color (	GTK_STATE_NORMAL,
					&(ptr->text_color[DEFAULT]));
	schedwi_style_get_text_color (	GTK_STATE_PRELIGHT,
					&(ptr->text_color[HIGHLIGHTED]));
	schedwi_style_get_text_color (	GTK_STATE_SELECTED,
					&(ptr->text_color[SELECTED]));
	schedwi_style_get_text_color (	GTK_STATE_SELECTED,
				&(ptr->text_color[SELECTED_HIGHLIGHTED]));
	schedwi_style_get_text_color (	GTK_STATE_INSENSITIVE,
					&(ptr->text_color[CUT]));

	schedwi_style_get_base_color (	GTK_STATE_NORMAL,
					&(ptr->base_color[DEFAULT]));
	schedwi_style_get_base_color (	GTK_STATE_PRELIGHT,
					&(ptr->base_color[HIGHLIGHTED]));
	schedwi_style_get_base_color (	GTK_STATE_SELECTED,
					&(ptr->base_color[SELECTED]));
	schedwi_style_get_base_color (	GTK_STATE_SELECTED,
				&(ptr->base_color[SELECTED_HIGHLIGHTED]));
	schedwi_style_get_base_color (	GTK_STATE_INSENSITIVE,
					&(ptr->base_color[CUT]));

	/* Initialize the selection buffer */
	ptr->selection = new_job_select ();

	/* Create the tooltip object */
	if (mode == ACTIVE) {
		ptr->tooltips = job_tooltips_new ();
	}
	else {
		ptr->tooltips = NULL;
	}

	return TRUE;
}



/*
 * Redraw the canvas
 */
static void
jobset_item_reload (jobset_item_t *ptr)
{
	void *current_selection;
	gchar *id;
	canvas_mode_t mode;
	GnomeCanvas *canvas;
	GtkEntry *location;
	gboolean grid_show, grid_align;

	if (ptr == NULL) {
		return;
	}

	/* Save some data */
	current_selection = jobset_save_selection (ptr);
	id = g_strdup (ptr->id);
	mode = ptr->mode;
	canvas = ptr->canvas;
	location = ptr->location;
	grid_show = ptr->grid_show;
	grid_align = ptr->grid_align;


	/* RAZ */
	jobset_item_raz (ptr);

	/* Init */
	jobset_item_init (ptr, id, mode); 	
	g_free (id);

	/* Draw */
	jobset_item_draw_init (ptr, canvas, location, grid_show, grid_align);

	/* Restore the selection */
	jobset_restore_selection (ptr, current_selection);
}


/*
 * Retrieve the details of the jobset from the database
 *
 * Return:
 *   TRUE --> No error
 *  FALSE --> Error.  An error message has been displayed
 */
static gboolean
get_jobset_details_from_database (	const gchar *id, jobset_item_t *ptr,
					canvas_mode_t mode)
{
	char *err_msg;
	lwc_LL *list;

	/*
	 * Retrieve the full path of the job.  This path will be displayed
	 * in the location toolbar
	 */
	if (get_job_path (id, &(ptr->path), &err_msg) != 0) {
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
		return FALSE;
	}

	/*
	 * Retrieve the canvas width and height for this jobset
	 */
	if (sql_jobset_get_canvas_size (
				id,
				set_canvas_size,
				ptr,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		if (ptr->path != NULL) {
			free (ptr->path);
		}
		return FALSE;
	}


	/* Copy the jobset ID and the mode */
	ptr->id = g_strdup (id);
	ptr->mode = mode;

	/*
	 * Build the hierarchy list of the jobset.  This list is required
	 * for the following calls to get_job_parameters()
	 */
	list = get_hierarchy_list (id, &err_msg);
	if (list == NULL) {
		g_free (ptr->id);
		if (ptr->path != NULL) {
			free (ptr->path);
		}
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
		return FALSE;
	}

	/* Retrieve the background/wallpaper parameters */
	ptr->background = new_jobset_background (list, id,
					(void (*)(void *)) jobset_item_reload,
					ptr);
	if (ptr->background == NULL) {
		lwc_delLL (list, (void (*)(const void *))free);
		g_free (ptr->id);
		if (ptr->path != NULL) {
			free (ptr->path);
		}
		return FALSE;
	}

	/*
	 * Load the default icons for the jobs/jobsets that will be drawn
	 * on the canvas.  This avoids to have to retrieve from the whole
	 * hierarchy the icons for each job/jobset to be drawn
	 */
	if (	   load_icon (
			list,
			job_status_state_to_job_icon_tablename (
						JOB_STATUS_STATE_UNDEFINED),
			ptr->job_icon[JOB_STATUS_STATE_UNDEFINED]) != TRUE
		|| load_icon (
			list,
			job_status_state_to_jobset_icon_tablename (
						JOB_STATUS_STATE_UNDEFINED),
			ptr->jobset_icon[JOB_STATUS_STATE_UNDEFINED]) != TRUE)
	{
		lwc_delLL (list, (void (*)(const void *))free);
		g_free (ptr->id);
		if (ptr->path != NULL) {
			free (ptr->path);
		}
		return FALSE;
	} 
	/*
	 * In editing mode, copy the default icon to the other ones.
	 * However, this shouldn't be necessary as the other icons won't be
	 * used in editing mode
	 */  
	if (mode == EDITING) {
		ptr->job_icon[JOB_STATUS_STATE_COMPLETED][DEFAULT] =
		ptr->job_icon[JOB_STATUS_STATE_WAITING][DEFAULT] =
		ptr->job_icon[JOB_STATUS_STATE_RUNNING][DEFAULT] =
		ptr->job_icon[JOB_STATUS_STATE_FAILED][DEFAULT] =
		ptr->job_icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT];

		ptr->job_icon[JOB_STATUS_STATE_COMPLETED][HIGHLIGHTED] =
		ptr->job_icon[JOB_STATUS_STATE_WAITING][HIGHLIGHTED] =
		ptr->job_icon[JOB_STATUS_STATE_RUNNING][HIGHLIGHTED] =
		ptr->job_icon[JOB_STATUS_STATE_FAILED][HIGHLIGHTED] =
		ptr->job_icon[JOB_STATUS_STATE_UNDEFINED][HIGHLIGHTED];

		ptr->job_icon[JOB_STATUS_STATE_COMPLETED][SELECTED] =
		ptr->job_icon[JOB_STATUS_STATE_WAITING][SELECTED] =
		ptr->job_icon[JOB_STATUS_STATE_RUNNING][SELECTED] =
		ptr->job_icon[JOB_STATUS_STATE_FAILED][SELECTED] =
		ptr->job_icon[JOB_STATUS_STATE_UNDEFINED][SELECTED];

		ptr->job_icon[JOB_STATUS_STATE_COMPLETED][SELECTED_HIGHLIGHTED] =
		ptr->job_icon[JOB_STATUS_STATE_WAITING][SELECTED_HIGHLIGHTED] =
		ptr->job_icon[JOB_STATUS_STATE_RUNNING][SELECTED_HIGHLIGHTED] =
		ptr->job_icon[JOB_STATUS_STATE_FAILED][SELECTED_HIGHLIGHTED] =
		ptr->job_icon[JOB_STATUS_STATE_UNDEFINED][SELECTED_HIGHLIGHTED];

		ptr->job_icon[JOB_STATUS_STATE_COMPLETED][CUT] =
		ptr->job_icon[JOB_STATUS_STATE_WAITING][CUT] =
		ptr->job_icon[JOB_STATUS_STATE_RUNNING][CUT] =
		ptr->job_icon[JOB_STATUS_STATE_FAILED][CUT] =
		ptr->job_icon[JOB_STATUS_STATE_UNDEFINED][CUT];

		ptr->jobset_icon[JOB_STATUS_STATE_COMPLETED][DEFAULT] =
		ptr->jobset_icon[JOB_STATUS_STATE_WAITING][DEFAULT] =
		ptr->jobset_icon[JOB_STATUS_STATE_RUNNING][DEFAULT] =
		ptr->jobset_icon[JOB_STATUS_STATE_FAILED][DEFAULT] =
		ptr->jobset_icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT];

		ptr->jobset_icon[JOB_STATUS_STATE_COMPLETED][HIGHLIGHTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_WAITING][HIGHLIGHTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_RUNNING][HIGHLIGHTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_FAILED][HIGHLIGHTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_UNDEFINED][HIGHLIGHTED];

		ptr->jobset_icon[JOB_STATUS_STATE_COMPLETED][SELECTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_WAITING][SELECTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_RUNNING][SELECTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_FAILED][SELECTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_UNDEFINED][SELECTED];

		ptr->jobset_icon[JOB_STATUS_STATE_COMPLETED][SELECTED_HIGHLIGHTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_WAITING][SELECTED_HIGHLIGHTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_RUNNING][SELECTED_HIGHLIGHTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_FAILED][SELECTED_HIGHLIGHTED] =
		ptr->jobset_icon[JOB_STATUS_STATE_UNDEFINED][SELECTED_HIGHLIGHTED];

		ptr->jobset_icon[JOB_STATUS_STATE_COMPLETED][CUT] =
		ptr->jobset_icon[JOB_STATUS_STATE_WAITING][CUT] =
		ptr->jobset_icon[JOB_STATUS_STATE_RUNNING][CUT] =
		ptr->jobset_icon[JOB_STATUS_STATE_FAILED][CUT] =
		ptr->jobset_icon[JOB_STATUS_STATE_UNDEFINED][CUT];
	}
	else {
		if (	   load_icon (list,
				job_status_state_to_job_icon_tablename (
						JOB_STATUS_STATE_COMPLETED),
				ptr->job_icon[JOB_STATUS_STATE_COMPLETED]) != TRUE
			|| load_icon (list,
				job_status_state_to_job_icon_tablename (
						JOB_STATUS_STATE_FAILED),
				ptr->job_icon[JOB_STATUS_STATE_FAILED]) != TRUE
			|| load_icon (list,
				job_status_state_to_job_icon_tablename (
						JOB_STATUS_STATE_RUNNING),
				ptr->job_icon[JOB_STATUS_STATE_RUNNING]) != TRUE
			|| load_icon (list,
				job_status_state_to_job_icon_tablename (
						JOB_STATUS_STATE_WAITING),
				ptr->job_icon[JOB_STATUS_STATE_WAITING]) != TRUE

			|| load_icon (list,
				job_status_state_to_jobset_icon_tablename (
						JOB_STATUS_STATE_COMPLETED),
				ptr->jobset_icon[JOB_STATUS_STATE_COMPLETED]) != TRUE
			|| load_icon (list,
				job_status_state_to_jobset_icon_tablename (
						JOB_STATUS_STATE_FAILED),
				ptr->jobset_icon[JOB_STATUS_STATE_FAILED]) != TRUE
			|| load_icon (list,
				job_status_state_to_jobset_icon_tablename (
						JOB_STATUS_STATE_RUNNING),
				ptr->jobset_icon[JOB_STATUS_STATE_RUNNING]) != TRUE
			|| load_icon (list,
				job_status_state_to_jobset_icon_tablename (
						JOB_STATUS_STATE_WAITING),
				ptr->jobset_icon[JOB_STATUS_STATE_WAITING]) != TRUE)
		{
			lwc_delLL (list, (void (*)(const void *))free);
			g_free (ptr->id);
			if (ptr->path != NULL) {
				free (ptr->path);
			}
			return FALSE;
		}
		ptr->jobset_icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT] =
			ptr->jobset_icon[JOB_STATUS_STATE_UNDEFINED][CUT];
		ptr->job_icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT] =
			ptr->job_icon[JOB_STATUS_STATE_UNDEFINED][CUT];
	}

	lwc_delLL (list, (void (*)(const void *))free);
	return TRUE;
}


/*
 * Create and return a new initialized jobset_item_t object
 *
 * Return:
 *   The new object to be freed by the caller by destroy_jobset_item() or
 *   NULL in case of error (an error message has been displayed)
 */
jobset_item_t *
new_jobset_item (const gchar *id, canvas_mode_t mode)
{
	jobset_item_t *ptr;

	ptr = g_new0 (jobset_item_t, 1);
	
	if (jobset_item_init (ptr, id, mode) == FALSE) {
		return NULL;
	}
	return ptr;
}


/*
 * Draw the provided link
 */
void
jobset_item_draw_link (jobset_item_t *ptr, link_t *link)
{
	link_draw (link, ptr->group_links);
}


/*
 * Draw the jobset on the provided canvas
 */ 
void
jobset_item_draw_init (	jobset_item_t *ptr,
			GnomeCanvas *canvas,
			GtkEntry *location,
			gboolean grid_show, gboolean grid_align)
{
	if (ptr != NULL) {
		/* Initialize the canvas */
		canvas_utils_set_size (	canvas,
					ptr->canvas_width, ptr->canvas_height);
		g_object_set_data (G_OBJECT (canvas), "jobset", ptr);
		ptr->canvas = canvas;
		ptr->location = location;
		ptr->grid_show = grid_show;
		ptr->grid_align = grid_align;

		/* Draw the background */
		jobset_background_draw (ptr->background, canvas,
					ptr->canvas_width,
					ptr->canvas_height);

		/* Initialize the grid */
		if (ptr->group_grid != NULL) {
			hide_grid (ptr->grid);
                        gtk_object_destroy (GTK_OBJECT (ptr->group_grid));
		}
			
		ptr->group_grid = (GnomeCanvasGroup *)gnome_canvas_item_new (
                                        gnome_canvas_root (canvas),
                                        gnome_canvas_group_get_type (),
                                        "x", (gdouble)0,
                                        "y", (gdouble)0,
                                        NULL);
		set_grid_parent_group (	ptr->grid, ptr->group_grid,
					ptr->canvas_width,
					ptr->canvas_height);
		if (grid_show == TRUE) {
			show_grid (ptr->grid);
		}
		snap_to_grid (ptr->grid, grid_align);

		/* Draw the links group */
		ptr->group_links = (GnomeCanvasGroup *)gnome_canvas_item_new (
                                        gnome_canvas_root (canvas),
                                        gnome_canvas_group_get_type (),
                                        "x", (gdouble)0,
                                        "y", (gdouble)0,
                                        NULL);

		/* Draw the jobs group */
		ptr->group_jobs = (GnomeCanvasGroup *)gnome_canvas_item_new (
                                        gnome_canvas_root (canvas),
                                        gnome_canvas_group_get_type (),
                                        "x", (gdouble)0,
                                        "y", (gdouble)0,
                                        NULL);

		/* Draw the jobs and the links */
		child_job_draw_list (ptr->jobs, ptr->group_jobs);
		link_draw_list (ptr->links, ptr->group_links);

		/* Fill the location entry */
		gtk_entry_set_text (location, ptr->path);
	}
}


/*
 * Scroll the view to `see' the world box (centered around (x,y))
 */
void
jobset_item_adjust_view (	jobset_item_t *ptr,
				gdouble x, gdouble y,
				gint width, gint height)
{
	GtkAdjustment *adj_v, *adj_h;
	gdouble cx, cy, adj_x, adj_y, cw, ch, page_size_h, page_size_v;


	if (ptr != NULL) {
		gnome_canvas_w2c_d (ptr->canvas, x, y, &cx, &cy);
		gnome_canvas_w2c_d (ptr->canvas, width, height, &cw, &ch);
		cw /= 2.0;
		ch /= 2.0;

		adj_h = gtk_layout_get_hadjustment (GTK_LAYOUT (ptr->canvas));
		adj_v = gtk_layout_get_vadjustment (GTK_LAYOUT (ptr->canvas));
		adj_x = gtk_adjustment_get_value (adj_h);
		adj_y = gtk_adjustment_get_value (adj_v);
		if (cx - cw < adj_x) {
			gtk_adjustment_set_value (adj_h, MAX (0, cx - cw));
			gtk_adjustment_value_changed (adj_h);
		}
		if (cy - ch < adj_y) {
			gtk_adjustment_set_value (adj_v, MAX (0, cy - ch));
			gtk_adjustment_value_changed (adj_v);
		}

		g_object_get (	(gpointer)adj_h,
				"page-size", &page_size_h, NULL);
		g_object_get (	(gpointer)adj_v,
				"page-size", &page_size_v, NULL);
		if (cx + cw > adj_x + page_size_h) {
			gtk_adjustment_set_value (	adj_h,
							cx + cw - page_size_h);
			gtk_adjustment_value_changed (adj_h);
		}
		if (cy + ch> adj_y + page_size_v) {
			gtk_adjustment_set_value (	adj_v,
							cy + ch - page_size_v);
			gtk_adjustment_value_changed (adj_v);
		}
	}
}


/*
 * Select the provided job
 */
void
jobset_item_select_job (jobset_item_t *ptr, child_job_t *job_ptr)
{
	if (job_ptr != NULL) {
		job_select_job (ptr->selection, job_ptr, TRUE);
	}
}


/*
 * Select the provided job ID
 */
void
jobset_item_select_job_by_id (jobset_item_t *ptr, const gchar *id)
{
	child_job_t *job_ptr;

	job_ptr = child_job_find (ptr->jobs, id);
	if (job_ptr != NULL) {
		job_select_job (ptr->selection, job_ptr, TRUE);
	}
}


/*
 * Selected the link connecting the two provided job IDs
 */
static void
jobset_item_select_link_by_id (	jobset_item_t *ptr,
				const gchar *id_src,
				const gchar *id_dst)
{
	link_t *link_ptr;

	link_ptr = link_find (ptr->links, id_src, id_dst);
	if (link_ptr != NULL) {
		job_select_link (ptr->selection, link_ptr, TRUE);
	}
}


/*
 * Internal structure to save the current selection
 */
struct _jobset_save_selection {
	item_type_t selected_item_type;
	gchar *job_id1, *job_id2;
};
typedef struct _jobset_save_selection jobset_save_selection_t;


/*
 * Store the currently selected object in the private returned object
 */
void *
jobset_save_selection (jobset_item_t *ptr)
{
	jobset_save_selection_t *save;

	if (ptr->selection == NULL || ptr->selection->set == FALSE) {
		return NULL;
	}

	save = (jobset_save_selection_t *) g_new0 (jobset_save_selection_t, 1);
	save->selected_item_type = ptr->selection->item_type;
	switch (ptr->selection->item_type) {
		case JOB_SELECT_TYPE_JOB:
			save->job_id1 = g_strdup (
						ptr->selection->item.job->id);
			save->job_id2 = NULL;
			break;

		default:
			save->job_id1 = g_strdup (
					ptr->selection->item.link->src->id);
			save->job_id2 = g_strdup (
					ptr->selection->item.link->dst->id);
			break;
	}
	return save;
}


/*
 * Restore the previously saved selection
 */
void
jobset_restore_selection (jobset_item_t *ptr, void *data)
{
	jobset_save_selection_t *save = (jobset_save_selection_t *)data;

	if (save == NULL) {
		return;
	}

	switch (save->selected_item_type) {
		case JOB_SELECT_TYPE_JOB:
			jobset_item_select_job_by_id (ptr, save->job_id1);
			break;

		default:
			jobset_item_select_link_by_id (	ptr,
							save->job_id1,
							save->job_id2);
			break;
	}
	g_free (save->job_id1);
	g_free (save->job_id2);
	g_free (save);
}


/*
 * Dump the previously saved selection
 */
void
jobset_dump_saved_selection (void *data)
{
	jobset_save_selection_t *save = (jobset_save_selection_t *)data;

	if (save != NULL) {
		g_free (save->job_id1);
		g_free (save->job_id2);
		g_free (save);
	}
}

/*-----------------============== End Of File ==============-----------------*/
