/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
     
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
     
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * jobset_item.h -- Structure associated with a jobset to draw
 *                  on a GnomeCanvas
 */

#ifndef _JOBSET_ITEM_H
#define _JOBSET_ITEM_H

#include <schedwi.h>
#include <gtk/gtk.h>
#include <libgnomecanvas/libgnomecanvas.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <job_status_state.h>
#include <background.h>
#include <job_select.h>
#include <grid.h>
#include <job_tooltips.h>


/* Allow the use in C++ code.  */
#ifdef __cplusplus
extern "C" {
#endif


/*
 * Mode:
 *   EDITING to add/remove/configure jobs and jobsets.  In this mode all the
 *           icons are displayed with there default color.
 *    ACTIVE to follow in real-time the submission of the jobs and jobsets.
 *           In this mode the icon representing a job or jobset depends of the
 *           state (running, completed, ...) of this jobs/jobset.
 */
enum canvas_mode {
	EDITING,
	ACTIVE
};
typedef enum canvas_mode canvas_mode_t;


/*
 * Highlight types for the icons:
 *               DEFAULT --> Icon as stored in the database
 *           HIGHLIGHTED --> Same icon but highlighted (used when the mouse
 *                           is over the icon)
 *              SELECTED --> Same icon but darkened (used when the icon is
 *                           selected)
 *  SELECTED_HIGHLIGHTED --> Same as SELECTED but slightly less dark (used when
 *                           the icon is selected and the mouse is over it)
 */
enum pixbuf_highlight_type {
	DEFAULT = 0,
	HIGHLIGHTED,
	SELECTED,
	SELECTED_HIGHLIGHTED,
	CUT,

	NUM_HIGHLIGHT_MODES
};
typedef enum pixbuf_highlight_type pixbuf_highlight_t;


/* Structure used to represent the jobset to draw in the canvas */
struct jobset_item {
	char *path;	/* Full name (with the path from the root jobset) */
	gchar *id;

	canvas_mode_t mode;
	gint canvas_width, canvas_height;

	/* Canvas used to draw the jobset */
	GnomeCanvas *canvas;

	/* Location text bar */
	GtkEntry *location;

	/* Grid parameters */
	gboolean grid_show, grid_align;

	/* Selection object */
	job_select_t *selection;

	/* Background/Wallpaper */
	jobset_background_t *background;

	/* Grid */
	grid_t *grid;

	/*
	 * Default job and jobset icons.
	 * Each icon is also stored in its highlight mode (when the mouse
	 * is over the icon), selected mode (when the icon is selected) and
	 * selected and highlight mode (when the icon is selected and the mouse
	 * is over it)
	 */
	GdkPixbuf *job_icon[NUM_JOB_STATUS_STATE][NUM_HIGHLIGHT_MODES];
	GdkPixbuf *jobset_icon[NUM_JOB_STATUS_STATE][NUM_HIGHLIGHT_MODES];

	/*
	 * Text colors
	 */
	GdkColor text_color[NUM_HIGHLIGHT_MODES];
	GdkColor base_color[NUM_HIGHLIGHT_MODES];

	/* Sub job/jobset list */
	GSList *jobs;

	/* Links between the sub jobs/jobsets */
	GSList *links;

	/* Canvas groups (to draw the grid, the links and the jobs/jobsets) */
	GnomeCanvasGroup *group_grid;
	GnomeCanvasGroup *group_links;
	GnomeCanvasGroup *group_jobs;

	/* tooltips for the job and jobset canvas items */
	job_tooltips_ptr tooltips;
};
typedef struct jobset_item jobset_item_t;


extern void destroy_jobset_item OF((jobset_item_t *ptr));
extern jobset_item_t *new_jobset_item OF((	const gchar *id,
						canvas_mode_t mode));
extern void jobset_item_draw_init OF((
			jobset_item_t *ptr,
			GnomeCanvas *canvas,
			GtkEntry *location,
			gboolean grid_show, gboolean grid_align));
extern void jobset_item_draw_link OF((jobset_item_t *ptr, link_t *link));
extern void jobset_item_adjust_view OF((jobset_item_t *ptr,
					gdouble x, gdouble y,
					gint width, gint height));
extern void jobset_item_select_job OF((	jobset_item_t *ptr,
					child_job_t *job_ptr));
extern void jobset_item_select_job_by_id OF((	jobset_item_t *ptr,
						const gchar *id));
extern void *jobset_save_selection OF((jobset_item_t *ptr));
extern void jobset_restore_selection OF((jobset_item_t *ptr, void *data));
extern void jobset_dump_saved_selection OF((void *data));
extern void jobset_item_add_link OF((jobset_item_t *ptr, link_t *link));

#define jobset_item_get_canvas_width(x) ((x)->canvas_width)
#define jobset_item_get_canvas_height(x) ((x)->canvas_height)
#define jobset_item_set_show_grid(x,y) ((x)->grid_show = (y))
#define jobset_item_set_snap_grid(x,y) ((x)->grid_align = (y))


#ifdef __cplusplus
}
#endif  /* C++ */

#endif /* jobset_item.h */

/*-----------------============== End Of File ==============-----------------*/
