/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* main_cb.c -- GUI functions for the main window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <math.h>

#include <grid.h>
#include <canvas_utils.h>
#include <schedwi_g_utils.h>
#include <copy_paste.h>
#include <child_job.h>
#include <schedwi_style.h>
#include <sql_jobsets.h>
#include <sql_hierarchy.h>
#include <message_windows.h>
#include <schedwi_main.h>
#include <cursor.h>
#include <main_cb.h>

GnomeAppBar *application_statusbar = NULL;


/*
 * Job/jobset description from the database
 */
struct job_jobset {
	guint64 id, parent_id;
	gchar *name;
};
typedef struct job_jobset job_jobset_t; 


/*
 * Free a job_jobset_t structure
 */
static void
destroy_job_jobset (job_jobset_t *ptr)
{
	if (ptr != NULL) {
		g_free (ptr->name);
		g_free (ptr);
	}
}


/*
 * Allocate and initialise a new job_jobset structure
 *
 * Return:
 *   The new structure (to be freed by destroy_job_jobset())
 */
static job_jobset_t *
new_job_jobset (const gchar *id, const gchar *parent_id, const gchar *name)
{
	job_jobset_t *ptr;

	ptr = (job_jobset_t *) g_malloc (sizeof (job_jobset_t));
	if (id != NULL && id[0] != '\0') {
		ptr->id = g_ascii_strtoull (id, NULL, 0);
	}
	else {
		ptr->id = 0;
	}
	if (parent_id != NULL && parent_id[0] != '\0') {
		ptr->parent_id = g_ascii_strtoull (parent_id, NULL, 0);
	}
	else {
		ptr->parent_id = 0;
	}
	ptr->name = g_strdup (name);
	return ptr;
}


/*
 * Free a job_jobset structure.  This function is used by g_slist_foreach()
 * to free all the elements of a list
 */
static void
remove_job_jobset_from_list (gpointer data, gpointer user_data)
{
	destroy_job_jobset ((job_jobset_t *)data);
}


/*
 * Callback for the `Delete' button/menu item
 */
void
main_cb_delete (GtkWidget *w)
{
	GtkWidget *canvas;
	jobset_item_t *jobset;

	canvas = lookup_widget (GTK_WIDGET (w), "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");

	if (jobset->selection == NULL || jobset->selection->set == FALSE) {
		return;
	}

	if (jobset->selection->item_type == JOB_SELECT_TYPE_LINK) {
		link_delete (jobset->selection->item.link);
	}
	else {
		child_job_delete (jobset->selection->item.job);
	}
}


/*
 * Callback for the `cut' button/menu item
 */
void
main_cb_cut (GtkWidget *w)
{
	GtkWidget *canvas;
	jobset_item_t *jobset;

	canvas = lookup_widget (GTK_WIDGET (w), "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");

	if (	   jobset->selection == NULL
		|| jobset->selection->set == FALSE
		|| jobset->selection->item_type != JOB_SELECT_TYPE_JOB)
	{
		return;
	}

	child_job_cut (jobset->selection->item.job);
}


/*
 * Callback for the `copy' button/menu item
 */
void
main_cb_copy (GtkWidget *w)
{
	GtkWidget *canvas;
	jobset_item_t *jobset;

	canvas = lookup_widget (GTK_WIDGET (w), "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");

	if (	   jobset->selection == NULL
		|| jobset->selection->set == FALSE
		|| jobset->selection->item_type != JOB_SELECT_TYPE_JOB)
	{
		return;
	}

	child_job_copy (jobset->selection->item.job);
}


/*
 * Callback for the `paste' button/menu item
 */
void
main_cb_paste (GtkWidget *w)
{
	GtkWidget *canvas;
	jobset_item_t *jobset;
	gdouble x, y;

	canvas = lookup_widget (GTK_WIDGET (w), "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");

	canvas_utils_get_position (GNOME_CANVAS (canvas), &x, &y);
	x += g_random_double_range (10.0, 100.0);
	y += g_random_double_range (10.0, 100.0);

	copy_cut_paste (jobset->id, x, y);
}


/*
 * TREE VIEW WIDGET MANAGEMENT
 */

/*
 * Callback when the selection changes
 * The buttons are made sensitive if an item is selected
 */
static void
tree_selection_changed_cb (GtkTreeSelection *selection, gpointer data)
{
	GtkWidget *widget = data;
	GtkTreeIter iter, parent;
	GtkTreeModel *model;
	guint64 id;
	gint type;
	gchar *id_str;
	GnomeCanvas *canvas;
	jobset_item_t *jobset;
	GtkWidget *location;

	if (gtk_tree_selection_get_selected (selection, &model, &iter) == TRUE)
	{
		/* Retrieve the details of the selected row */
		gtk_tree_model_get (model, &iter, 1, &id, 2, &type, -1);

		/* Retrieve the canvas and the displayed jobset */
		canvas = (GnomeCanvas *)lookup_widget (widget, "canvas_main");
		jobset = g_object_get_data (G_OBJECT (canvas), "jobset");
		location = lookup_widget (widget, "entry_main_location");

		/* Empty row selected */
		if (type != 0) {
			/* Retrieve the parent jobset */
			gtk_tree_model_iter_parent (model, &parent, &iter);
			gtk_tree_model_get (model, &parent, 1, &id, -1);
		}

		id_str = schedwi_ulltostr (id);
		if (jobset != NULL) {
			destroy_jobset_item (jobset);
		}
		zoom_normal (widget);
		jobset = new_jobset_item (id_str, EDITING);
		copy_paste_clear_if_this_id_is_cut (id_str);
		g_free (id_str);
		jobset_item_draw_init (	jobset,
					canvas,
					GTK_ENTRY (location),
					show_grid_get_active (),
					snap_to_grid_get_active ());
	}
}


/*
 * Structure to exchange data between gtk_tree_model_foreach() and its
 * callback function
 */
struct main_find_and_select_user_data {
	guint64 id;
	GtkTreeSelection *selection;
	GtkTreeView *view;
};


/*
 * gtk_tree_model_foreach() callback function
 */
static gboolean
find_and_select (	GtkTreeModel *model,
			GtkTreePath *path,
			GtkTreeIter *iter,
			gpointer data)
{
	struct main_find_and_select_user_data *ptr =
				(struct main_find_and_select_user_data *) data;
	guint64 id_job;

	gtk_tree_model_get (model, iter, 1, &id_job, -1);
	if (id_job == ptr->id) {
		/* Expand the view and select the jobset */
		gtk_tree_view_expand_to_path (ptr->view, path);
		gtk_tree_selection_select_path (ptr->selection, path);
		return TRUE;
	}
	return FALSE;
}


/*
 * Select the provided job/jobset
 */
void
main_select_job (	GtkWidget *w,
			const gchar *id,
			const gchar *parent_id,
			const gchar *job_type)
{
	struct main_find_and_select_user_data str;
	GtkWidget *canvas;
	jobset_item_t *jobset;

	str.view = (GtkTreeView *)lookup_widget (w, "treeview_main");
	str.selection = gtk_tree_view_get_selection (str.view);
	if (job_type != NULL && job_type[0] != '0') {
		/*
		 * If the path does not point to a jobset,
		 * the parent jobset is selected instead
		 */
		str.id = g_ascii_strtoull (parent_id, NULL, 0);
	}
	else {
		str.id = g_ascii_strtoull (id, NULL, 0);
	}

	/* Select the jobset */
	gtk_tree_model_foreach (gtk_tree_view_get_model (str.view),
				find_and_select,
				&str);

	/* Select the job */
	if (job_type != NULL && job_type[0] != '0') {
		canvas = lookup_widget (w, "canvas_main");
		jobset = g_object_get_data (G_OBJECT (canvas), "jobset");
		if (jobset != NULL) {
			jobset_item_select_job_by_id (jobset, id);
		}
	}
}


/*
 * Select the jobset entered in the location entry
 */
void
main_location_changed (const gchar *path, GtkWidget *w)
{
	char *id, *parent, *type;

	/* Retrieve the ID of the job (and its parent and type) */
	if (get_job_id (path, &id, &parent, &type,  
			(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
			_("Database error")) != 0)
	{
		return;
	}

	main_select_job (w, id, parent, type);
	free (id);
	free (parent);
	free (type);
}


/*
 * Return a step value for the mouse wheel.  This code is strongly inspired by
 * gtk_scrolled_window_scroll_event() in gtkscrolledwindow.c from GTK+
 */
static gdouble
get_wheel_delta (gdouble page_size, GdkScrollDirection direction)
{
	gdouble delta;

	delta = pow (page_size, 2.0 / 3.0);
	if (direction == GDK_SCROLL_UP || direction == GDK_SCROLL_LEFT) {
		delta = - delta;
	}
	return delta;
}


/*
 * Mouse wheel scroll.  This code is strongly inspired by
 * gtk_scrolled_window_scroll_event() in gtkscrolledwindow.c from GTK+
 */
gboolean
canvas_main_scroll_event (GtkWidget *w, GdkEventScroll *event)
{
	GtkAdjustment *adj;
	gdouble delta, new_value, page_size, lower, upper, value;

	if (	   event->direction == GDK_SCROLL_UP
		|| event->direction == GDK_SCROLL_DOWN)
	{
		adj = gtk_layout_get_vadjustment (GTK_LAYOUT (w));
	}
	else {
		adj = gtk_layout_get_hadjustment (GTK_LAYOUT (w));
	}

	value = gtk_adjustment_get_value (adj);
	g_object_get (	(gpointer)adj,
			"page-size", &page_size,
			"lower", &lower,
			"upper", &upper,
			NULL);

	delta = get_wheel_delta (page_size, event->direction);
	new_value = CLAMP (value + delta, lower, upper - page_size);
	gtk_adjustment_set_value (adj, new_value);
	return TRUE;
}


/*
 * Zoom in
 */
void
zoom_in (GtkWidget *w)
{
	GtkWidget *canvas, *label;
	guint zoom_level;
	gchar *s;

	canvas = lookup_widget (w, "canvas_main");
	label = lookup_widget (w, "label_main_zoom");

	zoom_level = canvas_utils_zoom_in (GNOME_CANVAS (canvas));
	s = g_strdup_printf ("%3u%%", zoom_level);
	gtk_label_set_text (GTK_LABEL (label), s);
	g_free (s);
}


/*
 * Zoom out
 */
void
zoom_out (GtkWidget *w)
{
	GtkWidget *canvas, *label;
	guint zoom_level;
	gchar *s;

	canvas = lookup_widget (w, "canvas_main");
	label = lookup_widget (w, "label_main_zoom");

	zoom_level = canvas_utils_zoom_out (GNOME_CANVAS (canvas));
	s = g_strdup_printf ("%3u%%", zoom_level);
	gtk_label_set_text (GTK_LABEL (label), s);
	g_free (s);
}


/*
 * Set the zoom level to normal
 */
void
zoom_normal (GtkWidget *w)
{
	GtkWidget *canvas, *label;
	guint zoom_level;
	gchar *s;

	canvas = lookup_widget (w, "canvas_main");
	label = lookup_widget (w, "label_main_zoom");

	zoom_level = canvas_utils_zoom_normal (GNOME_CANVAS (canvas));
	s = g_strdup_printf ("%3u%%", zoom_level);
	gtk_label_set_text (GTK_LABEL (label), s);
	g_free (s);
}


/*
 * Save the current zoom level
 */
static gint
zoom_save (GtkWidget *w)
{
	GtkWidget *canvas;

	canvas = lookup_widget (w, "canvas_main");
	return canvas_utils_zoom_save (GNOME_CANVAS (canvas));
}


/*
 * Restore the zoom level
 */
static void
zoom_restore (GtkWidget *w, guint saved_zoom)
{
	GtkWidget *canvas, *label;
	guint zoom_level;
	gchar *s;

	canvas = lookup_widget (w, "canvas_main");
	label = lookup_widget (w, "label_main_zoom");

	zoom_level = canvas_utils_zoom_restore (GNOME_CANVAS (canvas),
						saved_zoom);
	s = g_strdup_printf ("%3u%%", zoom_level);
	gtk_label_set_text (GTK_LABEL (label), s);
	g_free (s);
}


/*
 * Switch to show/hide the grid
 */
void
show_hide_grid (GtkWidget *w, gboolean activate)
{
	GtkWidget *canvas;
	jobset_item_t *jobset;

	canvas = lookup_widget (w, "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");
	if (jobset != NULL) {
		jobset_item_set_show_grid (jobset, activate);
		if (activate == TRUE) {
			show_grid (jobset->grid);
		}
		else {
			hide_grid (jobset->grid);
		}
	}
}


/*
 * Switch to snap or not the objects to the grid
 */
void
snap_grid (GtkWidget *w, gboolean snap)
{
	GtkWidget *canvas;
	jobset_item_t *jobset;

	canvas = lookup_widget (w, "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");
	if (jobset != NULL) {
		jobset_item_set_snap_grid (jobset, snap);
		snap_to_grid (jobset->grid, snap);
	}
}



/*
 * INITIALISATION FUNCTIONS
 */


/*
 * Recursively fill the model
 */
static GtkTreeRowReference *
fill_model_recursive (	GtkTreeStore *store, GSList *lst,
			guint64 parent_id, GtkTreeIter *parent_iter,
			guint64 selected_id)
{
	GtkTreeIter iter;
	GSList *item;
	job_jobset_t *ptr;
	gchar flag_added;
	GtkTreeRowReference *ref = NULL, *ref_tmp;
	GtkTreePath *path;

	flag_added = 0;
	for (item = lst; item != NULL; item = g_slist_next (item)) {
		ptr = (job_jobset_t *)(item->data);

		/*
		 * Only the items which direct parent is PARENT_ID are
		 * taken into account (and the item to not display is
		 * ignored)
		 */
		if (ptr->parent_id == parent_id) {
			gtk_tree_store_append (store, &iter, parent_iter);
			gtk_tree_store_set (	store, &iter,
						0, ptr->name,
						1, ptr->id,
						2, 0,
						3, 0,
						-1);

			/*
			 * If the item is the one that must be selected,
			 * create a new GtkTreeRowReference
			 */
			if (ptr->id == selected_id) {
				path = gtk_tree_model_get_path (
							GTK_TREE_MODEL (store),
							&iter);
				ref = gtk_tree_row_reference_new (
							GTK_TREE_MODEL (store),
							path);
				gtk_tree_path_free (path);
			}

			/*
			 * Recursively call this function
			 */ 
			ref_tmp = fill_model_recursive (store, lst,
							ptr->id, &iter,
							selected_id);
			if (ref_tmp != NULL) {
				gtk_tree_row_reference_free (ref);
				ref = ref_tmp;
			}
			flag_added = 1;
		}
	}

	/* If the folder is empty, add an `(Empty)' tag */
	if (flag_added == 0) {
		gtk_tree_store_append (store, &iter, parent_iter);
		gtk_tree_store_set (	store, &iter,
					0, NULL,
					1, (guint64)0,
					2, 2,
					3, 0,
					-1);
	}
	return ref;
}


/*
 * Fill the model
 * selected_id is the item ID for which a GtkTreeRowReference must be created.
 * This is used when the tree is refreshed (destroyed and recreated) to
 * re-select the item that was previously selected before the refresh.
 *
 * Return:
 *    The GtkTreeRowReference corresponding to `selected_id' or
 *    NULL if `selected_id' does not exist in the new tree
 *
 *    The returned GtkTreeRowReference must be freed by the caller using
 *    gtk_tree_row_reference_free()
 */
static GtkTreeRowReference *
fill_model (GtkTreeStore *store, lwc_LL *jobsets, guint64 selected_id)
{
	char **row;
	GSList *lst = NULL;
	GtkTreeRowReference *ref;

	/* Build a GSList with the result of the SQL result */
	lwc_rewindLL (jobsets);
	while ((row = (char **) lwc_nextLL (jobsets)) != NULL) {
		lst = g_slist_prepend (lst, new_job_jobset (
							row[0],	/* ID */ 
							row[1],	/* Parent ID */
							row[2])); /* Name */
	}
	lst = g_slist_reverse (lst);

	/* Fill the model */
	ref = fill_model_recursive (store, lst, 0, NULL, selected_id);

	/* Destroy the list */
	g_slist_foreach (lst, remove_job_jobset_from_list, NULL);
	g_slist_free (lst);	
	return ref;
}


/*
 * Retrieve the jobset list from the database and fill the provided
 * GtkTreeStore model.  
 * If ref is not NULL it is set with the GtkTreeRowReference corresponding
 * to SELECTED_ID item ID.  It must be freed by the caller with 
 * gtk_tree_row_reference_free()
 *
 * Return:
 *       0 --> No error (ref is set if not NULL)
 *   other --> SQL error (a message has been displayed to the user)
 */
static gint
retrieve_jobsets_and_fill_model (	GtkTreeStore *store,
					guint64 selected_id,
					GtkTreeRowReference **ref)
{
	lwc_LL *jobsets;
	unsigned int ret;
	GtkTreeRowReference *ref_tmp;

	ret = sql_jobset_list (	&jobsets,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (ret != 0) {
		return -1;
	}

	/* Fill the GtkTreeStore model */
	ref_tmp = fill_model (store, jobsets, selected_id);
	if (ref != NULL) {
		*ref = ref_tmp;
	}
	else {
		gtk_tree_row_reference_free (ref_tmp);
	}

	lwc_delLL (jobsets, (void (*)(const void *)) sql_free_row);
	return 0;
}


/*
 * Clear the canvas and the tree view
 */
void
main_clear (GtkWidget *main_window)
{
	GtkWidget *canvas, *view;
	jobset_item_t *jobset;
	GtkTreeModel *model;

	/* Clear the copy/paste clipboard */
	copy_paste_clear ();

	/* Clear the canvas */
	canvas = lookup_widget (main_window, "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");
	destroy_jobset_item (jobset);
	g_object_set_data (G_OBJECT (canvas), "jobset", NULL);

	/* Clear the tree view */
	view = lookup_widget (main_window, "treeview_main");
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));
	gtk_tree_store_clear (GTK_TREE_STORE (model));

	/* Make the menu and toolbar items insensitive */
	main_deactivate ();
}


/*
 * Retrieve the jobsets from the database and fill the treeview
 *
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error.  An error message has been displayed
 */
gboolean
main_init (GtkWidget *main_window)
{
	GtkWidget *view;
	GtkTreeModel *model;
	GtkTreePath *root_path;
	GtkTreeSelection *select;

	view = lookup_widget (main_window, "treeview_main");
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));
	gtk_tree_store_clear (GTK_TREE_STORE (model));

	/* Retrieve the jobsets and fill the model */
	if (retrieve_jobsets_and_fill_model (	GTK_TREE_STORE (model),
						0, NULL) < 0)
	{
		gtk_tree_store_clear (GTK_TREE_STORE (model));
		return FALSE;
	}

	/* Expand and select the first level */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	root_path = gtk_tree_path_new_first ();
	gtk_tree_view_expand_to_path (GTK_TREE_VIEW (view), root_path);
	gtk_tree_selection_select_path (select, root_path);
	gtk_tree_path_free (root_path);

	/* Make the menu and toolbar items sensitive */
	main_activate ();

	return TRUE;
}


/*
 * Cell renderer function
 */
static void
cell_renderer_func (	GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *tree_model,
			GtkTreeIter *iter,
			gpointer data)
{
	gchar *name;
	gint type;

	gtk_tree_model_get (tree_model, iter, 0, &name, 2, &type, -1);
	if (type == 0 || type == 1) {
		/* Jobset of job */
		g_object_set (cell, "text", name, NULL);
	}
	else {
		/* Empty jobset */
		g_object_set (cell, "markup", _("    <i>(Empty)</i>"), NULL);
	}
	g_free (name);
}


/*
 * Add the column and cell to a GtkTreeView
 */
static void
build_view (GtkTreeView *view, GtkTreeSelection *select)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	/* Jobset name column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Jobset"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (
				column, renderer,
				cell_renderer_func, NULL, NULL);
	gtk_tree_view_append_column (view, column);

	/*
	 * Selection callbacks
	 */
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	/* Selection change */
	g_signal_connect (	G_OBJECT (select), "changed",
				G_CALLBACK (tree_selection_changed_cb), view);
}


/*
 * Build the new window
 */
static gint
dialog_main_build (GtkWidget *main_window)
{
	GtkWidget *h_scrollbar, *v_scrollbar, *location;
	GtkAdjustment *adj;
	GnomeCanvas *canvas;
	GtkWidget *view;
	GtkTreeStore *store;
	GtkTreeSelection *select;


	/*
	 * Initialize the jobset canvas
	 */
	location = lookup_widget (main_window, "entry_main_location");
	canvas = (GnomeCanvas *) lookup_widget (main_window, "canvas_main");
	h_scrollbar = lookup_widget (main_window, "hscrollbar_canvas_main");
	v_scrollbar = lookup_widget (main_window, "vscrollbar_canvas_main");

	adj = gtk_range_get_adjustment (GTK_RANGE (h_scrollbar));
	gtk_layout_set_hadjustment (GTK_LAYOUT (canvas), adj);
	adj = gtk_range_get_adjustment (GTK_RANGE (v_scrollbar));
	gtk_layout_set_vadjustment (GTK_LAYOUT (canvas), adj);

	schedwi_style_init (location);


	/*
	 * Initialize the jobset list
	 */
	view = lookup_widget (main_window, "treeview_main");

	/* Create the GtkTreeStore model */
	store = gtk_tree_store_new (4,
			G_TYPE_STRING,	/* Jobset name */
			G_TYPE_UINT64,	/* Jobset ID (not displayed) */
			G_TYPE_INT,	/* Type: Jobset(0), empty(2) */
			G_TYPE_INT);	/* Jobset status */

	/* Associate the model with the view */
	gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (store));
	g_object_unref (store);
	gtk_tree_view_set_search_column (GTK_TREE_VIEW (view), 0);

	/* Add the columns, cells and selection callbacks */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	build_view (GTK_TREE_VIEW (view), select);

	/* Make the menu and toolbar items insensitive */
	store_application_main (main_window);
	main_deactivate ();

	/* Store the application bar widget */
	application_statusbar = (GnomeAppBar *)lookup_widget (	main_window,
								"appbar_main");

	return 0;
}


/*
 * Create the main window
 *
 * Return:
 *   The new GtkWidget (`show'ed by this function) or
 *   NULL in case of error (an error message has been displayed for the user)
 */
GtkWidget *
new_dialog_main ()
{
	GtkWidget *widget;

	widget = create_application_main ();
	if (dialog_main_build (widget) != 0) {
		gtk_widget_destroy (widget);
		return NULL;
	}
	else {
		gtk_widget_show (widget);
		return widget;
	}
}



/*
 * REFRESH
 */


/*
 * Structure used to pass data to the for `foreach' functions
 *  gtk_tree_view_map_expanded_rows() and gtk_tree_model_foreach()
 */
struct expanded_items {
	GArray *array;
	GtkTreeModel *model;
	GtkTreeView *view;
};


/*
 * Compare two guint64.  This function is used by g_array_sort to sort the
 * array of expanded items in the view
 */
static gint
compare_guint64 (gconstpointer a, gconstpointer b)
{
	guint64 i = *((guint64 *)a), j = *((guint64 *)b);

	return (gint)(i - j);
}


/*
 * Function called by gtk_tree_view_map_expanded_rows() to build the
 * array of expanded item ID
 */
static void
build_array_expanded (GtkTreeView *tree_view, GtkTreePath *path, gpointer data)
{
	struct expanded_items *ptr = data;
	GtkTreeIter iter;
	guint64 id;

	gtk_tree_model_get_iter (ptr->model, &iter, path);
	gtk_tree_model_get (ptr->model, &iter, 1, &id, -1);
	ptr->array = g_array_append_val (ptr->array, id);
}

/*
 * Function called by gtk_tree_model_foreach() to expand the items which IDs
 * are in the array of items previously expanded before the refresh
 */
static gboolean
expand_item_in_array (  GtkTreeModel *model,
			GtkTreePath *path,
			GtkTreeIter *iter,
			gpointer data)
{
	struct expanded_items *ptr = data;
	guint64 id, tmp_id;
	guint i;

	gtk_tree_model_get (model, iter, 1, &id, -1);

	i = 0;
	do {
		tmp_id = g_array_index (ptr->array, guint64, i);
		i++;
	} while (tmp_id != 0 && tmp_id < id);

	if (tmp_id == id) {
		gtk_tree_view_expand_to_path (ptr->view, path);
	}
	return FALSE;
}


/*
 * Refresh the jobset list
 */
void
jobset_list_refresh (GtkWidget *w)
{
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkWidget *canvas;
	jobset_item_t *jobset;
	guint64 selected_id;
	GtkTreeRowReference *ref = NULL;
	GtkTreePath *path;
	struct expanded_items expanded;
	void *saved_selection;
	gint saved_zoom;
	gdouble saved_scroll_v, saved_scroll_h;

	cursor_busy (w);

	view = (GtkTreeView *)lookup_widget (w, "treeview_main");

	/* Retrieve the selected job in the canvas */
	canvas = lookup_widget (w, "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");
	if (jobset != NULL) {
		saved_selection = jobset_save_selection (jobset);
	}
	else {
		saved_selection = NULL;
	}

	/* Retrieve the current selected jobset in the list */
	selection = gtk_tree_view_get_selection (view);
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == TRUE)
	{
		gtk_tree_model_get (model, &iter, 1, &selected_id, -1);
	}
	else {
		selected_id = 0;
	}

	/* Retrieve the expanded rows */
	expanded.array = g_array_new (TRUE, FALSE, sizeof (guint64));
	expanded.model = model;
	expanded.view = view;
	gtk_tree_view_map_expanded_rows (	view, build_array_expanded,
						&expanded);
	g_array_sort (expanded.array, compare_guint64);

	/* Save the zoom level */
	saved_zoom = zoom_save (w);

	/* Save the position in the scroll region */
	canvas_utils_get_position (	GNOME_CANVAS (canvas),
					&saved_scroll_v, &saved_scroll_h);

	/* Clear the model */
	gtk_tree_store_clear (GTK_TREE_STORE (model));

	/* Retrieve the jobset list and fill the model */
	retrieve_jobsets_and_fill_model (     GTK_TREE_STORE (model),
						selected_id, &ref);

	/* Expand the previously expanded items */
	gtk_tree_model_foreach (model, expand_item_in_array, &expanded);
	g_array_free (expanded.array, TRUE);

	if (gtk_tree_row_reference_valid (ref) == TRUE) {
		/* Select the item */
		path = gtk_tree_row_reference_get_path (ref);
		gtk_tree_selection_select_path (selection, path);
	}
	else {
		/* Expand the first level */
		path = gtk_tree_path_new_first ();
		gtk_tree_view_expand_to_path (view, path);
		gtk_tree_path_free (path);
	}
	gtk_tree_row_reference_free (ref);

	/* Restore the selection in the canvas */
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");
	if (jobset != NULL) {
		jobset_restore_selection (jobset, saved_selection);
	}
	else {
		jobset_dump_saved_selection (saved_selection);
	}

	/* Restore the zoom level */
	zoom_restore (w, saved_zoom);

	/* Restore the canvas position */
	canvas_utils_set_position (	GNOME_CANVAS (canvas),
					saved_scroll_v, saved_scroll_h);
	cursor_normal (w);
}

/*------------------------======= End Of File =======------------------------*/
