/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* navigator.c -- GUI functions for the navigation window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <cursor.h>
#include <canvas_utils.h>
#include <navigator.h>


#define NAVIGATOR_WIDTH 100
#define NAVIGATOR_HEIGHT 100
#define NAVIGATOR_WINDOW_BORDER_THICK 2
#define NAVIGATOR_SELECTION_THICK 2
#define NAVIGATOR_SELECTION_COLOR "#000000"


/* Navigation window */
static navigator_t *popup_navigator = NULL;


/*
 * Draw the selection rectangle
 */
static void
navigator_draw_selection (navigator_t *ptr)
{
	gint thick;

	if (ptr == NULL) {
		return;
	}

	if (ptr->selection != NULL) {
		gtk_object_destroy (GTK_OBJECT (popup_navigator->selection));	
	}
	thick = NAVIGATOR_SELECTION_THICK / 2;
	ptr->selection = gnome_canvas_item_new (
				gnome_canvas_root (ptr->canvas),
				gnome_canvas_rect_get_type (),
				"x1",	  ptr->selection_x + thick,
				"y1",	  ptr->selection_y + thick,
				"x2",	  ptr->selection_x
					+ ptr->selection_width
					- thick,
				"y2",	  ptr->selection_y
					+ ptr->selection_height
					- thick,
				"width-pixels", NAVIGATOR_SELECTION_THICK,
				"outline-color", NAVIGATOR_SELECTION_COLOR,
				NULL);
}


/*
 * Compute the reduction ratio used to represent the real canvas into
 * the navigation window
 */
static void
navigator_set_ratio (navigator_t *ptr, gint canvas_width, gint canvas_height)
{
	gdouble ratio_w, ratio_h;


	ratio_w = (gdouble)NAVIGATOR_WIDTH / canvas_width;
	ratio_h = (gdouble)NAVIGATOR_HEIGHT / canvas_height;
	ptr->ratio = (ratio_w < ratio_h) ? ratio_w: ratio_h;
	ptr->width = ptr->ratio * canvas_width;
	ptr->height = ptr->ratio * canvas_height;

	/* Adjust the canvas size */
	gtk_widget_set_size_request (	GTK_WIDGET (ptr->canvas),
					ptr->width, ptr->height);
	gnome_canvas_set_scroll_region (ptr->canvas,
					(gdouble)0,
					(gdouble)0,
					(gdouble)ptr->width,
					(gdouble)ptr->height);
	gnome_canvas_set_center_scroll_region (ptr->canvas, FALSE);
}


/*
 * Hide the navigation window
 */
static void
navigator_hide (navigator_t *ptr)
{
	if (ptr != NULL) {

		/* Hide the window */
		gtk_widget_hide (ptr->w);

		/* Clear the canvas */
		canvas_utils_clean (ptr->canvas);
		ptr->selection = NULL;

		gdk_pointer_ungrab (GDK_CURRENT_TIME);
		gtk_grab_remove (ptr->w);
	}
}


/*
 * Callback function used to hide the navigation window
 */
static gboolean
navigator_hide_cb (	GtkWidget *widget,
			GdkEventButton *event,
			gpointer user_data)
{
	switch (event->button) {
		case 1:
			navigator_hide ((navigator_t *)user_data);
			return TRUE;
		default:
			return FALSE;
	}
}


/*
 * Move the selection rectangle in the navigation window and adjust
 * the view in the real canvas
 */
static void
navigator_selection_motion (navigator_t *ptr, gdouble mouse_x, gdouble mouse_y)
{
	gdouble value_v, value_h;
	GtkAdjustment *adj;
	gdouble adj_v, adj_h;

	if (ptr == NULL) {
		return;
	}

	/*
	 * Compute the coordinates of the selection (as the mouse pointer
	 * is centered in the selection rectangle)
	 */
	value_h = mouse_x - ptr->selection_width * 0.5;
	value_v = mouse_y - ptr->selection_height * 0.5;
	if (value_h < 0) {
		value_h = 0;
	}
	if (value_h + ptr->selection_width > ptr->width) {
		value_h = ptr->width - ptr->selection_width;
	}
	if (value_v < 0) {
		value_v = 0;
	}
	if (value_v + ptr->selection_height > ptr->height) {
		value_v = ptr->height - ptr->selection_height;
	}

	if (value_h == ptr->selection_x && value_v == ptr->selection_y) {
		return;
	}


	/*
	 * Scroll the real canvas
	 */
	gnome_canvas_w2c_d (	ptr->real_canvas,
				value_h / ptr->ratio,
				value_v / ptr->ratio,
				&adj_h,
				&adj_v);
	adj = gtk_layout_get_hadjustment (GTK_LAYOUT (ptr->real_canvas));
	gtk_adjustment_set_value (adj, adj_h);
	gtk_adjustment_value_changed (adj);
	adj = gtk_layout_get_vadjustment (GTK_LAYOUT (ptr->real_canvas));
	gtk_adjustment_set_value (adj, adj_v);
	gtk_adjustment_value_changed (adj);

	/*
	 * Redraw the selection box in the navigation canvas
	 */
	ptr->selection_x = value_h;
	ptr->selection_y = value_v;
	navigator_draw_selection (ptr);
}


/*
 * Callback function used to move the selection
 */
static gboolean
navigator_motion_cb (	GtkWidget *widget,
			GdkEventMotion *event,
			gpointer user_data)
{
	navigator_selection_motion (	(navigator_t *)user_data,
					event->x, event->y);
	return TRUE;
}


/*
 * Show the navigation window
 */
static void
navigator_show (GnomeCanvas *canvas,
		gint canvas_width, gint canvas_height,
		GtkWidget *widget, gint click_x, gint click_y)
{
	GdkScreen *screen;
	gint x_org, y_org;
	gint x, y;
	gint real_width, real_height;
	gint screen_width, screen_height;
	gdouble wx, wy;
	GtkAdjustment *adj_v, *adj_h;
	gdouble page_size_v, page_size_h;

	/* Create the popup window if required */
	if (popup_navigator == NULL) {
		popup_navigator = g_new0 (navigator_t, 1);
		popup_navigator->w = create_window_navigator ();
		popup_navigator->canvas = (GnomeCanvas *)lookup_widget (
							popup_navigator->w,
							"canvas_navigator");
		g_signal_connect (	(gpointer)popup_navigator->w,
					"button-release-event",
					G_CALLBACK (navigator_hide_cb),
					popup_navigator);
		g_signal_connect (	(gpointer)popup_navigator->canvas,
					"motion-notify-event",
					G_CALLBACK (navigator_motion_cb),
					popup_navigator);
	}

	/* Clear the canvas */
	canvas_utils_clean (popup_navigator->canvas);
	popup_navigator->selection = NULL;

	/* Compute the ratio between the real canvas and the navigator one */
	navigator_set_ratio (popup_navigator, canvas_width, canvas_height);

	/* Copy the real canvas to the navigator canvas */
	canvas_utils_copy (	popup_navigator->canvas, canvas,
				popup_navigator->ratio);

	/*
	 * Compute the coordinates of the selection mask and its size
	 */
	adj_h = gtk_layout_get_hadjustment (GTK_LAYOUT (canvas));
	adj_v = gtk_layout_get_vadjustment (GTK_LAYOUT (canvas));
	gnome_canvas_c2w (canvas,
			(int)gtk_adjustment_get_value (adj_h),
			(int)gtk_adjustment_get_value (adj_v),
			&wx,
			&wy);
	popup_navigator->selection_x = wx * popup_navigator->ratio;
	popup_navigator->selection_y = wy * popup_navigator->ratio;

	g_object_get ((gpointer)adj_h, "page-size", &page_size_h, NULL);
	g_object_get ((gpointer)adj_v, "page-size", &page_size_v, NULL);
	gnome_canvas_c2w (canvas,
			(int)page_size_h,
			(int)page_size_v,
			&wx,
			&wy);
	popup_navigator->selection_width = wx * popup_navigator->ratio;
	popup_navigator->selection_height = wy * popup_navigator->ratio;

	/*
	 * Draw the selection mask
	 */
	navigator_draw_selection (popup_navigator);

	/*
	 * Position and show the window
	 */
	screen = gtk_widget_get_screen (widget);
	gtk_window_set_screen (GTK_WINDOW (popup_navigator->w), screen);

	gdk_window_get_origin (widget->window, &x_org, &y_org);



	x =	  x_org + click_x
		- popup_navigator->selection_x
		- popup_navigator->selection_width * 0.5
		- NAVIGATOR_WINDOW_BORDER_THICK;
	y =	  y_org + click_y
		- popup_navigator->selection_y
		- popup_navigator->selection_height * 0.5
		- NAVIGATOR_WINDOW_BORDER_THICK;

	screen_width = gdk_screen_get_width (screen);
	screen_height = gdk_screen_get_height (screen);
	real_width = 	  popup_navigator->width
			+ 2 * NAVIGATOR_WINDOW_BORDER_THICK;
	real_height = 	  popup_navigator->height
			+ 2 * NAVIGATOR_WINDOW_BORDER_THICK;
	if (x + real_width > screen_width) {
		x = screen_width - real_width;
	}
	if (y + real_height > screen_height) {
		y = screen_height - real_height;
	}
	
	gtk_window_move (GTK_WINDOW (popup_navigator->w), x, y);
	gtk_widget_show (popup_navigator->w);

	gtk_grab_add (popup_navigator->w);
	cursor_grab (popup_navigator->w);
	gdk_pointer_grab (	popup_navigator->w->window,
				TRUE,
				GDK_BUTTON_RELEASE_MASK      |
				GDK_POINTER_MOTION_HINT_MASK |
				GDK_BUTTON_MOTION_MASK       |
				GDK_EXTENSION_EVENTS_ALL,
				popup_navigator->w->window,
				NULL,
				GDK_CURRENT_TIME);

	popup_navigator->real_canvas = canvas;
}


/*
 * Button press event callback
 */
gboolean
eventbox_main_navigator_button_press (GtkWidget *widget, GdkEventButton *event)
{
	GnomeCanvas *canvas;
	gint width, height;

	switch (event->button) {
		case 1:
			canvas = (GnomeCanvas *)lookup_widget (
							widget,
							"canvas_main");

			/* Retrieve the canvas size */
			canvas_utils_get_size (canvas, &width, &height);

			navigator_show (canvas,
					width, height,
					widget, event->x, event->y);
			return TRUE;

		default:
			return FALSE;
	}
}

/*------------------------======= End Of File =======------------------------*/
