/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sql_host_env.c -- host_environment table management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_host_env.h>

#define SQL_HOST_ENV_LIST "SELECT position,id,name FROM host_environment,environments WHERE host_id=\"%s\" AND env_id=id ORDER BY position" 
#define SQL_HOST_ENV_DELETE "DELETE FROM host_environment WHERE host_id=\"%s\""
#define SQL_HOST_ENV_ADD "REPLACE INTO host_environment (host_id,env_id,position) VALUES "
#define SQL_HOST_ENV_ADD_VALUES "(\"%s\",%ld,%d)"


/*
 * Retrieve from the database the environments associated with a host
 *
 * Return:
 *   0 --> No error.  rows is set and must be freed by the caller by
 *             lwc_delLL (rows, (void (*)(const void *))sql_free_row)
 *         Each row is composed as follow:
 *               row[0] --> Position
 *               row[1] --> Environment ID
 *               row[2] --> Environment name
 *  -1 --> Other error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_hosts_env_list (const char *host_id,
		lwc_LL **rows,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL && host_id != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows, NULL,
				SQL_HOST_ENV_LIST,
				SQL_STRING, host_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -1;
	}

	return 0;
}


/*
 * Remove all the associations between the environments and the
 * provided host from the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_hosts_env_delete (	const char *host_id,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (host_id != NULL);
#endif


	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_HOST_ENV_DELETE,
				SQL_STRING, host_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Associate the provided host with the environments provided in the list
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called 
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 *  -3 --> Internal error ((if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_hosts_env_add (	const char *host_id,
			lwc_LL *list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *req;
	unsigned int req_len, idx;
	char *err_msg = NULL;
	char *esc_hostid;
	int pos, ret;
	char *env;
	unsigned int sql_ret;

#if HAVE_ASSERT_H
	assert (host_id != NULL);
#endif

	if (list == NULL || lwc_getNumNode (list) <= 0) {
		return 0;
	}

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Escape the host id string */
	esc_hostid = sql_escape (sql, host_id);
	if (esc_hostid == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	/* Build the request */
	req_len =	  schedwi_strlen (SQL_HOST_ENV_ADD)
			+ (	  schedwi_strlen (SQL_HOST_ENV_ADD_VALUES)
				+ 25 + 10
				+ schedwi_strlen (esc_hostid))
					* lwc_getNumNode (list);
	req = (char *) malloc (req_len);
	if (req == NULL) {
		end_mysql ();
		free (esc_hostid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	strcpy (req, SQL_HOST_ENV_ADD);
	idx = schedwi_strlen (req);

	pos = 1;
	lwc_rewindLL (list);
	while ((env = (char *) lwc_nextLL (list)) != NULL) {

		ret = snprintf (req + idx, req_len - idx,
				SQL_HOST_ENV_ADD_VALUES,
				esc_hostid, strtol (env, NULL, 0), pos);

		if (ret >= req_len - idx || ret < 0) {
			end_mysql ();
			free (esc_hostid);
			free (req);
			if (error_func != NULL) {
				error_func (user_data_error_func,
					_("Internal error: buffer too small"),
					0);
			}
			return -3;
		}

		idx = schedwi_strlen (req);
		req[idx++] = ',';
		pos++;
	}
	req[idx -1] = '\0';
	free (esc_hostid);

	/* Insert the host environments */
	sql_ret = mysql_query (sql, req);
	if (sql_ret != 0) {
		free (req);
		compose_error_message (_("REPLACE"), &err_msg);
		end_mysql ();
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, sql_ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}
	end_mysql ();
	free (req);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
