/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sql_job_tree.c -- Manage a job/jobset tree in the database */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_link.h>
#include <sql_job_tree.h>


#define SQL_JOB_TREE_CHILDREN "SELECT id FROM job_main WHERE parent=\"%s\""
#define SQL_JOB_TREE_DELETE "DELETE FROM %s WHERE job_id IN (%s)"
#define SQL_JOB_TREE_DELETE_MAIN "DELETE FROM job_main WHERE id IN (%s)"
#define SQL_JOB_TREE_DELETE_LINK "DELETE FROM link WHERE job_id_source IN (%s) OR job_id_destination IN (%s)"
#define SQL_JOB_TREE_COPY "INSERT INTO %s SELECT %ld,%s FROM %s WHERE job_id=\"%s\"" 
#define SQL_JOB_TREE_COPY_MAIN_WITH_XY "INSERT INTO job_main (parent,x,y,name,type,enabled,description,cal_id,start_time) SELECT %ld,%ld,%ld,name,type,enabled,description,cal_id,start_time FROM job_main WHERE id=\"%s\""
#define SQL_JOB_TREE_COPY_MAIN "INSERT INTO job_main (parent,x,y,name,type,enabled,description,cal_id,start_time) SELECT %ld,x,y,name,type,enabled,description,cal_id,start_time FROM job_main WHERE id=\"%s\""


/*
 * Database table lists
 */
static char *table_list[][2] = {
	{ "job_start_limit",	"start_limit" },
	{ "job_max_duration",	"max_duration" },
	{ "job_username",	"username" },
	{ "job_file_out",	"file_out" },
	{ "job_file_err",	"file_err" },
	{ "job_loadenv",	"loadenv" },
	{ "job_success_return_code",	"success_ret" },
	{ "job_command",	"command" },
	{ "job_arguments",	"position,argument" },
	{ "job_environment",	"env_id,position" },
	{ "job_host",		"host_id" },
	{ "constraint_file",	"host_id,filename,exist" },
	{ "jobset_background",	"image,background_mode,gradient_color_primary,gradient_color_secondary,canvas_width,canvas_height" },
	{ "job_icon_default",	"icon" },
	{ "job_icon_completed",	"icon" },
	{ "job_icon_failed",	"icon" },
	{ "job_icon_running",	"icon" },
	{ "job_icon_waiting",	"icon" },
	{ "jobset_icon_default",	"icon" },
	{ "jobset_icon_completed",	"icon" },
	{ "jobset_icon_failed",		"icon" },
	{ "jobset_icon_running",	"icon" },
	{ "jobset_icon_waiting",	"icon" },
	{ NULL, NULL }
};

static char *working_table_list[] = {
	"job_status",
	"job_stat",
	"commands",
	NULL
};



/*
 * Find and return a job/jobset in a tree
 *
 * Return:
 *   The found job/jobset OR
 *   NULL if not found
 */
static sql_job_tree_node_ptr
sql_job_tree_find (sql_job_tree_node_ptr ptr, const char *id)
{
	sql_job_tree_node_ptr tmp, tmp2;

	if (ptr == NULL) {
		return NULL;
	}

	if (strcmp (id, ptr->id) == 0) {
		return ptr;
	}

	for (tmp = ptr->children; tmp != NULL; tmp = tmp->next) {
		tmp2 = sql_job_tree_find (tmp, id);
		if (tmp2 != NULL) {
			return tmp2;
		}
	}
	return NULL;
}


/*
 * Add the ID strings to the provided string
 *
 * Return:
 *    The length of the string
 */
static unsigned int
sql_job_tree_add_id (sql_job_tree_node_ptr ptr, char *s)
{
	unsigned int i;
	sql_job_tree_node_ptr tmp;

	if (ptr != NULL) {
		*s = '"';
		strcpy (s + 1, ptr->id);
		i = schedwi_strlen (s);
		s[i++] = '"';
		s[i++] = ',';
		for (tmp = ptr->children; tmp != NULL; tmp = tmp->next) {
			i += sql_job_tree_add_id (tmp, s + i);
		}
		s[i] = '\0';
		return i;
	}
	return 0;
}


/*
 * Return the length of all the ID strings
 */
static unsigned int
sql_job_tree_len (sql_job_tree_node_ptr ptr)
{
	unsigned int len;
	sql_job_tree_node_ptr tmp;

	if (ptr != NULL) {
		len = schedwi_strlen (ptr->id) + 3; /* id + 2x`"' + `,' */
		for (tmp = ptr->children; tmp != NULL; tmp = tmp->next) {
			len += sql_job_tree_len (tmp);
		}
		return len;
	}
	return 0;
}


/*
 * Build the string list of IDs
 * 
 * Return:
 *   The list (to be fred by the caller by free()) OR
 *   NULL in case of memory error
 */
static char *
sql_job_tree_get_id_list (sql_job_tree_node_ptr ptr)
{
	char *s;
	unsigned int len;

	len = sql_job_tree_len (ptr);
	s = (char *) malloc (len + 1);
	if (s == NULL) {
		return NULL;
	}
	len = sql_job_tree_add_id (ptr, s);
	if (len > 0) {
		s[len - 1] = '\0'; /* Remove the last `,' */
	}
	return s;
}


/*
 * Complete the tree by adding links between jobs/jobsets
 *
 * Return:
 *     0 --> No error.  children_list is set and must be freed by the caller by
 *           lwc_delLL (children_id, (void (*)(const void *)) sql_free_row);
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
static unsigned int
sql_job_tree_build_links (sql_job_tree_node_ptr ptr,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	unsigned int ret;
	char *s;
	lwc_LL *rows;
	char **row;
	sql_job_tree_node_ptr src, dst;
	sql_job_tree_link_ptr link;

	/* Build the Ids list */
	s = sql_job_tree_get_id_list (ptr);
	if (s == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return 1;
	}

	/* Retrieve the links from the databases */
	ret = sql_link_get_list (&rows, s, error_func, user_data_error_func);
	free (s);
	if (ret != 0) {
		return ret;
	}

	while ((row = (char **) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Source job ID
		 * row[1] --> Destination job ID
		 * row[2] --> Required status
		 */
		
		/* Find the source and destination jobs/jobsets in the tree */
		src = sql_job_tree_find (ptr, row[0]);
		dst = sql_job_tree_find (ptr, row[1]);

		if (src != NULL && dst != NULL) {
			link = (sql_job_tree_link_ptr)malloc (
						sizeof (sql_job_tree_link));
			if (link == NULL) {
				lwc_delLL (rows,
					(void (*)(const void *)) sql_free_row);
				if (error_func != NULL) {
					error_func (user_data_error_func,
						_("Memory allocation error"),
						0);
				}
				return 1;
			}
			link->linked_node = dst;
			link->required_status = job_status_state_str2status (
							row[2]);
			link->next = src->links;
			src->links = link;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, NULL);
	return 0;
}


/*
 * Get the children list of the provided jobset
 *
 * Return:
 *     0 --> No error.  children_list is set and must be freed by the caller by
 *           lwc_delLL (children_id, (void (*)(const void *)) sql_free_row);
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
static unsigned int
sql_job_tree_get_children (const char *job_id,
			lwc_LL **children_list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL && children_list != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL,
				children_list, NULL,
				SQL_JOB_TREE_CHILDREN,
				SQL_STRING, job_id,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	return 0;
}


/*
 * Recursively free a provided sql_job_tree_node_ptr object
 */
static void
sql_job_tree_destroy_links (sql_job_tree_link_ptr ptr)
{
	sql_job_tree_link_ptr tmp;

	while (ptr != NULL) {
		tmp = ptr->next;
		free (ptr);
		ptr = tmp;
	}
}


/*
 * Free the provided sql_job_tree_node_ptr object
 */
void
sql_job_tree_destroy (sql_job_tree_node_ptr ptr)
{
	sql_job_tree_node_ptr tmp;

	if (ptr != NULL) {
		while (ptr->children != NULL) {
			tmp = ptr->children->next;
			sql_job_tree_destroy (ptr->children);
			ptr->children = tmp;
		}
		sql_job_tree_destroy_links (ptr->links);
		free (ptr->id);
		free (ptr);
	}
}


/*
 * Recursively build the sql_job_tree_node_ptr object
 *
 * Return:
 *   The new (which must be freed by the caller by sql_job_tree_destroy()) or
 *   NULL in case of error (if error_func() is not NULL, it is called with
 *   user_data_error_func as its first parameter and the error message
 *   as the second parameter)
 */
static sql_job_tree_node_ptr
sql_job_tree_new_recur (const char *job_id,
			sql_job_tree_node_ptr parent,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	sql_job_tree_node_ptr ptr, child;
	unsigned int ret;
	lwc_LL *children_id;
	char **child_id;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	/* Build the new object */
	ptr = (sql_job_tree_node_ptr) malloc (sizeof (sql_job_tree_node));
	if (ptr == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return NULL;
	}

	ptr->id = (char *) malloc (schedwi_strlen (job_id) + 1);
	if (ptr->id == NULL) {
		free (ptr);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return NULL;
	}
	strcpy (ptr->id, job_id);
	ptr->parent = parent;
	ptr->children = NULL;
	ptr->next = NULL;
	ptr->links = NULL;
	ptr->new_id = 0;

	/* Retrieve all the children */
	ret = sql_job_tree_get_children (	job_id, &children_id,
						error_func,
						user_data_error_func);
	if (ret != 0) {
		free (ptr->id);
		free (ptr);
		return NULL;
	}

	/* Build all the childen object */
	while ((child_id = (char **) lwc_delStartLL (children_id)) != NULL) {
		child = sql_job_tree_new_recur (child_id[0], ptr, error_func,
						user_data_error_func);
		sql_free_row (child_id);
		if (child == NULL) {
			lwc_delLL (	children_id,
					(void (*)(const void *)) sql_free_row);
			sql_job_tree_destroy (ptr);
			return NULL;
		}
		child->next = ptr->children;
		ptr->children = child;
	}
	lwc_delLL (children_id, (void (*)(const void *)) sql_free_row);

	return ptr;
}


/*
 * Create and return a new sql_job_tree_node_ptr object
 *
 * Return:
 *   The new (which must be freed by the caller by sql_job_tree_destroy()) or
 *   NULL in case of error (if error_func() is not NULL, it is called with
 *   user_data_error_func as its first parameter and the error message
 *   as the second parameter)
 */
sql_job_tree_node_ptr
sql_job_tree_new (	const char *job_id,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	sql_job_tree_node_ptr ptr;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ptr = sql_job_tree_new_recur (	job_id, NULL,
					error_func, user_data_error_func);
	if (ptr == NULL) {
		return NULL;
	}

	if (sql_job_tree_build_links (	ptr, error_func,
					user_data_error_func) != 0)
	{
		sql_job_tree_destroy (ptr);
		return NULL;
	}
	return ptr;
}


/*
 * Run the provided function for each job ID in the provided tree
 */
static void
sql_job_tree_foreach (	sql_job_tree_node_ptr ptr,
			void (*foreach_func)(const char *, void *),
			void *foreach_func_data)
{
	sql_job_tree_node_ptr tmp;

	if (ptr != NULL && foreach_func != NULL) {
		foreach_func (ptr->id, foreach_func_data);
		for (tmp = ptr->children; tmp != NULL; tmp = tmp->next) {
			sql_job_tree_foreach (	tmp, foreach_func,
						foreach_func_data);
		}
	}
}


/*
 * Delete the provided job/jobset from the database (and all its children
 * if it is a jobset)
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_tree_delete (	sql_job_tree_node_ptr ptr,
			void (*foreach_func)(const char *, void *),
			void *foreach_func_data,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL, *s;
	unsigned int ret, i;

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	/* Build the list of all the IDs */
	s = sql_job_tree_get_id_list (ptr);
	if (s == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return 1;
	}

	/* Run the provided user function for each ID */
	sql_job_tree_foreach (ptr, foreach_func, foreach_func_data);

	/*
	 * Delete the jobs/jobsets in all the tables
	 */

	/* In the main table */
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_JOB_TREE_DELETE_MAIN,
				SQL_STRING_NON_ESCAPE, s,
				SQL_END);
	if (ret != 0) {
		free (s);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* In the link table */
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_JOB_TREE_DELETE_LINK,
				SQL_STRING_NON_ESCAPE, s,
				SQL_STRING_NON_ESCAPE, s,
				SQL_END);
	if (ret != 0) {
		free (s);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* In all the other tables */
	for (i = 0; table_list[i][0] != NULL; i++) {
		ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
					SQL_JOB_TREE_DELETE,
					SQL_STRING_NON_ESCAPE,
							table_list[i][0],
					SQL_STRING_NON_ESCAPE, s,
					SQL_END);
		if (ret != 0) {
			free (s);
			if (error_func != NULL) {
				error_func (	user_data_error_func, err_msg,
						ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return ret;
		}
	}

	for (i = 0; working_table_list[i] != NULL; i++) {
		ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
					SQL_JOB_TREE_DELETE,
					SQL_STRING_NON_ESCAPE,
							working_table_list[i],
					SQL_STRING_NON_ESCAPE, s,
					SQL_END);
		if (ret != 0) {
			free (s);
			if (error_func != NULL) {
				error_func (	user_data_error_func, err_msg,
						ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return ret;
		}
	}

	free (s);
	return 0;
}


/*
 * Recursively copy the provided job/jobset
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
static unsigned int
sql_job_tree_copy_recur (sql_job_tree_node_ptr ptr,
			unsigned long int parent_id,
			long int x,
			long int y,
			unsigned long int *new_id,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret, i;
	sql_job_tree_node_ptr tmp;
	unsigned long int id;

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif


	/* Copy the main parameters */
	if (x < 0 || y < 0) {
		ret = sql_non_select (	NULL, NULL, &err_msg, &id, NULL,
					SQL_JOB_TREE_COPY_MAIN,
					SQL_INT, (long int)parent_id,
					SQL_STRING, ptr->id,
					SQL_END);
	}
	else {
		ret = sql_non_select (	NULL, NULL, &err_msg, &id, NULL,
					SQL_JOB_TREE_COPY_MAIN_WITH_XY,
					SQL_INT, (long int)parent_id,
					SQL_INT, x,
					SQL_INT, y,
					SQL_STRING, ptr->id,
					SQL_END);
	}
	
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Copy all the other tables */
	for (i = 0; table_list[i][0] != NULL; i++) {
		ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
					SQL_JOB_TREE_COPY,
					SQL_STRING_NON_ESCAPE,
							table_list[i][0],
					SQL_INT, (long int)id,
					SQL_STRING_NON_ESCAPE,
							table_list[i][1],
					SQL_STRING_NON_ESCAPE,
							table_list[i][0],
					SQL_STRING, ptr->id,
					SQL_END);
		if (ret != 0) {
			if (error_func != NULL) {
				error_func (	user_data_error_func, err_msg,
						ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return ret;
		}
	}

	/* Copy the children */
	for (tmp = ptr->children; tmp != NULL; tmp = tmp->next) {
		ret = sql_job_tree_copy_recur (	tmp, id, -1, -1, NULL,
						error_func,
						user_data_error_func);
		if (ret != 0) {
			return ret;
		}
	}

	ptr->new_id = id;

	if (new_id != NULL) {
		*new_id = id;
	}

	return 0;
}


/*
 * Recursively copy the links
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
static unsigned int
sql_job_tree_copy_links (sql_job_tree_node_ptr ptr,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	sql_job_tree_link_ptr link;
	sql_job_tree_node_ptr tmp;
	unsigned int ret;
	char src[25], dst[25];

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	/* Copy all the links */
	snprintf (src, 25, "%ld", ptr->new_id);
	for (link = ptr->links; link != NULL; link = link->next) {
		snprintf (dst, 25, "%ld", link->linked_node->new_id);
		ret = sql_link_add (	src, dst, 
					(long int)job_status_state_status2int (
						link->required_status),
					error_func, user_data_error_func);
		if (ret != 0) {
			return ret;
		}
	}

	/* Copy the links for children */
	for (tmp = ptr->children; tmp != NULL; tmp = tmp->next) {
		ret =  sql_job_tree_copy_links(	tmp,
						error_func,
						user_data_error_func);
		if (ret != 0) {
			return ret;
		}
	}

	return 0;
}


/*
 * Copy the provided job/jobset (and all its children if it is a jobset)
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_tree_copy (	sql_job_tree_node_ptr ptr,
			const char *parent_id,
			long int x,
			long int y,
			unsigned long int *new_id,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (ptr != NULL && parent_id != NULL);
#endif

	ret =  sql_job_tree_copy_recur (ptr,
					strtoul (parent_id, NULL, 0),
					x, y,
					new_id,
					error_func, user_data_error_func);
	if (ret != 0) {
		return ret;
	}

	return sql_job_tree_copy_links (ptr, error_func, user_data_error_func);
}

/*-----------------============== End Of File ==============-----------------*/
