/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sql_link.c -- functions to retrieve link details from the database */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_link.h>

#define SQL_LINK_LIST "SELECT job_id_source,job_id_destination,required_status FROM link WHERE job_id_source IN (%s) ORDER BY job_id_source"
#define SQL_LINK_DELETE "DELETE FROM link WHERE job_id_source=\"%s\" OR job_id_destination=\"%s\""
#define SQL_LINK_DELETE1 "DELETE FROM link WHERE job_id_source=\"%s\" AND job_id_destination=\"%s\""
#define SQL_LINK_UPDATE "UPDATE link SET required_status=%ld WHERE job_id_source=\"%s\" AND job_id_destination=\"%s\""
#define SQL_LINK_ADD "INSERT INTO link (job_id_source,job_id_destination,required_status) VALUES (\"%s\",\"%s\",%ld)"


/*
 * Get the links which source jobs/jobsets are in the provided list
 *
 * For each link, the provided add_row() function is called with the
 * following parameters:
 *     1) the provided user_data_add_row parameter
 *     2) the source job ID
 *     3) the destination job ID 
 *     4) the status (1:waiting, 2:running, 3:completed or 4:failed)
 * add_row() must return 0 if successful or an other code in case of error.
 *
 * Return:
 *   0 --> No error
 *  -1 --> The add_row() function returned with a code different from 0
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_link_list (	const char *id_list,
		int (*add_row)(	void *, const char *, const char *,
				const char *),
		void *user_data_add_row,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	unsigned int ret;
	char **row;

#if HAVE_ASSERT_H
	assert (id_list != NULL && add_row != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_LINK_LIST,
				SQL_STRING_NON_ESCAPE, id_list,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	while ((row = (char **) lwc_delStartLL (rows)) != NULL) {
		if (add_row (	user_data_add_row,
				row[0],		/* job source ID */
				row[1],		/* job destination ID */
				row[2]) != 0)	/* status */
		{
			sql_free_row (row);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return -1;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, NULL);
	return 0;
}


/*
 * Get the links which source jobs/jobsets are in the provided list
 *
 * Return:
 *   0 --> No error.  rows contains the retrieved links and must be freed
 *         by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_link_get_list (	lwc_LL **rows,
			const char *id_list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL && id_list != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows, NULL,
				SQL_LINK_LIST,
				SQL_STRING_NON_ESCAPE, id_list,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Remove all the links associated with the provided job/jobset ID
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_link_del (	const char *job_id,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_LINK_DELETE,
				SQL_STRING, job_id,
				SQL_STRING, job_id,
				SQL_END);
	if (ret != 0) { 
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Remove the provided link
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_link_del1 (	const char *job_src, const char *job_dst,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_src != NULL && job_dst != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_LINK_DELETE1,
				SQL_STRING, job_src,
				SQL_STRING, job_dst,
				SQL_END);
	if (ret != 0) { 
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Change the type of the provided link
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_link_type (	const char *job_src, const char *job_dst,
		long int new_type,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_src != NULL && job_dst != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_LINK_UPDATE,
				SQL_INT, new_type,
				SQL_STRING, job_src,
				SQL_STRING, job_dst,
				SQL_END);
	if (ret != 0) { 
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Add a new link
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_link_add (	const char *job_src, const char *job_dst,
		long int new_type,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_src != NULL && job_dst != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_LINK_ADD,
				SQL_STRING, job_src,
				SQL_STRING, job_dst,
				SQL_INT, new_type,
				SQL_END);
	if (ret != 0) { 
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
