/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* certificate_cb.c - Certificate window */

#include <schedwi.h>

#include <gnutls/gnutls.h>
#include <gnutls/x509.h>

#include "support.h"
#include "interface.h"

#include <message_windows.h>
#include <certificate_cb.h>


/*
 * Callback for the `SAVE' button
 *
 * This function gets the certificate text from the binary textview and
 * write it in the file choosen by the user
 */
void
certificate_save (GtkButton *button)
{
	GtkWidget *dialog_cert;
	GtkWidget *dialog;
	gchar *filename;
	GError *err = NULL;
	GtkWidget *text;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *cert_txt;
	GIOChannel *out;
	gsize nb;

	dialog_cert = lookup_widget (	GTK_WIDGET (button),
					"dialog_certificate");

	/* Save file chooser dialog */
	dialog = gtk_file_chooser_dialog_new (
					"Save File",
					GTK_WINDOW (dialog_cert),
					GTK_FILE_CHOOSER_ACTION_SAVE,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					NULL);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {

		/* Get the text to write from the binary textview */
		text = lookup_widget (	GTK_WIDGET (button),
					"textview_certificate_binary");
		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
		gtk_text_buffer_get_start_iter (buffer, &start);
		gtk_text_buffer_get_end_iter (buffer, &end);
		cert_txt = gtk_text_buffer_get_text (	buffer,
							&start, &end,
							FALSE);

		/* Get the output file name */
		filename = gtk_file_chooser_get_filename (
						GTK_FILE_CHOOSER (dialog));

		/* Open the file */
		out = g_io_channel_new_file (filename, "w", &err);
		if (out == NULL) {
			gtk_widget_destroy (dialog);
			error_window (	_("Failed to write the certificate"),
					(err != NULL) ? err->message : NULL);
			if (err != NULL) {
				g_error_free (err);
			}
			g_free (filename);
			g_free (cert_txt);
			return;
		}
		if (err != NULL) {
			g_error_free (err);
			err = NULL;
		}
		g_free (filename);

		/* Write the certificate to the file */
		if (g_io_channel_write_chars (	out,
						cert_txt, -1,
						&nb,
						&err) != G_IO_STATUS_NORMAL)
		{
			gtk_widget_destroy (dialog);
			error_window (	_("Failed to write the certificate"),
					(err != NULL) ? err->message : NULL);
			if (err != NULL) {
				g_error_free (err);
			}
			g_io_channel_unref (out);
			g_free (cert_txt);
			return;
		}
		if (err != NULL) {
			g_error_free (err);
		}

		/* Close the file */
		g_io_channel_unref (out);
		g_free (cert_txt);
	}
	gtk_widget_destroy (dialog);
}


/*
 * Print the provided certificate details in the provided text_buffer.
 *
 * This function has been inspired by the print_certificate_info() function
 * from the certtool tool from the GNUTLS library.
 */
static void
print_certificate_info (GString *cert_txt, GtkTextBuffer *text_buffer)
{
	gnutls_datum_t certificate_string;
	gnutls_x509_crt_t crt;
	int ret;
	GtkTextIter start;

	gchar *s;
	unsigned char buffer[50 * 1024];
	char oid[128] = "", old_oid[128] = "";
	char serial[40], dn[256];
	size_t serial_size = sizeof (serial), dn_size, size;
	unsigned int i, j, version, bits, critical, key_usage, indx;
	const char* cprint;
	time_t tim;


	/*
	 * Convert the text certificate to a proper GNUTLS X509 certificate.
	 */
	gnutls_x509_crt_init (&crt);
	certificate_string.data = (unsigned char *)cert_txt->str;
	certificate_string.size = cert_txt->len;

	/* Try to convert from both formats (PEM and DER) */
	ret = gnutls_x509_crt_import (	crt,
					&certificate_string,
					GNUTLS_X509_FMT_PEM);
	if (ret != 0) {
		/*
		 * Not a PEM encoded certificate so check if it's a
		 * DER encoded one.
		 */
		if (gnutls_x509_crt_import (	crt,
						&certificate_string,
						GNUTLS_X509_FMT_DER) != 0)
		{
			gnutls_x509_crt_deinit (crt);
			error_window (
			_("Cannot display the text form of the certificate"),
				gnutls_strerror (ret));
			return;
		}
	}


	gtk_text_buffer_get_start_iter (text_buffer, &start);
	gtk_text_buffer_insert (text_buffer, &start, _("Certificate:\n"), -1);

	/* Version */
	version = gnutls_x509_crt_get_version (crt);
	s = g_strdup_printf (_("\tVersion: %d\n"), version);
	gtk_text_buffer_insert (text_buffer, &start, s, -1);
	g_free (s);

	/* Serial number */
	if (gnutls_x509_crt_get_serial (crt, serial, &serial_size) >= 0) {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\tSerial Number: "), -1);
		for (i = 0; i < serial_size; i++) {
			s = g_strdup_printf (	"%.2x ",
						(unsigned char) serial[i]);
			gtk_text_buffer_insert (text_buffer, &start, s, -1);
			g_free (s);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
	}

	/* Signature algorithm */
	gtk_text_buffer_insert (text_buffer, &start,
				_("\tSignature Algorithm: "), -1);
	ret = gnutls_x509_crt_get_signature_algorithm (crt);
	cprint = gnutls_sign_algorithm_get_name (ret);
	if (cprint == NULL) {
		gtk_text_buffer_insert (text_buffer, &start, _("Unknown"), -1);
	}
	else {
		gtk_text_buffer_insert (text_buffer, &start, cprint, -1);
	}
	gtk_text_buffer_insert (text_buffer, &start, "\n", 1);

	/* Issuer */
	dn_size = sizeof (dn);
	ret = gnutls_x509_crt_get_issuer_dn (crt, dn, &dn_size);
	if (ret >= 0) {
		s = g_strdup_printf (_("\tIssuer: %s\n"), dn);
	}
	else {
		s = g_strdup_printf (	_("\tIssuer: ERROR (%s)\n"),
					gnutls_strerror(ret));
	}
	gtk_text_buffer_insert (text_buffer, &start, s, -1);
	g_free (s);

	/* Validity */
	gtk_text_buffer_insert (text_buffer, &start, _("\tValidity\n"), -1);

	tim = gnutls_x509_crt_get_activation_time (crt);
	s = g_strdup_printf (_("\t\tNot Before: %s"), ctime (&tim));
	gtk_text_buffer_insert (text_buffer, &start, s, -1);
	g_free (s);

	tim = gnutls_x509_crt_get_expiration_time (crt);
	s = g_strdup_printf (_("\t\tNot After: %s"), ctime (&tim));
	gtk_text_buffer_insert (text_buffer, &start, s, -1);
	g_free (s);

	/* Subject */
	dn_size = sizeof (dn);
	ret = gnutls_x509_crt_get_dn (crt, dn, &dn_size);
	if (ret >= 0) {
		s = g_strdup_printf (_("\tSubject: %s\n"), dn);
	}
	else {
		s = g_strdup_printf (	_("\tSubject: ERROR (%s)\n"),
					gnutls_strerror(ret));
	}
	gtk_text_buffer_insert (text_buffer, &start, s, -1);
	g_free (s);

	/* Subject public key info */
	gtk_text_buffer_insert (text_buffer, &start,
		_("\tSubject Public Key Info:\n\t\tPublic Key Algorithm: "),
		-1);
	ret = gnutls_x509_crt_get_pk_algorithm (crt, &bits);
	cprint = gnutls_pk_algorithm_get_name (ret);
	if (cprint == NULL) {
		gtk_text_buffer_insert (text_buffer, &start, _("Unknown"), -1);
	}
	else {
		gtk_text_buffer_insert (text_buffer, &start, cprint, -1);
	}
	if (bits != 0) {
		s = g_strdup_printf (_(" (%u bits)"), bits);
		gtk_text_buffer_insert (text_buffer, &start, s, -1);
		g_free (s);
	}
	gtk_text_buffer_insert (text_buffer, &start, "\n", 1);

	/*
	 * X509 Extensions
	 */
	if (version >= 3) {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\tX509 extensions:\n"), -1);
	}
	
	/* Subject alternative name */
	ret = 0;
	for (i = 0; ret >= 0; i++) {
		size = sizeof (buffer);
		ret = gnutls_x509_crt_get_subject_alt_name (	crt, i,
								buffer, &size,
								&critical);

		if (i == 0 && ret != GNUTLS_E_REQUESTED_DATA_NOT_AVAILABLE) {
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tSubject Alternative name:"),
					-1);
			if (critical != 0) {
				gtk_text_buffer_insert (text_buffer, &start,
							_(" (critical)"), -1);
			}
			gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
		}
		
		if (ret < 0 && ret != GNUTLS_E_REQUESTED_DATA_NOT_AVAILABLE) {
			gtk_text_buffer_insert (text_buffer, &start,
			_("\t\t\tFound unsupported alternative name.\n"), -1);
		}
		else {
			s = NULL;
			switch (ret) {
				case GNUTLS_SAN_DNSNAME:
					s = g_strdup_printf (
						_("\t\t\tDNSname: %s\n"),
						buffer);
					break;
				case GNUTLS_SAN_RFC822NAME:
					s = g_strdup_printf (
						_("\t\t\tRFC822name: %s\n"),
						buffer);
					break;
				case GNUTLS_SAN_URI:
					s = g_strdup_printf (
						_("\t\t\tURI: %s\n"),
						buffer);
					break;
				case GNUTLS_SAN_IPADDRESS:
					s = g_strdup_printf (
						_("\t\t\tIPAddress: %s\n"),
						buffer);
					break;
			}
			if (s != NULL) {
				gtk_text_buffer_insert (text_buffer, &start,
							s, -1);
				g_free (s);
			}
		}
	}

	/* CRL dist points */
	ret = 0;
	for (i = 0; ret >= 0; i++) {
		size = sizeof (buffer);
		ret = gnutls_x509_crt_get_crl_dist_points (	crt, i,
								buffer, &size,
								NULL,
								&critical);

		if (i == 0 && ret != GNUTLS_E_REQUESTED_DATA_NOT_AVAILABLE) {
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tCRL Distribution points:"),
					-1);
			if (critical != 0) {
				gtk_text_buffer_insert (text_buffer, &start,
							_(" (critical)"), -1);
			}
			gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
		}
		
		if (ret < 0 && ret != GNUTLS_E_REQUESTED_DATA_NOT_AVAILABLE) {
			s = g_strdup_printf (_("\t\t\tError decoding: %s\n"),
						gnutls_strerror (ret));
			gtk_text_buffer_insert (text_buffer, &start, s, -1);
			g_free (s);
		}
		else {
			s = NULL;
			switch (ret) {
				case GNUTLS_SAN_DNSNAME:
					s = g_strdup_printf (
						_("\t\t\tDNSname: %s\n"),
						buffer);
					break;
				case GNUTLS_SAN_RFC822NAME:
					s = g_strdup_printf (
						_("\t\t\tRFC822name: %s\n"),
						buffer);
					break;
				case GNUTLS_SAN_URI:
					s = g_strdup_printf (
						_("\t\t\tURI: %s\n"),
						buffer);
					break;
				case GNUTLS_SAN_IPADDRESS:
					s = g_strdup_printf (
						_("\t\t\tIPAddress: %s\n"),
						buffer);
					break;
			}
			if (s != NULL) {
				gtk_text_buffer_insert (text_buffer, &start,
							s, -1);
				g_free (s);
			}
		}
	}
	
	/* Check for basic constraints */
	ret = gnutls_x509_crt_get_ca_status (crt, &critical);
	if (ret >= 0) {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tBasic Constraints:"),
					-1);
		if (critical != 0) {
			gtk_text_buffer_insert (text_buffer, &start,
						_(" (critical)"), -1);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);

		if (ret==0) {
			gtk_text_buffer_insert (text_buffer, &start,
						_("\t\t\tCA:FALSE\n"), -1);
		}
		else {
			gtk_text_buffer_insert (text_buffer, &start,
						_("\t\t\tCA:TRUE\n"), -1);
		}
	}

	/* Key Usage */
	ret = gnutls_x509_crt_get_key_usage (crt, &key_usage, &critical);
	if (ret >= 0) {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tKey usage:"), -1);
		if (critical != 0) {
			gtk_text_buffer_insert (text_buffer, &start,
						_(" (critical)"), -1);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);

		if (key_usage & GNUTLS_KEY_DIGITAL_SIGNATURE)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tDigital signature.\n"), -1);
		if (key_usage & GNUTLS_KEY_NON_REPUDIATION)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tNon repudiation.\n"), -1);
		if (key_usage & GNUTLS_KEY_KEY_ENCIPHERMENT)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tKey encipherment.\n"), -1);
		if (key_usage & GNUTLS_KEY_DATA_ENCIPHERMENT)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tData encipherment.\n"), -1);
		if (key_usage & GNUTLS_KEY_KEY_AGREEMENT)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tKey agreement.\n"), -1);
		if (key_usage & GNUTLS_KEY_KEY_CERT_SIGN)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tCertificate signing.\n"), -1);
		if (key_usage & GNUTLS_KEY_CRL_SIGN)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tCRL signing.\n"), -1);
		if (key_usage & GNUTLS_KEY_ENCIPHER_ONLY)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tKey encipher only.\n"), -1);
		if (key_usage & GNUTLS_KEY_DECIPHER_ONLY)
			gtk_text_buffer_insert (text_buffer, &start,
					_("\t\t\tKey decipher only.\n"), -1);
	}

	/* Key Purpose identifiers */
	i = 0;
	size = sizeof (buffer);
	ret = gnutls_x509_crt_get_key_purpose_oid (	crt, 0, buffer, &size,
							&critical);
	if (ret >= 0) {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tKey purpose OIDs:"), -1);
		if (critical != 0) {
			gtk_text_buffer_insert (text_buffer, &start,
						_(" (critical)"), -1);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
		do {
			size = sizeof (buffer);
			ret = gnutls_x509_crt_get_key_purpose_oid (crt, i,
						buffer, &size, &critical);
			if (ret >= 0) {
				if (g_ascii_strcasecmp ((const gchar *)buffer,
						GNUTLS_KP_TLS_WWW_SERVER) == 0)
					gtk_text_buffer_insert (
						text_buffer, &start,
						_("\t\t\tTLS WWW Server.\n"),
						-1);

				else if (g_ascii_strcasecmp ((const gchar *)buffer,
						GNUTLS_KP_TLS_WWW_CLIENT) == 0)
					gtk_text_buffer_insert (
						text_buffer, &start,
						_("\t\t\tTLS WWW Client.\n"),
						-1);

				else if (g_ascii_strcasecmp ((const gchar *)buffer,
						GNUTLS_KP_CODE_SIGNING) == 0)
					gtk_text_buffer_insert (
						text_buffer, &start,
						_("\t\t\tCode signing.\n"),
						-1);

				else if (g_ascii_strcasecmp ((const gchar *)buffer,
					GNUTLS_KP_EMAIL_PROTECTION) == 0)
					gtk_text_buffer_insert (
						text_buffer, &start,
						_("\t\t\tEmail protection.\n"),
						-1);

				else if (g_ascii_strcasecmp ((const gchar *)buffer,
						GNUTLS_KP_TIME_STAMPING) == 0)
					gtk_text_buffer_insert (
						text_buffer, &start,
						_("\t\t\tTime stamping.\n"),
						-1);

				else if (g_ascii_strcasecmp ((const gchar *)buffer,
						GNUTLS_KP_OCSP_SIGNING) == 0)
					gtk_text_buffer_insert (
						text_buffer, &start,
						_("\t\t\tOCSP signing.\n"),
						-1);

				else if (g_ascii_strcasecmp ((const gchar *)buffer,
						GNUTLS_KP_ANY) == 0)
					gtk_text_buffer_insert (
						text_buffer, &start,
						_("\t\t\tAny purpose.\n"),
						-1);

				else {
					gtk_text_buffer_insert (
							text_buffer, &start,
							_("\t\t\t"), -1);
					gtk_text_buffer_insert (
							text_buffer, &start,
							(const gchar *)buffer, -1);
					gtk_text_buffer_insert (
							text_buffer, &start,
							"\n", 1);
				}
			}
			i++;
		} while (ret >= 0);
	}

	/* Subject Key ID */
	size = sizeof (buffer);
	ret = gnutls_x509_crt_get_subject_key_id (	crt, buffer, &size,
							&critical);
	if (ret < 0 && ret != GNUTLS_E_REQUESTED_DATA_NOT_AVAILABLE) {
		s = g_strdup_printf (
				_("\t\tError getting subject key id: %s\n"),
				gnutls_strerror(ret));
		gtk_text_buffer_insert (text_buffer, &start, s, -1);
		g_free (s);
	}

	if (ret >= 0) {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tSubject Key ID:"), -1);
		if (critical != 0) {
			gtk_text_buffer_insert (text_buffer, &start,
						_(" (critical)"), -1);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n\t\t\t", -1);

		for (i = 0; i < size; i++) {
			s = g_strdup_printf (	"%.2x ",
						(unsigned char) buffer[i]);
			gtk_text_buffer_insert (text_buffer, &start, s, -1);
			g_free (s);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
	}

	/* Authority Key ID */
	size = sizeof (buffer);
	ret = gnutls_x509_crt_get_authority_key_id (	crt, buffer, &size,
							&critical);
	if (ret < 0 && ret != GNUTLS_E_REQUESTED_DATA_NOT_AVAILABLE) {
		s = g_strdup_printf (
				_("\t\tError getting authority key id: %s\n"),
				gnutls_strerror(ret));
		gtk_text_buffer_insert (text_buffer, &start, s, -1);
		g_free (s);
	}
	
	if (ret >= 0) {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tAuthority Key ID:"), -1);
		if (critical != 0) {
			gtk_text_buffer_insert (text_buffer, &start,
						_(" (critical)"), -1);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);

		for (i = 0; i < size; i++) {
			s = g_strdup_printf (	"%.2x ",
						(unsigned char) buffer[i]);
			gtk_text_buffer_insert (text_buffer, &start, s, -1);
			g_free (s);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
	}

	/* Other extensions */
	indx = 0;
	ret = 0;
	for (i = 0; ret >= 0; i++) {
		size = sizeof (oid);
		ret = gnutls_x509_crt_get_extension_oid (crt, i, oid, &size);

		if (ret >= 0) {
			if (	   strcmp (oid, "2.5.29.17") == 0
				|| strcmp (oid, "2.5.29.19") == 0
				|| strcmp (oid, "2.5.29.31") == 0
				|| strcmp (oid, "2.5.29.37") == 0
				|| strcmp (oid, "2.5.29.14") == 0
				|| strcmp (oid, "2.5.29.35") == 0
				|| strcmp (oid, "2.5.29.15") == 0)
			{
				continue;
			}

			if (strcmp (oid, old_oid) == 0) {
				indx++;
			}
			else {
				indx = 0;
			}

			s = g_strdup_printf (	_("\t\t%s: "), oid);
			gtk_text_buffer_insert (text_buffer, &start, s, -1);
			g_free (s);

			size = sizeof (buffer);
			ret = gnutls_x509_crt_get_extension_by_oid (crt, oid,
					indx, buffer, &size, &critical);
			if (ret >= 0) {
				if (critical != 0) {
					gtk_text_buffer_insert (
							text_buffer, &start,
							_("(critical)\n"), -1);
				}
				else {
					gtk_text_buffer_insert (
							text_buffer, &start,
							"\n", 1);
				}

				gtk_text_buffer_insert (text_buffer, &start,
							_("\t\t\tDER Data:"),
							-1);
				for (j = 0; j < size; j++) {
					s = g_strdup_printf ("%.2x",
						(unsigned char) buffer[j]);
					gtk_text_buffer_insert (text_buffer,
								&start, s, -1);
					g_free (s);
				}
				gtk_text_buffer_insert (text_buffer, &start,
							"\n", 1);
			}
			ret = 0;
			strcpy (old_oid, oid);
		}
	}

	/*
	 * Other information
	 */
	gtk_text_buffer_insert (text_buffer, &start,
				_("\tOther information:\n"), -1);

	/* Fingerprint */
	size = sizeof (buffer);
	ret = gnutls_x509_crt_get_fingerprint (	crt, GNUTLS_DIG_MD5,
						buffer, &size);
	if (ret < 0) {
		s = g_strdup_printf (
			_("\t\tError in fingerprint calculation: %s\n"),
			gnutls_strerror(ret));
		gtk_text_buffer_insert (text_buffer, &start, s, -1);
		g_free (s);
	}
	else {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tFingerprint: "), -1);
		for (i = 0; i < size; i++) {
			s = g_strdup_printf (	"%.2x ",
						(unsigned char) buffer[i]);
			gtk_text_buffer_insert (text_buffer, &start, s, -1);
			g_free (s);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
	}

	size = sizeof (buffer);
	ret = gnutls_x509_crt_get_key_id (crt, 0, buffer, &size);
	if (ret < 0) {
		s = g_strdup_printf (
			_("\t\tError in public key id calculation: %s\n"),
			gnutls_strerror(ret));
		gtk_text_buffer_insert (text_buffer, &start, s, -1);
		g_free (s);
	}
	else {
		gtk_text_buffer_insert (text_buffer, &start,
					_("\t\tPublic Key ID: "), -1);
		for (i = 0; i < size; i++) {
			s = g_strdup_printf (	"%.2x ",
						(unsigned char) buffer[i]);
			gtk_text_buffer_insert (text_buffer, &start, s, -1);
			g_free (s);
		}
		gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
	}

	gtk_text_buffer_insert (text_buffer, &start, "\n", 1);
	gnutls_x509_crt_deinit (crt);
}


/*
 * Initialize the fields of a certificate window
 *
 * Return:
 *   0 --> No error (even if the translation of the certificate to a human
 *         readable text failed)
 *   1 --> No certificate is provided
 */
static int 
dialog_certificate_init (GtkWidget *dialog_cert, GString *cert_txt)
{
	GtkWidget *text_bin, *text_text;
	GtkTextBuffer *buffer;

	gtk_window_set_type_hint (	GTK_WINDOW (dialog_cert),
					GDK_WINDOW_TYPE_HINT_NORMAL);

	if (cert_txt == NULL || cert_txt->len == 0) {
		return 1;
	}

	/* Copy the certificate to the binary text view */
	text_bin = lookup_widget (	GTK_WIDGET (dialog_cert),
					"textview_certificate_binary");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_bin));
	gtk_text_buffer_set_text (buffer, cert_txt->str, cert_txt->len);


	/* Display a text version of the certificate */
	text_text = lookup_widget (	GTK_WIDGET (dialog_cert),
					"textview_certificate_text");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_text));
	print_certificate_info (cert_txt, buffer);

	return 0;
}


/*
 * Create a certificate window
 *
 * Return:
 *   The new window or
 *   NULL in case of error or if the provided certificate id NULL or empty
 */
GtkWidget *
new_dialog_certificate (GString *cert_txt)
{
	GtkWidget *dialog_cert;

	dialog_cert = create_dialog_certificate ();
	if (dialog_certificate_init (dialog_cert, cert_txt) != 0) {
		info_window (NULL, _("No certificate to display"));
		gtk_widget_destroy (dialog_cert);
		return NULL;
	}
	else {
		gtk_widget_show (dialog_cert);
		return dialog_cert;
	}
}

/*------------------------======= End Of File =======------------------------*/
