/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
     
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
     
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* child_job.c -- Structure to draw a job/jobset on a GnomeCanvas */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <math.h>

#include <message_windows.h>
#include <cache_pixbuf.h>
#include <copy_paste.h>
#include <sql_children_job.h>
#include <commands.h>
#include <canvas_utils.h>
#include <cursor.h>
#include <child_job.h>
#include <schedwi_main.h>
#include <main_cb.h>
#include <schedwi_g_utils.h>
#include <sql_link.h>
#include <job_cb.h>
#include <sql_link.h>
#include <statistics_cb.h>
#include <workload.h>
#include <job_select.h>

#define CHILD_JOB_MENU_ICON_SIZE 20

#define CHILD_JOB_INNER_SPACE 5
#define CHILD_JOB_TEXT_SPACE 3
#define CHILD_JOB_TEXT_LENGTH_BY_LINE 12
#define CHILD_JOB_BG_ROUND_SIZE 10
#define CHILD_JOB_GAUGE_WIDTH 10
#define CHILD_JOB_GAUGE_HEIGHT 60
#define CHILD_JOB_GAUGE_FRAME_COLOR "#707070"
#define CHILD_JOB_GAUGE_COLOR "#33DD33"
#define CHILD_JOB_GAUGE_FRAME_WIDTH 1


/*
 * Destroy the provided child_job_t object
 */
void
destroy_child_job (child_job_t *ptr)
{
	if (ptr != NULL) {
		g_free (ptr->id);
		g_free (ptr->name);
		g_free (ptr->status_message);
		g_slist_free (ptr->links_out);
		g_slist_free (ptr->links_in);
		if (ptr->canvas_group != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->canvas_group));
		}
		g_free (ptr);
	}
}


/*
 * Free the provided element of the job list.  Function used by
 * g_slist_foreach() to free all the elements of the list
 */
static void
destroy_element_child_job (gpointer data, gpointer user_data)
{
	destroy_child_job ((child_job_t *)data);
}


/*
 * Destroy the provided list of child_job_t objects
 */
void
destroy_child_job_list (GSList *list)
{
	g_slist_foreach (list, destroy_element_child_job, NULL);
	g_slist_free (list);
}


/*
 * Load an icon from the database
 *
 * Return:
 *   TRUE --> No error.  If there is no icon directly assosiated with the job
 *            in the database, the icon is copied from the provided default
 *            pixbuf.
 *            These returned pixbufs belongs to the cache and must not be
 *            freed/unreferenced by the caller
 *  FALSE --> Error.  An error message popup has been displayed
 */
static gboolean
load_child_icon (	int workload_date,
			const gchar *id,
			canvas_mode_t mode,
			const gchar *name, gchar type,
			GdkPixbuf **pixbufs,
			const gchar *tablename,
			GdkPixbuf **default_pixbufs)
{
	char *pixdata_stream;
	unsigned long int pixdata_stream_len;
	gchar *s1, *s2, *err_msg;

	if (sql_children_get_icon (workload_date, id, tablename,
				&pixdata_stream, &pixdata_stream_len,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return FALSE;
	}

	/*
	 * No pixdata directly associated with this job/jobset.
	 * Copy the default provided one instead
	 */
	if (pixdata_stream == NULL) {
		pixbufs[DEFAULT] = default_pixbufs[DEFAULT];
		pixbufs[HIGHLIGHTED] = default_pixbufs[HIGHLIGHTED];
		pixbufs[SELECTED] = default_pixbufs[SELECTED];
		pixbufs[SELECTED_HIGHLIGHTED] =
					default_pixbufs[SELECTED_HIGHLIGHTED];
		pixbufs[CUT] = default_pixbufs[CUT];
	}
	else {
		if (add_pixbuf_to_cache (pixdata_stream, pixdata_stream_len,
				&(pixbufs[DEFAULT]),
				&(pixbufs[HIGHLIGHTED]),
				&(pixbufs[SELECTED]),
				&(pixbufs[SELECTED_HIGHLIGHTED]),
				&(pixbufs[CUT]),
				&err_msg) != TRUE)
		{
			s1 = g_strdup_printf (
				(type == 0) ?
				_("Could not load an icon for the jobset %s"):
				_("Could not load an icon for the job %s"),
				name);
			s2 = g_strdup_printf (
			_("%s.\nA replacement icon will be used instead."),
				err_msg);
			g_free (err_msg);
			warning_window (s1, s2);
			g_free (s1);
			g_free (s2);
			pixbufs[DEFAULT] = NULL;
			pixbufs[HIGHLIGHTED] = NULL;
			pixbufs[SELECTED] = NULL;
			pixbufs[SELECTED_HIGHLIGHTED] = NULL;
			pixbufs[CUT] = NULL;
		}
		free (pixdata_stream);
	}
	return TRUE;
}


/*
 * Initialize a new child_job_t object
 *
 * Return:
 *   TRUE --> No error
 *   FALSE --> Error (a message has been displayed)
 */
static gboolean
child_job_init (child_job_t *ptr,
		const gchar *id, const gchar *name, gchar type,
		gdouble x, gdouble y,
		jobset_item_t *parent_jobset)
{
	canvas_mode_t mode;

	mode = parent_jobset->mode;

	/* Jobset */
	if (type == 0) {
		/* Retrieve the default icon  */
		if (load_child_icon (parent_jobset->workload_date,
			id, mode, name, type,
			ptr->icon[JOB_STATUS_STATE_UNDEFINED],
			job_status_state_to_jobset_icon_tablename (
						JOB_STATUS_STATE_UNDEFINED),
			parent_jobset->jobset_icon[JOB_STATUS_STATE_UNDEFINED])
				!= TRUE)
		{
			return FALSE;
		}
		/* If the mode is not editing, retrieve the other icons too */
		if (mode != EDITING) {
			ptr->icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT] =
				ptr->icon[JOB_STATUS_STATE_UNDEFINED][CUT];
			if (	   load_child_icon (parent_jobset->workload_date,
					id, mode, name, type,
					ptr->icon[JOB_STATUS_STATE_COMPLETED],
					job_status_state_to_jobset_icon_tablename (JOB_STATUS_STATE_COMPLETED),
					parent_jobset->jobset_icon[JOB_STATUS_STATE_COMPLETED])
						!= TRUE
				|| load_child_icon (parent_jobset->workload_date,
					id, mode, name, type,
					ptr->icon[JOB_STATUS_STATE_FAILED],
					job_status_state_to_jobset_icon_tablename (JOB_STATUS_STATE_FAILED),
					parent_jobset->jobset_icon[JOB_STATUS_STATE_FAILED])
						!= TRUE
				|| load_child_icon (parent_jobset->workload_date,
					id, mode, name, type,
					ptr->icon[JOB_STATUS_STATE_WAITING],
					job_status_state_to_jobset_icon_tablename (JOB_STATUS_STATE_WAITING),
					parent_jobset->jobset_icon[JOB_STATUS_STATE_WAITING])
						!= TRUE
				|| load_child_icon (parent_jobset->workload_date,
					id, mode, name, type,
					ptr->icon[JOB_STATUS_STATE_RUNNING],
					job_status_state_to_jobset_icon_tablename (JOB_STATUS_STATE_RUNNING),
					parent_jobset->jobset_icon[JOB_STATUS_STATE_RUNNING])
						!= TRUE)
			{
				return FALSE;
			}
		}
	}
	else {
	/* Job */
		if (load_child_icon (parent_jobset->workload_date,
			id, mode, name, type,
			ptr->icon[JOB_STATUS_STATE_UNDEFINED],
			job_status_state_to_job_icon_tablename (
							JOB_STATUS_STATE_UNDEFINED),
			parent_jobset->job_icon[JOB_STATUS_STATE_UNDEFINED])
				!= TRUE)
		{
			return FALSE;
		}
		if (mode != EDITING) {
			ptr->icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT] =
				ptr->icon[JOB_STATUS_STATE_UNDEFINED][CUT];
			if (	   load_child_icon (parent_jobset->workload_date,
					id, mode, name, type,
					ptr->icon[JOB_STATUS_STATE_COMPLETED],
					job_status_state_to_job_icon_tablename (JOB_STATUS_STATE_COMPLETED),
					parent_jobset->job_icon[JOB_STATUS_STATE_COMPLETED])
						!= TRUE
				|| load_child_icon (parent_jobset->workload_date,
					id, mode, name, type,
					ptr->icon[JOB_STATUS_STATE_FAILED],
					job_status_state_to_job_icon_tablename (JOB_STATUS_STATE_FAILED),
					parent_jobset->job_icon[JOB_STATUS_STATE_FAILED])
						!= TRUE
				|| load_child_icon (parent_jobset->workload_date,
					id, mode, name, type,
					ptr->icon[JOB_STATUS_STATE_WAITING],
					job_status_state_to_job_icon_tablename (JOB_STATUS_STATE_WAITING),
					parent_jobset->job_icon[JOB_STATUS_STATE_WAITING])
						!= TRUE
				|| load_child_icon (parent_jobset->workload_date,
					id, mode, name, type,
					ptr->icon[JOB_STATUS_STATE_RUNNING],
					job_status_state_to_job_icon_tablename (JOB_STATUS_STATE_RUNNING),
					parent_jobset->job_icon[JOB_STATUS_STATE_RUNNING])
						!= TRUE)
			{
				return FALSE;
			}
		}
	}
	/* If the mode is editing, set the icons to the default one */
	if (mode == EDITING) {
		ptr->icon[JOB_STATUS_STATE_COMPLETED][DEFAULT] =
		ptr->icon[JOB_STATUS_STATE_FAILED][DEFAULT] =
		ptr->icon[JOB_STATUS_STATE_WAITING][DEFAULT] =
		ptr->icon[JOB_STATUS_STATE_RUNNING][DEFAULT] =
		ptr->icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT];

		ptr->icon[JOB_STATUS_STATE_COMPLETED][HIGHLIGHTED] =
		ptr->icon[JOB_STATUS_STATE_FAILED][HIGHLIGHTED] =
		ptr->icon[JOB_STATUS_STATE_WAITING][HIGHLIGHTED] =
		ptr->icon[JOB_STATUS_STATE_RUNNING][HIGHLIGHTED] =
		ptr->icon[JOB_STATUS_STATE_UNDEFINED][HIGHLIGHTED];

		ptr->icon[JOB_STATUS_STATE_COMPLETED][SELECTED] =
		ptr->icon[JOB_STATUS_STATE_FAILED][SELECTED] =
		ptr->icon[JOB_STATUS_STATE_WAITING][SELECTED] =
		ptr->icon[JOB_STATUS_STATE_RUNNING][SELECTED] =
		ptr->icon[JOB_STATUS_STATE_UNDEFINED][SELECTED];

		ptr->icon[JOB_STATUS_STATE_COMPLETED][SELECTED_HIGHLIGHTED] =
		ptr->icon[JOB_STATUS_STATE_FAILED][SELECTED_HIGHLIGHTED] =
		ptr->icon[JOB_STATUS_STATE_WAITING][SELECTED_HIGHLIGHTED] =
		ptr->icon[JOB_STATUS_STATE_RUNNING][SELECTED_HIGHLIGHTED] =
		ptr->icon[JOB_STATUS_STATE_UNDEFINED][SELECTED_HIGHLIGHTED];

		ptr->icon[JOB_STATUS_STATE_COMPLETED][CUT] =
		ptr->icon[JOB_STATUS_STATE_FAILED][CUT] =
		ptr->icon[JOB_STATUS_STATE_WAITING][CUT] =
		ptr->icon[JOB_STATUS_STATE_RUNNING][CUT] =
		ptr->icon[JOB_STATUS_STATE_UNDEFINED][CUT];
	}


	ptr->parent_jobset = parent_jobset;
	ptr->type = type;
	ptr->id = g_strdup (id);
	ptr->name = g_strdup (name);
	ptr->x = x;
	ptr->y = y;
	ptr->selected = FALSE;
	ptr->radius = 0.0;
	ptr->run_time = 0;
	ptr->start_time = 0;
	ptr->start_limit = 0;
	ptr->max_duration = 0;
	ptr->status_message = NULL;
	return TRUE;
}


/*
 * Create and return a new initialized child_job_t object
 *
 * Return:
 *   The new object to be freed by the caller by destroy_child_job() or
 *   NULL in case of error (an error message has been displayed)
 */
child_job_t *
new_child_job (	const gchar *id, const gchar *name, gchar type,
		gdouble x, gdouble y,
		jobset_item_t *parent_jobset)
{
	child_job_t *ptr;

	ptr = g_new0 (child_job_t, 1);

	if (child_job_init (ptr, id, name, type, x, y, parent_jobset) == FALSE)
	{
		g_free (ptr);
		return NULL;
	}
	return ptr;
}


/*
 * Structure to exchange data between get_children_job_list()
 * and add_child_to_list()
 */
struct callback_data {
	GSList *list;
	jobset_item_t *parent_jobset;
};


/*
 * Add a child to the job list
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  A message has been displayed
 */
static int
add_child_to_list (	void *user_data ,const char *id, const char *name,
			const char *type, const char *x, const char *y) 
{
	struct callback_data *data = user_data;
	child_job_t *child_ptr;

	child_ptr = new_child_job (	id, name,
					(type == NULL || *type == '0')? 0: 1,
					g_ascii_strtod (x, NULL),
					g_ascii_strtod (y, NULL),
					data->parent_jobset);
	if (child_ptr == NULL) {
		return -1;
	}
	data->list = g_slist_prepend (data->list, child_ptr);
	return 0;
}


/*
 * Build and return the list of all the children of the provided jobset
 *
 * Return:
 *   The new list to be free by the caller by
 *          destroy_child_job_list (list)
 *    The returned value may be NLL in case of error (a message has been
 *    displayed) or if the provided jobset contains no job
 */
GSList *
get_child_job_list (const gchar *parent_id, jobset_item_t *parent_jobset)
{
	struct callback_data data;

	data.list = NULL;
	data.parent_jobset = parent_jobset;
	if (sql_children_job_list (parent_jobset->workload_date, parent_id,
			add_child_to_list, &data,
			(void (*)(void *, const char*, unsigned int))
				error_window_ignore_errno,
			_("Database error")) != 0)
	{
		destroy_child_job_list (data.list);
		return NULL;
	}
	return data.list;
}


/*
 * Given a list of child_job_t object, return the string containing all the
 * job ids separated by `,'.  This string can then be used in SQL requests
 *
 * Return:
 *   The string (to be freed by the caller by g_free())
 */
gchar *
get_child_job_list_ids (GSList *list)
{
	GPtrArray *a;
	gchar *s;

	a = g_ptr_array_sized_new (20);
	while (list != NULL) {
		g_ptr_array_add (a, ((child_job_t *)list->data)->id);
		list = g_slist_next (list);
	}
	g_ptr_array_add (a, NULL);

	s = g_strjoinv (",", (gchar **)a->pdata);
	g_ptr_array_free (a, TRUE);
	return s;

}


/*
 * Add links to the job link lists
 */
void
child_job_add_link_out (child_job_t *ptr, link_t *link)
{
	if (ptr != NULL) {
		ptr->links_out = g_slist_prepend (ptr->links_out, link);
	}
}

void
child_job_add_link_in (child_job_t *ptr, link_t *link)
{
	if (ptr != NULL) {
		ptr->links_in = g_slist_prepend (ptr->links_in, link);
	}
}


/*
 * Comparison function used by g_slist_find_custom() to find a job given
 * its ID
 */
static gint
compare_child_job (gconstpointer a, gconstpointer b)
{
	const child_job_t *ptr = a;
	const gchar *id = b;

	return strcmp (ptr->id, id);
}


/*
 * Find a job in a list given its ID
 *
 * Return the child_job_t object or
 * NULL if the job ID is not in the list
 */
child_job_t *
child_job_find (GSList *jobs, const gchar *id)
{
	GSList *item;

	item = g_slist_find_custom (jobs, id, compare_child_job);
	if (item == NULL) {
		return NULL;
	}
	return (child_job_t *)item->data;
}


/*
 * Wrap a string using `max_chars' characters by line
 *
 * Return:
 *   The new string to be freed by the caller by g_free()
 */
static gchar *
wrap_string (const gchar *str, guint max_chars)
{
	gchar **a, *s;
	guint a_len, str_len, i;

	str_len = strlen (str);
	a_len = str_len / max_chars + ((str_len % max_chars != 0) ? 1: 0);

	a = (gchar **) g_malloc (sizeof (gchar *) * (a_len + 1));
	for (i = 0; i < a_len; i++) {
		a[i] = g_strndup (str + i * max_chars, max_chars);
	}
	a[i] = NULL;
	s = g_strjoinv ("\n", a);
	for (i = 0; i < a_len; i++) {
		g_free (a[i]);
	}
	g_free (a);
	return s;
}


/*
 * Redraw the job/jobset according to its new mode (selected, highlighted, ...)
 */
static void
child_job_set_mode (child_job_t *ptr, pixbuf_highlight_t new_mode)
{
	GdkColor *bg_color;

	if (ptr == NULL) {
		return;
	}

	gnome_canvas_item_set (	ptr->canvas_icon,
				"pixbuf", ptr->icon[ptr->state][new_mode],
				NULL);
	gnome_canvas_item_set (
		ptr->canvas_name,
		"fill-color-gdk", &(ptr->parent_jobset->text_color[new_mode]),
		NULL);

	if (new_mode == DEFAULT) {
		bg_color = NULL;
	}
	else {
		bg_color =  &(ptr->parent_jobset->base_color[new_mode]);
	}

	gnome_canvas_item_set (	ptr->canvas_name_bg,
				"fill-color-gdk", bg_color,
				NULL);
	gnome_canvas_item_set (	ptr->canvas_name_bg_left,
				"fill-color-gdk", bg_color,
				NULL);
	gnome_canvas_item_set (	ptr->canvas_name_bg_right,
				"fill-color-gdk", bg_color,
				NULL);
}


/*
 * Select the provided job/jobset
 */
void
child_job_select (child_job_t *ptr)
{
	ptr->selected = TRUE;
	child_job_set_mode (ptr, SELECTED);
}


/*
 * Un-select the provided job/jobset
 */
void
child_job_unselect (child_job_t *ptr)
{
	ptr->selected = FALSE;
	if (copy_paste_is_this_id_cut (ptr->id) == TRUE) {
		child_job_set_mode (ptr, CUT);
	}
	else {
		child_job_set_mode (ptr, DEFAULT);
	}
}


/*
 * Comparison function for the g_slist_find_custom() function
 */
static gint
child_job_find_job (gconstpointer a, gconstpointer b)
{
	child_job_t *ptr = (child_job_t *)a;
	const gchar *id = (const gchar *)b;

	return (gint)strcmp (ptr->id, id);
}


/*
 * Redraw the provided job/jobset (because it has been removed from the
 * copy/paste clipboard).  The job/jobset is redrew only if it is
 * currently displayed in the canvas.
 */
static void
child_job_cut_paste_clear (const gchar *id, gpointer data)
{
	GtkWidget *canvas;
	jobset_item_t *jobset;
	GSList *item;

	canvas = lookup_widget (application_main, "canvas_main");
	jobset = g_object_get_data (G_OBJECT (canvas), "jobset");

	/* Search the provided ID in the currently displayed items */
	item = g_slist_find_custom (jobset->jobs, id, child_job_find_job);
	if (item != NULL) {
		child_job_draw ((child_job_t *)(item->data),
				jobset->group_jobs);
	}

	/* Nothing in the copy/paste clipboard so nothing to paste */
	main_paste_set_sensitive (FALSE);
}


/*
 * The provided job/jobset has been removed from the copy/paste clipboard
 */
static void
child_job_copy_paste_clear (const gchar *id, gpointer data)
{
	/* Nothing in the copy/paste clipboard so nothing to paste */
	main_paste_set_sensitive (FALSE);
}


/*
 * The provided job/jobset has been pasted so refresh the window to display
 * it's new copy
 */
static void
child_job_copy_paste_pasted (	const gchar *id,
				gdouble x, gdouble y,gpointer data)
{
	jobset_list_refresh (application_main);
}


/*
 * Copy the provided job/jobset in the copy/paste clipboard
 *
 * Return:
 *    TRUE --> The job/jobset has been copied in the buffer
 *   FALSE --> Error
 */
gboolean
child_job_copy (child_job_t *ptr)
{
	copy_paste_copy (	ptr->id,
				child_job_copy_paste_clear, NULL,
				child_job_copy_paste_pasted, NULL);
	main_paste_set_sensitive (TRUE);
	return TRUE;
}


/*
 * Cut the provided job/jobset in the copy/paste clipboard
 *
 * Return:
 *    TRUE --> The job/jobset has been cut in the buffer
 *   FALSE --> Error
 */
gboolean
child_job_cut (child_job_t *ptr)
{
	copy_paste_cut (ptr->id,
			child_job_cut_paste_clear, NULL,
			child_job_copy_paste_pasted, NULL);
	main_paste_set_sensitive (TRUE);
	child_job_set_mode (ptr, CUT);
	return TRUE;
}


/*
 * Callback for the `Copy' item in the popup menu
 */
void
child_job_menu_copy (GtkMenuItem *item)
{
	child_job_t *ptr;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	child_job_copy (ptr);
}


/*
 * Callback for the `Cut' item in the popup menu
 */
void
child_job_menu_cut (GtkMenuItem *item)
{
	child_job_t *ptr;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	child_job_cut (ptr);
}


/*
 * Callback for the `Statistics' item in the popup menu
 */
void
child_job_menu_statistics (GtkMenuItem *item)
{
	child_job_t *ptr;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	new_dialog_statistics (ptr->id, ptr->name);
}


/*
 * Callback to refresh the canvas
 */
static void
jobset_refresh (void *user_data, const gchar *foo)
{
	jobset_list_refresh (application_main);
}


/*
 * Callback for the `Properties' item in the popup menu
 */
void
child_job_menu_properties (GtkMenuItem *item)
{
	child_job_t *ptr;
	int workload_date;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	workload_date = (int) workload_get_workload (application_main);
	new_dialog_job (workload_date, ptr->id, ptr->name,
			ptr->type, ptr->x, ptr->y,
			ptr->parent_jobset->id,
			jobset_refresh, NULL);
}


/*
 * Check for loops in the dependency links
 *
 * Return:
 *    TRUE --> A look has been found
 *   FALSE --> No loop
 */
static gboolean
child_job_link_check_loop (GSList *links, child_job_t *src)
{
	GSList *l;
	link_t *link;

	for (l = links; l != NULL; l = l->next) {
		link = (link_t *)(l->data);
		if (	   link->dst == src
			|| child_job_link_check_loop (	link->dst->links_out,
							src) == TRUE)
		{
			return TRUE;
		}
	}
	return FALSE;
}


/*
 * Callback for the create link menu item
 */
static void
child_job_link_activated (GtkMenuItem *item, gpointer user_data) 
{
	child_job_t *dst = (child_job_t *) user_data;
	child_job_t *src;
	GtkWidget *menu;
	job_status_state link_type;
	link_t *link;

	cursor_busy (NULL);

	/* Retrieve the source job/jobset */
	src = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");

	/* Check for loops */
	if (	   child_job_link_check_loop (dst->links_out, src) == TRUE
		&& question_window (application_main,
			_("A loop has been detected"),
_("In some case this may cause a dead lock and prevent jobs to start.  Are you sure you want to create this link anyway?"))
							== FALSE)
	{
		cursor_normal (NULL);
		return;
	}

	/* Retrieve the new link type */
	menu = gtk_widget_get_ancestor (GTK_WIDGET (item), GTK_TYPE_MENU);
	link_type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (menu),
							"link_type"));

	/* Add the link to the database */
	sql_link_add (	src->id, dst->id, (long int)link_type,
			(void (*)(void *, const char*, unsigned int))
				error_window_ignore_errno,
			_("Database error"));


	/* Build the link object and add it to the list of links */
	link = new_link_jobs (src, dst, link_type);
	jobset_item_add_link (src->parent_jobset, link);

	/* Draw the link */
	jobset_item_draw_link (src->parent_jobset, link);

	cursor_normal (NULL);
}


/*
 * Find a child_job_t object in a list of links
 */
static gint
child_job_find_job_in_link (gconstpointer a, gconstpointer b)
{
	link_t *link = (link_t *)a;
	child_job_t *ptr = (child_job_t *)b;

	if (link->dst == ptr || link->src == ptr) {
		return 0;
	}
	return 1;
}


/*
 * Internal structure to represent menu item job/jobset
 */
struct _child_job_item {
	child_job_t *ptr;
	GdkPixbuf *scaled_icon;
	gboolean sensitive;
};
typedef struct _child_job_item child_job_item_t;


/*
 * Free a child_job_item_t structure
 */
static void
child_job_item_free (gpointer data, gpointer user_data)
{
	child_job_item_t *item = (child_job_item_t *)data;

	if (item != NULL) {
		if (item->scaled_icon != NULL) {
			g_object_unref (item->scaled_icon);	
		}
		g_free (item);
	}
}


/*
 * Build the item menu for the provided job item
 */
static void
child_job_build_job_menu_foreach (gpointer data, gpointer user_data)
{
	child_job_item_t *item = (child_job_item_t *)data;
	GtkWidget *menu = (GtkWidget *)user_data;
	GtkWidget *menu_item, *image;

	menu_item = gtk_image_menu_item_new_with_label (item->ptr->name);
	if (item->scaled_icon != NULL) {
		image = gtk_image_new_from_pixbuf (item->scaled_icon);
		gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menu_item),
						image);
	}

	gtk_menu_append (GTK_MENU (menu), menu_item);

	/* Signal */
	g_signal_connect (	G_OBJECT (menu_item), "activate",
				G_CALLBACK (child_job_link_activated),
				item->ptr);

	if (item->sensitive == FALSE) {
		gtk_widget_set_sensitive (menu_item, FALSE);
	}
}


/*
 * Build a menu from the provided list.  This list must have been build by
 * the child_job_build_job_list() function
 */
static GtkWidget *
child_job_build_job_menu (GSList *jobs, job_status_state link_type)
{
	GtkWidget *menu;

	/* Create the menu */
	menu = gtk_menu_new ();

	/* Build the menu for each job/jobset */
        g_slist_foreach (jobs, child_job_build_job_menu_foreach, menu);

	/* Store the link type in the menu */
	g_object_set_data (	G_OBJECT (menu), "link_type",
				GINT_TO_POINTER (link_type));

	gtk_widget_show_all (menu);
	return menu;
}


/*
 * Compare two job/jobset names (used by g_slist_insert_sorted() to sort the
 * jobs/jobsets in the list)
 */
static gint
child_job_build_job_cmp (gconstpointer a, gconstpointer b)
{
	child_job_item_t *item1 = (child_job_item_t *)a;
	child_job_item_t *item2 = (child_job_item_t *)b;

	return g_ascii_strcasecmp (item1->ptr->name, item2->ptr->name); 
}


/*
 * Create a child_job_item_t structure and add it to the list of menu items
 */
static void
child_job_build_job_list_foreach (gpointer data, gpointer user_data)
{
	child_job_t *ptr = (child_job_t *)data;
	GPtrArray *a = (GPtrArray *)user_data;
	child_job_t *source;
	GSList **l;
	child_job_item_t *item;
	GdkPixbuf *p;
	gint width, height;
	gdouble ratio;

	/* Retrieve the parameters from the array */
	l = (GSList **)g_ptr_array_index (a, 0);
	source = (child_job_t *)g_ptr_array_index (a, 1);

	/* Do not add a menu item for the source job/jobset */
	if (source == ptr) {
		return;
	}

	/* Create the item */
	item = g_new (child_job_item_t, 1);
	item->ptr = ptr;

	/* Create the icon to add to the menu item */
	p = ptr->icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT];
	width = gdk_pixbuf_get_width (p);
	height = gdk_pixbuf_get_height (p);
	ratio = (gdouble)width / (gdouble)height;
	if (width > height) {
		width = CHILD_JOB_MENU_ICON_SIZE;
		height = CHILD_JOB_MENU_ICON_SIZE / ratio;
	}
	else {
		width = CHILD_JOB_MENU_ICON_SIZE * ratio;
		height = CHILD_JOB_MENU_ICON_SIZE;
	}
	item->scaled_icon = gdk_pixbuf_scale_simple (
				ptr->icon[JOB_STATUS_STATE_UNDEFINED][DEFAULT],
				width, height, GDK_INTERP_BILINEAR);

	/*
	 * If there is already a link to the provided job/jobset
	 * set the menu item insensible
	 */
	if (	   g_slist_find_custom (source->links_out, ptr,
					child_job_find_job_in_link) != NULL
		|| g_slist_find_custom (source->links_in, ptr,
					child_job_find_job_in_link) != NULL)
	{
		item->sensitive = FALSE;
	}
	else {
		item->sensitive = TRUE;
	}

	/* Add the item to the list */
	*l = g_slist_insert_sorted (*l, item, child_job_build_job_cmp);
}


/*
 * Build the list of item.  This list can then be used by the 
 * child_job_build_job_menu() to actually build the menu widget.
 *
 * Return:
 *   The new list to be freed by the caller by
 *           g_slist_foreach (l, child_job_item_free, NULL);
 *           g_slist_free (l);
 */
static GSList *
child_job_build_job_list (child_job_t *ptr)
{
	GSList *l;
	GPtrArray *a;

	/*
	 * Create and fill the array used to exchange data with the
	 * g_slist_foreach() function below
	 */
	l = NULL;
	a = g_ptr_array_sized_new (2);
	g_ptr_array_add (a, &l);
	g_ptr_array_add (a, ptr);

	/* Build the job list */
	g_slist_foreach (	ptr->parent_jobset->jobs,
				child_job_build_job_list_foreach, a);
	g_ptr_array_free (a, TRUE);
	return l;
}


/*
 * Callback for the Stop menu item
 */
void
child_job_menu_stop (GtkMenuItem *item)
{
	child_job_t *ptr;
	char *err_msg;
	guint32 workload;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	if (ptr == NULL) {
		return;
	}

	/* Ask the user if she really wants to stop the selected job */
	if (ptr->type != 0) {
		if (question_window (application_main, _("Stop the job"),
			_("Are you sure you want to stop this job?")) == FALSE)
		{
			return;
		}
	}
	else {
		if (question_window (application_main, _("Stop the jobset"),
_("Are you sure you want to stop this jobsets and all its jobs?")) == FALSE)
		{
			return;
		}
	}

	cursor_busy (NULL);

	/* Retrieve the currently displayed workload */
	workload = workload_get_workload (application_main);

	/* Insert the command in the database */
	if (command_add (	workload, ptr->id, user_name,
				COMMAND_STOP, 0, &err_msg) != 0)
	{
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
	}
	else {
		info_window (_("Command successfully submitted"),
_("Your command has been submitted to the Schedwi Server.\nIt may take up to a minute to complete."));
	}
	cursor_normal (NULL);
}


/*
 * Callback for the Start/Restart menu item
 */
void
child_job_menu_start (GtkMenuItem *item)
{
	child_job_t *ptr;
	char *err_msg;
	guint32 workload;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	if (ptr == NULL) {
		return;
	}

	/* Ask the user if she really wants to start the selected job */
	if (question_window (application_main, _("Start the job"),
		_("Are you sure you want to start this job?")) == FALSE)
	{
		return;
	}

	cursor_busy (NULL);

	/* Retrieve the currently displayed workload */
	workload = workload_get_workload (application_main);

	/* Insert the command in the database */
	if (command_add (	workload, ptr->id, user_name,
				COMMAND_CHANGE_STATUS,
				JOB_STATUS_STATE_RUNNING, &err_msg) != 0)
	{
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
	}
	else {
		info_window (_("Command successfully submitted"),
_("Your command has been submitted to the Schedwi Server.\nIt may take up to a minute to complete."));
	}
	cursor_normal (NULL);
}


/*
 * Callback for the `Change to completed' menu item
 */
void
child_job_menu_status_completed (GtkMenuItem *item)
{
	child_job_t *ptr;
	char *err_msg;
	guint32 workload;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	if (ptr == NULL) {
		return;
	}

	/* Ask the user if she really wants to set the status */
	if (question_window (application_main, _("Force to completed"),
		_("Are you sure you want to force this status?")) == FALSE)
	{
		return;
	}

	cursor_busy (NULL);

	/* Retrieve the currently displayed workload */
	workload = workload_get_workload (application_main);

	/* Insert the command in the database */
	if (command_add (	workload, ptr->id, user_name,
				COMMAND_CHANGE_STATUS,
				JOB_STATUS_STATE_COMPLETED, &err_msg) != 0)
	{
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
	}
	else {
		info_window (_("Command successfully submitted"),
_("Your command has been submitted to the Schedwi Server.\nIt may take up to a minute to complete."));
	}
	cursor_normal (NULL);
}


/*
 * Callback for the `Change to failed' menu item
 */
void
child_job_menu_status_failed (GtkMenuItem *item)
{
	child_job_t *ptr;
	char *err_msg;
	guint32 workload;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	if (ptr == NULL) {
		return;
	}

	/* Ask the user if she really wants to set the status */
	if (question_window (application_main, _("Force to failed"),
		_("Are you sure you want to force this status?")) == FALSE)
	{
		return;
	}

	cursor_busy (NULL);

	/* Retrieve the currently displayed workload */
	workload = workload_get_workload (application_main);

	/* Insert the command in the database */
	if (command_add (	workload, ptr->id, user_name,
				COMMAND_CHANGE_STATUS,
				JOB_STATUS_STATE_FAILED, &err_msg) != 0)
	{
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
	}
	else {
		info_window (_("Command successfully submitted"),
_("Your command has been submitted to the Schedwi Server.\nIt may take up to a minute to complete."));
	}
	cursor_normal (NULL);
}


/*
 * Callback for the `Change to waiting' menu item
 */
void
child_job_menu_status_waiting (GtkMenuItem *item)
{
	child_job_t *ptr;
	char *err_msg;
	guint32 workload;

	ptr = (child_job_t *) g_object_get_data (G_OBJECT (application_main),
						"child_job");
	if (ptr == NULL) {
		return;
	}

	/* Ask the user if she really wants to set the status */
	if (question_window (application_main, _("Force to waiting"),
		_("Are you sure you want to force this status?")) == FALSE)
	{
		return;
	}

	cursor_busy (NULL);

	/* Retrieve the currently displayed workload */
	workload = workload_get_workload (application_main);

	/* Insert the command in the database */
	if (command_add (	workload, ptr->id, user_name,
				COMMAND_CHANGE_STATUS,
				JOB_STATUS_STATE_WAITING, &err_msg) != 0)
	{
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
	}
	else {
		info_window (_("Command successfully submitted"),
_("Your command has been submitted to the Schedwi Server.\nIt may take up to a minute to complete."));
	}
	cursor_normal (NULL);
}


/*
 * Popup the background menu
 */
static GtkWidget *
view_popup_menu_job (GdkEventButton *event, child_job_t *ptr)
{
	GtkWidget *menu, *job_menu, *links, *item;
	GSList *l;

	menu = create_menu_job ();
	gtk_widget_show_all (menu);

	/*
	 * Store the provided child_job_t object in the menu.
	 * This will be used by the callbacks to access data
	 */
	g_object_set_data (G_OBJECT (application_main), "child_job", ptr);

	if (ptr->parent_jobset->mode == EDITING) {
		/* Build the sub-menus */
		links = lookup_widget (menu, "new_link1");
		if (g_slist_length (ptr->parent_jobset->jobs) > 1) {

			l = child_job_build_job_list (ptr);

			item = lookup_widget (links, "completed1");
			job_menu = child_job_build_job_menu (l,
						JOB_STATUS_STATE_COMPLETED);
			gtk_menu_item_set_submenu (	GTK_MENU_ITEM (item),
							job_menu);

			item = lookup_widget (links, "failed1");
			job_menu = child_job_build_job_menu (l,
						JOB_STATUS_STATE_FAILED);
			gtk_menu_item_set_submenu (	GTK_MENU_ITEM (item),
							job_menu);

			item = lookup_widget (links, "running1");
			job_menu = child_job_build_job_menu (l,
						JOB_STATUS_STATE_RUNNING);
			gtk_menu_item_set_submenu (	GTK_MENU_ITEM (item),
							job_menu);

			item = lookup_widget (links, "waiting1");
			job_menu = child_job_build_job_menu (l,
						JOB_STATUS_STATE_WAITING);
			gtk_menu_item_set_submenu (	GTK_MENU_ITEM (item),
							job_menu);

			g_slist_foreach (l, child_job_item_free, NULL);
			g_slist_free (l);
		}
		else {
			gtk_widget_set_sensitive (links, FALSE);
		}
	}
	else {
		switch (ptr->state) {
			case JOB_STATUS_STATE_WAITING:
				gtk_widget_destroy (lookup_widget (menu,
							"stop_now1"));
				gtk_widget_destroy (lookup_widget (menu,
							"force_to_waiting1"));
				break;

			case JOB_STATUS_STATE_RUNNING:
				gtk_widget_destroy (lookup_widget (menu,
							"start_now1"));
				break;

			case JOB_STATUS_STATE_COMPLETED:
				gtk_widget_destroy (lookup_widget (menu,
							"stop_now1"));
				gtk_widget_destroy (lookup_widget (menu,
						"force_to_completed1"));
				item = lookup_widget (menu, "start_now1");
				gtk_label_set_text_with_mnemonic (
					GTK_LABEL (GTK_BIN (item)->child),
					_("Re_start"));
				break;

			case JOB_STATUS_STATE_FAILED:
				gtk_widget_destroy (lookup_widget (menu,
							"stop_now1"));
				gtk_widget_destroy (lookup_widget (menu,
							"force_to_failed1"));
				item = lookup_widget (menu, "start_now1");
				gtk_label_set_text_with_mnemonic (
					GTK_LABEL (GTK_BIN (item)->child),
					_("Re_start"));
				break;

			default:
				gtk_widget_destroy (lookup_widget (menu,
							"start_now1"));
				gtk_widget_destroy (lookup_widget (menu,
							"stop_now1"));
				gtk_widget_destroy (lookup_widget (menu,
							"separator_one"));
				gtk_widget_destroy (lookup_widget (menu,
						"force_to_completed1"));
				gtk_widget_destroy (lookup_widget (menu,
							"force_to_failed1"));
				gtk_widget_destroy (lookup_widget (menu,
							"force_to_waiting1"));
				gtk_widget_destroy (lookup_widget (menu,
							"separator_two"));
				break;
		}
	}

	/* Note: event can be NULL here when called from view_onPopupMenu;
	 *  gdk_event_get_time() accepts a NULL argument
	 */
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
			(event != NULL) ? event->button : 0,
			gdk_event_get_time((GdkEvent*)event));

	return menu;
}


/*
 * Prepare the tooltip for the provided job/jobset
 */
static void
child_job_tooltips_start (child_job_t *ptr)
{
	char *label[5];
	char *value[5];
	char *title, *s1, *s2, *s3;
	gchar *s4;
	unsigned int i;

	i = 0;
	s1 = s2 = s3 = NULL;
	s4 = NULL;
	switch (ptr->state) {
		case JOB_STATUS_STATE_WAITING:
			title = job_status_state2str (ptr->state);

			label[i] = _("Expected start time:");
			s1 = schedwi_time_strftime (NULL, ptr->run_time);
			value[i++] = s1;

			if (ptr->start_limit > 0) {
				label[i] = _("Must start before:");
				s2 = schedwi_time_strftime (NULL,
						schedwi_time_add_seconds (
							ptr->run_time,
							ptr->start_limit)); 
				value[i++] = s2;
			}

			if (ptr->average_duration > 0) {
				label[i] = _("Expected duration:");
				s3 = schedwi_time_duration_to_str (
							ptr->average_duration);
				value[i++] = s3;
			}

			if (	   ptr->status_message != NULL
				&& (ptr->status_message)[0] != '\0')
			{
				label[i] = _("Details:");
				value[i++] = ptr->status_message;
			}
			break;

		case JOB_STATUS_STATE_RUNNING:
			title = job_status_state2str (ptr->state);

			label[i] = _("Start time:");
			s1 = schedwi_time_strftime (NULL, ptr->start_time);
			value[i++] = s1;

			label[i] = _("Expected start time was:");
			s2 = schedwi_time_strftime (NULL, ptr->run_time);
			value[i++] = s2;

			if (ptr->average_duration > 0) {
				label[i] = _("Expected duration:");
				s3 = schedwi_time_duration_to_str (
							ptr->average_duration);

				if (s3 != NULL) {
					s4 = g_strdup_printf (
							_("%s (%d%% done)"),
							s3,
							ptr->completion_pct);
					free (s3);
					s3 = NULL;
				}
				else {
					s4 = g_strdup_printf (
							_("%d%% done"),
							ptr->completion_pct);
				}
				value[i++] = s4;
			}

			if (ptr->max_duration > 0) {
				label[i] = _("Must end before:");
				s3 = schedwi_time_strftime (NULL,
						schedwi_time_add_seconds (
							ptr->start_time,
							ptr->max_duration)); 
				value[i++] = s3;
			}

			if (	   ptr->status_message != NULL
				&& (ptr->status_message)[0] != '\0')
			{
				label[i] = _("Details:");
				value[i++] = ptr->status_message;
			}
			break;

		case JOB_STATUS_STATE_COMPLETED:
			title = job_status_state2str (ptr->state);

			label[i] = _("Completed at:");
			s1 = schedwi_time_strftime (NULL, ptr->start_time);
			value[i++] = s1;

			if (	   ptr->status_message != NULL
				&& (ptr->status_message)[0] != '\0')
			{
				label[i] = _("Details:");
				value[i++] = ptr->status_message;
			}
			break;

		case JOB_STATUS_STATE_FAILED:
			title = job_status_state2str (ptr->state);

			label[i] = _("Failed at:");
			s1 = schedwi_time_strftime (NULL, ptr->start_time);
			value[i++] = s1;

			if (	   ptr->status_message != NULL
				&& (ptr->status_message)[0] != '\0')
			{
				label[i] = _("Details:");
				value[i++] = ptr->status_message;
			}
			break;

		default:
			title = _("Not planned");
			break;
	}
	while (i < 5) {
		label[i] = NULL;
		value[i] = NULL;
		i++;
	}

	job_tooltips_start (	ptr->parent_jobset->tooltips,
				GTK_WIDGET (ptr->parent_jobset->canvas),
				title,
				label[0], value[0],
				label[1], value[1],
				label[2], value[2],
				label[3], value[3],
				label[4], value[4]);
	if (s1 != NULL) {
		free (s1);
	}
	if (s2 != NULL) {
		free (s2);
	}
	if (s3 != NULL) {
		free (s3);
	}
	g_free (s4);
}


/*
 * Events on the job/jobset canvas item
 */
static gboolean
job_event (GnomeCanvasItem *canvasitem, GdkEvent *event, gpointer user_data)
{
	child_job_t *ptr = (child_job_t *)user_data;
	static gboolean dragging = FALSE, dragging_begin = FALSE;
	static gdouble drag_begin_x, drag_begin_y;
	jobset_item_t *parent;
	GdkCursor *grab_cursor;
	gdouble new_x, new_y;
	gchar *x_str, *y_str;
	int workload_date;

	parent = ptr->parent_jobset;

	switch (event->type) {
		case GDK_ENTER_NOTIFY:
			child_job_set_mode (ptr, (ptr->selected == FALSE)
						? HIGHLIGHTED
						: SELECTED_HIGHLIGHTED);
			if (parent->mode == ACTIVE) {
				child_job_tooltips_start (ptr);
			}
			return TRUE;

		case GDK_LEAVE_NOTIFY:
			job_tooltips_stop (parent->tooltips);
			if (copy_paste_is_this_id_cut (ptr->id) == TRUE) {
				child_job_set_mode (ptr,
						(ptr->selected == FALSE)
							? CUT
							: SELECTED);
			}
			else {
				child_job_set_mode (ptr,
						(ptr->selected == FALSE)
							? DEFAULT
							: SELECTED);
			}
			return TRUE;

		case GDK_2BUTTON_PRESS:
			job_tooltips_stop (parent->tooltips);
			workload_date = (int) workload_get_workload (
							application_main);
			new_dialog_job (workload_date, ptr->id, ptr->name,
					ptr->type, ptr->x, ptr->y,
					ptr->parent_jobset->id,
					jobset_refresh, NULL);
			return TRUE;

		case GDK_BUTTON_PRESS:
			job_tooltips_stop (parent->tooltips);
			if (event->button.button == 3) {
				view_popup_menu_job ((GdkEventButton *) event,
							ptr);
				return TRUE;
			}
			if (parent->mode == ACTIVE) {
				return FALSE;
			}

			gnome_canvas_item_grab_focus (canvasitem);

			if (event->button.button == 1) {

				/* Select the job/jobset clicked */
				jobset_item_select_job (parent, ptr);

				/* Prepare for the drag */
				dragging = TRUE;
				dragging_begin = TRUE;
				drag_begin_x = ptr->x;
				drag_begin_y = ptr->y;
				return TRUE;
			}
			return FALSE;

		case GDK_MOTION_NOTIFY:
			if (parent->mode == ACTIVE) {
				return FALSE;
			}

			if (	   dragging == TRUE
				&& (event->motion.state & GDK_BUTTON1_MASK)
						!= 0) 
			{
				/* Begining of the drag */
				if (dragging_begin == TRUE) {
					dragging_begin = FALSE;
					grab_cursor = cursor_get_grab ();
					gnome_canvas_item_grab(
					GNOME_CANVAS_ITEM (ptr->canvas_group),
						GDK_POINTER_MOTION_MASK | 
						GDK_BUTTON_RELEASE_MASK,
						grab_cursor,
						event->button.time);
				}

				/* Get the new coordinates snapped to grid */
				grid_get_coordinate (
					parent->grid,
					ptr->width, ptr->height,
					event->button.x, event->button.y,
					&new_x, &new_y);
				if (new_x != ptr->x || new_y != ptr->y) {
					new_x = MAX (ptr->width * 0.5, new_x);
					new_y = MAX (ptr->height * 0.5, new_y);
					new_x = MIN (	new_x,
							  parent->canvas_width
							- ptr->width * 0.5);
					new_y = MIN (	new_y,
							  parent->canvas_height
							- ptr->height * 0.5);

					/* Move the job/jobset */
					gnome_canvas_item_move (
						GNOME_CANVAS_ITEM (
							ptr->canvas_group),
						new_x - ptr->x,
						new_y - ptr->y);
					ptr->x = new_x;
					ptr->y = new_y;
					/* Redraw the links */
					link_draw_list (ptr->links_out,
							parent->group_links);
					link_draw_list (ptr->links_in,
							parent->group_links);
					/*
					 * Adjust the view to see the dragged
					 * object
					 */
					jobset_item_adjust_view (parent,
								ptr->x,
								ptr->y,
								ptr->width,
								ptr->height);
				}
				return TRUE;
			}
			return FALSE;

		case GDK_BUTTON_RELEASE:
			if (parent->mode == ACTIVE) {
				job_tooltips_stop (parent->tooltips);
				return FALSE;
			}

			gnome_canvas_item_ungrab (
					GNOME_CANVAS_ITEM (ptr->canvas_group),
					event->button.time);
			if (	   dragging == TRUE
				&& (	   drag_begin_x != ptr->x
					|| drag_begin_y != ptr->y)
				&& parent->mode == EDITING)
			{
				cursor_busy (GTK_WIDGET (parent->canvas));
				x_str = schedwi_ulltostr (ptr->x);
				y_str = schedwi_ulltostr (ptr->y);
				sql_children_job_update_coord (
						ptr->id, x_str, y_str, 
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
						_("Database error"));
				g_free (x_str);
				g_free (y_str);
				cursor_normal (GTK_WIDGET (parent->canvas));
			}
			dragging = FALSE;
			return TRUE;

		default:
			return FALSE;
	}
}

/*
 * Draw the provided job/jobset on the provided group canvas
 */
void
child_job_draw (child_job_t *ptr, GnomeCanvasGroup *group)
{
	gint icon_width, icon_height;
	gdouble text_width, text_height;
	gint gauge_width, gauge_height;
	GnomeCanvasItem *item;
	gchar *s;
	jobset_item_t *j_ptr;
	canvas_mode_t mode;

	gdouble icon_x, icon_y;
	gdouble gauge_x, gauge_y;
	gdouble text_bg_x, text_bg_y;
	gdouble text_x, text_y;

	j_ptr = ptr->parent_jobset;

	/*
	 * Retrieve the size of the 3 components: the icon, the job name and
	 * the gauge (only if mode is ACTIVE)
	 */
	mode = j_ptr->mode;
	if (mode == ACTIVE && ptr->state != JOB_STATUS_STATE_UNDEFINED) {
		/*
		 * The gauge size includes the separation space between
		 * the icon and the gauge (width) and the gauge and the
		 * job name (height)
		 */
		gauge_width = CHILD_JOB_GAUGE_WIDTH + CHILD_JOB_INNER_SPACE;
		gauge_height = CHILD_JOB_GAUGE_HEIGHT + CHILD_JOB_INNER_SPACE;
	}
	else {
		gauge_width = 0;
		gauge_height = 0;
	}
	
	icon_width = gdk_pixbuf_get_width (ptr->icon[ptr->state][DEFAULT]);
	/* The icon height includes the space between the icon and the name */
	icon_height = gdk_pixbuf_get_height (ptr->icon[ptr->state][DEFAULT])
			+ CHILD_JOB_INNER_SPACE;

	/*
	 * Wrap the job name (no more than CHILD_JOB_TEXT_LENGTH_BY_LINE
	 * characters by line)
	 */
	s = wrap_string (ptr->name, CHILD_JOB_TEXT_LENGTH_BY_LINE);

	/* To get the job name size, it must be drawn on the canvas */
	item = gnome_canvas_item_new (	group,
					gnome_canvas_text_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					"size-points",
		canvas_utils_get_default_text_size (ptr->parent_jobset->canvas),
					"anchor", GTK_ANCHOR_NORTH_WEST,
					"text", s,
					NULL);
	g_object_get(	item,
			"text-width", &text_width,
			"text-height", &text_height,
			NULL);
	gtk_object_destroy (GTK_OBJECT (item));

	/*
	 * Compute the overall size of the job
	 */
	ptr->width = MAX (text_width
			+ 2 * (CHILD_JOB_TEXT_SPACE + CHILD_JOB_BG_ROUND_SIZE),
				icon_width + gauge_width);
	ptr->height =	  text_height + 2 * CHILD_JOB_TEXT_SPACE
			+ MAX (icon_height, gauge_height);

	/* Radius of the circle containing the job */
	ptr->radius =  sqrt ((	  ptr->width * ptr->width
				+ ptr->height * ptr->height) / 4.0);


	/*
	 * Get the coordinates of the items
	 */

	/* Job/jobset name */
	text_x = 0.5 * (ptr->width - text_width);
	text_y = ptr->height - text_height - CHILD_JOB_TEXT_SPACE;

	/* Text background */
	text_bg_x = text_x - CHILD_JOB_TEXT_SPACE;
	text_bg_y = text_y - CHILD_JOB_TEXT_SPACE;

	/* Icon */
	icon_x = 0.5 * (ptr->width - icon_width - gauge_width);
	icon_y = text_bg_y - icon_height;

	/* Gauge */
	if (mode == ACTIVE && ptr->state != JOB_STATUS_STATE_UNDEFINED) {
		gauge_x = icon_x + icon_width + CHILD_JOB_INNER_SPACE;
		gauge_y = text_bg_y - gauge_height;
	}
	else {
		gauge_x = gauge_y = 0.0;
	}


	/*
	 * Draw the items
	 */
	if (ptr->canvas_group) {
		gtk_object_destroy (GTK_OBJECT (ptr->canvas_group));
	}

	/* Make sure that the object is in the canvas */
	if (j_ptr->canvas_width < ptr->x + 0.5 * ptr->width) {
		ptr->x = j_ptr->canvas_width - 0.5 * ptr->width;
	}
	if (j_ptr->canvas_height < ptr->y + 0.5 * ptr->height) {
		ptr->y = j_ptr->canvas_height - 0.5 * ptr->height;
	}

	ptr->canvas_group =  (GnomeCanvasGroup *)gnome_canvas_item_new (
				group,
				gnome_canvas_group_get_type (),
				"x", ptr->x - 0.5 * ptr->width,
				"y", ptr->y - 0.5 * ptr->height,
				NULL);
	g_signal_connect (	(gpointer)ptr->canvas_group, "event",
				G_CALLBACK (job_event), ptr);

	ptr->canvas_icon = gnome_canvas_item_new (
				ptr->canvas_group,
				gnome_canvas_pixbuf_get_type (),
				"x", icon_x,
				"y", icon_y,
				NULL);

	ptr->canvas_name_bg = gnome_canvas_item_new (
				ptr->canvas_group,
				gnome_canvas_rect_get_type (),
				"x1", text_bg_x,
				"y1", text_bg_y,
				"x2",	  text_bg_x + text_width
					+ 2 * CHILD_JOB_TEXT_SPACE,
				"y2",	  text_bg_y + text_height
					+ 2 * CHILD_JOB_TEXT_SPACE,
				NULL);
	ptr->canvas_name_bg_left = gnome_canvas_item_new (
				ptr->canvas_group,
				gnome_canvas_ellipse_get_type (),
				"x1", text_bg_x - CHILD_JOB_BG_ROUND_SIZE,
				"y1", text_bg_y,
				"x2", text_bg_x + CHILD_JOB_BG_ROUND_SIZE,
				"y2",	  text_bg_y + text_height
					+ 2 * CHILD_JOB_TEXT_SPACE,
				NULL);
	ptr->canvas_name_bg_right = gnome_canvas_item_new (
				ptr->canvas_group,
				gnome_canvas_ellipse_get_type (),
				"x1",	  text_bg_x + text_width
					+ 2 * CHILD_JOB_TEXT_SPACE
					- CHILD_JOB_BG_ROUND_SIZE,
				"y1", text_bg_y,
				"x2",	  text_bg_x + text_width
					+ 2 * CHILD_JOB_TEXT_SPACE
					+ CHILD_JOB_BG_ROUND_SIZE,
				"y2",	  text_bg_y + text_height
					+ 2 * CHILD_JOB_TEXT_SPACE,
				NULL);
	ptr->canvas_name = gnome_canvas_item_new (
				ptr->canvas_group,
				gnome_canvas_text_get_type (),
				"x", text_x,
				"y", text_y,
				"size-points",
		canvas_utils_get_default_text_size (ptr->parent_jobset->canvas),
				"anchor", GTK_ANCHOR_NORTH_WEST,
				"text", s,
				NULL);
	g_free (s);
	if (mode == ACTIVE && ptr->state != JOB_STATUS_STATE_UNDEFINED) {
		/* Gauge */
		gnome_canvas_item_new (
				ptr->canvas_group,
				gnome_canvas_rect_get_type (),
				"x1", gauge_x,
				"y1", gauge_y + CHILD_JOB_GAUGE_HEIGHT
					- (	  ptr->completion_pct
						* CHILD_JOB_GAUGE_HEIGHT
						/ 100.0),
				"x2",	  gauge_x + gauge_width
					- CHILD_JOB_INNER_SPACE,
				"y2",	  gauge_y + gauge_height
					- CHILD_JOB_INNER_SPACE,
				"fill-color", CHILD_JOB_GAUGE_COLOR,
				NULL);

		/* Frame around the gauge level */
		gnome_canvas_item_new (
				ptr->canvas_group,
				gnome_canvas_rect_get_type (),
				"x1", gauge_x,
				"y1", gauge_y,
				"x2",	  gauge_x + gauge_width
					- CHILD_JOB_INNER_SPACE,
				"y2",	  gauge_y + gauge_height
					- CHILD_JOB_INNER_SPACE,
				"width-units",
					(gdouble)CHILD_JOB_GAUGE_FRAME_WIDTH,
				"outline-color", CHILD_JOB_GAUGE_FRAME_COLOR,
				NULL);
	}

	if (copy_paste_is_this_id_cut (ptr->id) == TRUE) {
		child_job_set_mode (ptr,
				(ptr->selected == FALSE) ? CUT :SELECTED);
	
	}
	else {
		child_job_set_mode (ptr,
				(ptr->selected == FALSE) ? DEFAULT :SELECTED);
	}
}


/*
 * Draw all the jobs/jobsets in the list to the provided canvas group
 */
void
child_job_draw_list (GSList *list, GnomeCanvasGroup *group)
{
        g_slist_foreach (list, (GFunc)child_job_draw, group);
}

/*-----------------============== End Of File ==============-----------------*/
