/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* copy_paste.c - Job/jobset copy/cut/paste functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include <sql_link.h>
#include <sql_job_hierarchy.h>
#include <message_windows.h>
#include <cursor.h>
#include <schedwi_g_utils.h>
#include <copy_paste.h>


/* Current copied job/jobset ID */
static gchar *id = NULL;

/* Copied (TRUE) or cut (FALSE) */
static gboolean is_cut = FALSE;

/* User function called when the copy/cut buffer must be cleared and pasted */
static void (*f_user_clear)(const gchar *, gpointer) = NULL;
static gpointer d_user_clear;
static void (*f_user_paste)(const gchar *, gdouble, gdouble, gpointer) = NULL;
static gpointer d_user_paste;


/*
 * Clear the copied/cut job/jobset
 */
void
copy_paste_clear ()
{
	gchar *id_tmp;

	id_tmp = id;
	id = NULL;
	is_cut = FALSE;

	if (f_user_clear != NULL) {
		f_user_clear (id_tmp, d_user_clear);
	}

	g_free (id_tmp);

	f_user_clear = NULL;
	f_user_paste = NULL;
}


/*
 * Clear the copied/cut job/jobset if it matches the provided ID
 */
void
copy_paste_clear_if (const gchar *job_id)
{
	if (id != NULL && job_id != NULL && strcmp (id, job_id) == 0) {
		copy_paste_clear ();
	}
}


/*
 * Clear the cut clipboard
 */
void
copy_paste_clear_if_cut ()
{
	if (is_cut == TRUE && id != NULL) {
		copy_paste_clear ();
	}
}


/*
 * Clear the cut job/jobset if it matches the provided ID
 */
void
copy_paste_clear_if_this_id_is_cut (const gchar *job_id)
{
	if (	   is_cut == TRUE && id != NULL && job_id != NULL
		&& strcmp (id, job_id) == 0)
	{
		copy_paste_clear ();
	}
}


/*
 * Is the provided ID is in the cut clipboard
 */
gboolean
copy_paste_is_this_id_cut (const gchar *job_id)
{
	if (	   is_cut == TRUE && id != NULL && job_id != NULL
		&& strcmp (id, job_id) == 0)
	{
		return TRUE;
	}
	return FALSE;
}


/*
 * Cut the provided job/jobset.
 * The provided function_clear() function is called with the job/jobset ID
 * and the provided user_data_clear value as parameters when the cut/copy buffer
 * must be cleared.
 * The function_paste() function is called with the new job/jobset ID and the
 * provided user_data_paste value as parameters just after the job/jobset
 * has been pasted.
 */
void
copy_paste_cut (const gchar *job_id,
		void (*function_clear)(const gchar *, gpointer),
		gpointer user_data_clear,
		void (*function_paste)(	const gchar *,
					gdouble, gdouble, gpointer),
		gpointer user_data_paste)
{
	copy_paste_clear ();
	if (job_id != NULL) {
		id = g_strdup (job_id);
	}
	f_user_clear = function_clear;
	d_user_clear = user_data_clear;
	f_user_paste = function_paste;
	d_user_paste = user_data_paste;
	is_cut = TRUE;
}


/*
 * Copy the provided job/jobset
 * The provided function_clear() function is called with the job/jobset ID
 * and the provided user_data_clear value as parameters when the cut/copy buffer
 * must be cleared.
 * The function_paste() function is called with the new job/jobset ID and the
 * provided user_data_paste value as parameters just after the job/jobset
 * has been pasted.
 */
void
copy_paste_copy (const gchar *job_id,
		void (*function_clear)(const gchar *, gpointer),
		gpointer user_data_clear,
		void (*function_paste)(	const gchar *,
					gdouble, gdouble, gpointer),
		gpointer user_data_paste)
{
	copy_paste_clear ();
	if (job_id != NULL) {
		id = g_strdup (job_id);
	}
	f_user_clear = function_clear;
	d_user_clear = user_data_clear;
	f_user_paste = function_paste;
	d_user_paste = user_data_paste;
	is_cut = FALSE;
}


/*
 * Tell if the clipboard contains a job
 */
gboolean
copy_paste_is_set ()
{
	return (id == NULL) ? FALSE : TRUE;
}

/*-----------------============== End Of File ==============-----------------*/
